/*
 * Copyright (c) 2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.voodoo2javie;

import java.lang.reflect.InvocationTargetException;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.wizard.Wizard;

import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.app.InjectorHolder;
import ch.kuramo.javie.app.UIUtil;
import ch.kuramo.javie.app.project.AbstractNewLayerOperation;
import ch.kuramo.javie.app.project.NewLayerCompositionOperation;
import ch.kuramo.javie.app.project.ProjectManager;
import ch.kuramo.javie.app.views.ProjectView;
import ch.kuramo.javie.core.CameraLayer;
import ch.kuramo.javie.core.CompositionItem;
import ch.kuramo.javie.core.Folder;
import ch.kuramo.javie.core.Item;
import ch.kuramo.javie.core.JavieRuntimeException;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.ProjectDecodeException;
import ch.kuramo.javie.core.services.ProjectDecoder;
import ch.kuramo.javie.core.services.ProjectElementFactory;
import ch.kuramo.javie.core.services.ProjectEncoder;

import com.google.inject.Inject;

class Voodoo2JavieWizard extends Wizard {

	private final Voodoo2Javie v2j;

	private final ProjectManager pm;

	private Voodoo2JavieWizardPage page;

	@Inject
	private ProjectElementFactory elementFactory;

	@Inject
	private ProjectEncoder encoder;

	@Inject
	private ProjectDecoder decoder;


	Voodoo2JavieWizard(Voodoo2Javie v2j, ProjectManager pm) {
		this.v2j = v2j;
		this.pm = pm;
		InjectorHolder.getInjector().injectMembers(this);
		setWindowTitle("Voodoo Camera Tracker カメラデータの読み込み");
	}

	@Override
	public void addPages() {
		page = new Voodoo2JavieWizardPage(v2j, pm);
		addPage(page);
	}

	@Override
	public boolean performFinish() {
		final double scale = page.getScale();
		final Time frameDuration = page.getFrameDuration();
		LayerComposition comp = page.getComposition();
		String newCompName = page.getNewCompositionName();

		final Project tmpPrj = elementFactory.newProject();
		try {
			IRunnableWithProgress runnable = new IRunnableWithProgress() {
				public void run(IProgressMonitor monitor) throws InvocationTargetException, InterruptedException {
					v2j.createCamera(scale, frameDuration, tmpPrj, elementFactory, monitor);
				}
			};

			try {
				getContainer().run(true, true, runnable);
			} catch (InterruptedException e) {
				return false;
			} catch (InvocationTargetException e) {
				e.printStackTrace();
				MessageDialog.openError(getShell(), "エラー",
						(e.getCause() != null) ? e.getCause().getMessage() : e.getMessage());
				return false;
			}

			if (comp == null) {
				CompositionItem tmpItem = (CompositionItem) tmpPrj.getItems().get(0);
				tmpItem.setName(newCompName);

				ProjectView projectView = UIUtil.showView(pm.getWorkbenchWindow().getActivePage(), ProjectView.ID);
				Folder parent = projectView.getItemTreeInsertionPoint();
				pm.postOperation(new NewLayerCompositionOperation(pm, parent,
						tmpItem.getName(), (LayerComposition) tmpItem.getComposition(),
						"Voodoo Camera Tracker カメラデータの読み込み"));
			} else {
				// FIXME AbstractNewLayerOperationの内部で、レイヤーアウトポイントをコンポジションの長さに合わされてしまう。
				pm.postOperation(new AbstractNewLayerOperation<CameraLayer>(
						pm, comp, "Voodoo Camera Tracker カメラデータの読み込み") {

					protected CameraLayer createNewLayer(
							ProjectManager projectManager, LayerComposition comp, ProjectElementFactory factory) {

						CompositionItem tmpItem = (CompositionItem) tmpPrj.getItems().get(0);
						CameraLayer cameraLayer = (CameraLayer) ((LayerComposition) tmpItem.getComposition()).getLayers().get(0);
						try {
							cameraLayer = decoder.decodeElement(encoder.encodeElement(cameraLayer), CameraLayer.class);
							cameraLayer.afterDecode(pm.getProject(), comp);
						} catch (ProjectDecodeException e) {
							throw new JavieRuntimeException(e);
						}
						return cameraLayer;
					}
				});
			}
			return true;

		} finally {
			if (tmpPrj != null) {
				for (Item i : tmpPrj.getItems()) {
					i.dispose();
				}
			}
		}
	}

}
