/*
 * Copyright (c) 2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie2mmd;

import java.io.File;

import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Text;

import ch.kuramo.javie.api.Size2i;
import ch.kuramo.javie.api.Vec3d;
import ch.kuramo.javie.app.widgets.GridBuilder;
import ch.kuramo.javie.core.output.Javie2MMDOutput.ViewingAngleRounding;

public class Javie2MMDWizardPage extends WizardPage {

	private final Size2i compSize;

	private File defaultFile;

	private File file;

	private Text scaleText;

	private Combo originCombo;

	private Text xoffText;

	private Text yoffText;

	private Text zoffText;

	private Combo roundingCombo;


	private final ModifyListener modifyListener = new ModifyListener() {
		public void modifyText(ModifyEvent e) { doValidate(); }
	};

	private final SelectionListener selectionListener = new SelectionListener() {
		public void widgetSelected(SelectionEvent e)		{ doValidate(); }
		public void widgetDefaultSelected(SelectionEvent e)	{ doValidate(); }
	};


	public Javie2MMDWizardPage(Size2i compSize, File defaultFile) {
		super("Javie2MMDWizardPage", "書き出し設定", null);
		this.compSize = compSize;
		this.defaultFile = defaultFile;
	}

	public void createControl(Composite parent) {
		GridBuilder gb = new GridBuilder(parent, 1, false);
		((GridLayout) gb.getComposite().getLayout()).verticalSpacing = 0;

		createFileGroup(gb);
		gb.size(10, 10).composite(SWT.NULL);
		createMainGroup(gb);

		Composite grid = gb.getComposite();
		grid.setTabList(gb.getTabList());

		setControl(grid);
		doValidate();
	}

	private void createFileGroup(GridBuilder gb) {
		Group group = gb.hAlign(SWT.FILL).hGrab().group(SWT.NONE, null);
		group.setLayout(new FillLayout(SWT.HORIZONTAL));
		GridBuilder gb2 = new GridBuilder(group, 2, false);
		GridLayout layout = (GridLayout) gb2.getComposite().getLayout();
		layout.marginWidth = 15;

								gb2.hSpan(1).hAlign(SWT.RIGHT).label(SWT.NULL, "出力ファイル:");
		final Text fileText =	gb2.hSpan(1).hAlign(SWT.FILL).hGrab().text(SWT.READ_ONLY, "");

		Button fileButton =		gb2.hSpan(2).hAlign(SWT.RIGHT).button(SWT.PUSH, "ファイルを指定...");

		fileText.setBackground(fileText.getDisplay().getSystemColor(SWT.COLOR_WHITE));

		fileButton.setFocus();
		fileButton.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				File newFile = showSaveDialog(defaultFile.getParentFile(), defaultFile.getName());
				if (newFile != null) {
					defaultFile = file = newFile;
					fileText.setText(newFile.getAbsolutePath());
					fileText.setFocus();
					fileText.selectAll();
					doValidate();
				}
			}
		});

		Composite grid2 = gb2.getComposite();
		grid2.setTabList(gb2.getTabList());
	}

	private void createMainGroup(GridBuilder gb) {
		String[] originComboItems = new String[] { "原点", "コンポジションの中心", "カスタム" };
		String[] roundingComboItems = new String[] { "四捨五入", "切り捨て", "切り上げ" };

		Group group = gb.hAlign(SWT.FILL).hGrab().group(SWT.NONE, null);
		group.setLayout(new FillLayout(SWT.HORIZONTAL));
		GridBuilder gb2 = new GridBuilder(group, 9, false);
		GridLayout layout = (GridLayout) gb2.getComposite().getLayout();
		layout.marginWidth = 15;

						gb2.hSpan(1).hAlign(SWT.RIGHT).label(SWT.NULL, "スケール:");
		scaleText =		gb2.hSpan(2).hAlign(SWT.FILL).width(70).text(SWT.BORDER, "5");
						gb2.hSpan(6).hAlign(SWT.LEFT).label(SWT.NULL, "%");

						gb2.hSpan(1).hAlign(SWT.RIGHT).label(SWT.NULL, "位置の基準:");
		originCombo =	gb2.hSpan(8).hAlign(SWT.LEFT).combo(SWT.READ_ONLY, originComboItems, originComboItems[1]);

						gb2.hSpan(1).size(10, 10).composite(SWT.NULL);
						gb2.hSpan(1).hAlign(SWT.LEFT).label(SWT.NULL, "x =");
		xoffText =		gb2.hSpan(2).hAlign(SWT.FILL).width(70).text(SWT.BORDER, "");
						gb2.hSpan(1).hAlign(SWT.LEFT).label(SWT.NULL, "  y =");
		yoffText =		gb2.hSpan(1).hAlign(SWT.FILL).width(70).text(SWT.BORDER, "");
						gb2.hSpan(1).hAlign(SWT.LEFT).label(SWT.NULL, "  z =");
		zoffText =		gb2.hSpan(1).hAlign(SWT.FILL).width(70).text(SWT.BORDER, "");
						gb2.hSpan(1).hAlign(SWT.LEFT).label(SWT.NULL, "");

						gb2.hSpan(1).hAlign(SWT.RIGHT).label(SWT.NULL, "視野角の丸め:");
		roundingCombo = gb2.hSpan(8).hAlign(SWT.LEFT).combo(SWT.READ_ONLY, roundingComboItems, roundingComboItems[0]);

		scaleText.addModifyListener(modifyListener);
		originCombo.addSelectionListener(selectionListener);
		xoffText.addModifyListener(modifyListener);
		yoffText.addModifyListener(modifyListener);
		zoffText.addModifyListener(modifyListener);
		//roundingCombo.addSelectionListener(selectionListener);

		Composite grid2 = gb2.getComposite();
		grid2.setTabList(gb2.getTabList());
	}

	private File showSaveDialog(File folder, String fileName) {
		String[] filterNames = new String[] { "VMD ファイル", "すべてのファイル" };
		String[] filterExtensions = new String[] { "*.vmd", "*" };

		String platform = SWT.getPlatform();
		if (platform.equals("win32") || platform.equals("wpf")) {
			filterExtensions[filterExtensions.length-1] = "*.*";
		}

		FileDialog dialog = new FileDialog(getShell(), SWT.SAVE | SWT.SHEET);
		dialog.setFilterNames(filterNames);
		dialog.setFilterExtensions(filterExtensions);
		dialog.setFilterPath(folder != null ? folder.getAbsolutePath() : null);
		dialog.setFileName(fileName);
		dialog.setOverwrite(true);

		String path = dialog.open();
		return (path != null) ? new File(path) : null;
	}

	private void doValidate() {
		setPageComplete(false);

		String scaleStr = scaleText.getText().trim();
		double scale;
		try {
			scale = Double.parseDouble(scaleStr);
		} catch (NumberFormatException e) {
			scale = 0;
		}

		xoffText.removeModifyListener(modifyListener);
		yoffText.removeModifyListener(modifyListener);
		zoffText.removeModifyListener(modifyListener);
		switch (originCombo.getSelectionIndex()) {
			case 0:
				xoffText.setEnabled(false); xoffText.setText("0");
				yoffText.setEnabled(false); yoffText.setText("0");
				zoffText.setEnabled(false); zoffText.setText("0");
				break;
			case 1:
				xoffText.setEnabled(false); xoffText.setText(String.valueOf(compSize.width * 0.5));
				yoffText.setEnabled(false); yoffText.setText(String.valueOf(compSize.height * 0.5));
				zoffText.setEnabled(false); zoffText.setText("0");
				break;
			case 2:
				boolean focus = !xoffText.isEnabled();
				xoffText.setEnabled(true);
				yoffText.setEnabled(true);
				zoffText.setEnabled(true);
				if (focus) xoffText.setFocus();
				break;
		}
		xoffText.addModifyListener(modifyListener);
		yoffText.addModifyListener(modifyListener);
		zoffText.addModifyListener(modifyListener);


		//String warnMsg = null;
		String errMsg = null;

		if (errMsg == null && file == null) {
			errMsg = "出力ファイルが指定されていません。";
		}
		if (errMsg == null && scaleStr.length() == 0) {
			errMsg = "スケールが入力されていません。";
		}
		if (errMsg == null && scale <= 0) {
			errMsg = "スケールには0より大きい数値を入力してください。";
		}

		if (originCombo.getSelectionIndex() == 2) {
			String xStr = xoffText.getText().trim();
			String yStr = yoffText.getText().trim();
			String zStr = zoffText.getText().trim();
			double x = Double.NaN, y = Double.NaN, z = Double.NaN;
			try {
				x = Double.parseDouble(xStr);
				y = Double.parseDouble(yStr);
				z = Double.parseDouble(zStr);
			} catch (NumberFormatException e) {
			}
			if (errMsg == null && (xStr.length() == 0 || yStr.length() == 0 || zStr.length() == 0)) {
				errMsg = "位置の基準が入力されていません。";
			}
			if (errMsg == null && (Double.isNaN(x) || Double.isNaN(y) || Double.isNaN(z))) {
				errMsg = "位置の基準には数値を入力してください。";
			}
		}

		//setMessage(warnMsg, IMessageProvider.WARNING);
		setErrorMessage(errMsg);
		setPageComplete(errMsg == null);
	}

	void saveDefaults() {
		// TODO
	}

	File getFile() {
		if (isPageComplete()) {
			return file;
		} else {
			throw new IllegalStateException();
		}
	}

	double getScale() {
		if (isPageComplete()) {
			return Double.parseDouble(scaleText.getText().trim()) * 0.01;
		} else {
			throw new IllegalStateException();
		}
	}

	Vec3d getOffset() {
		if (isPageComplete()) {
			double x = Double.parseDouble(xoffText.getText().trim());
			double y = Double.parseDouble(yoffText.getText().trim());
			double z = Double.parseDouble(zoffText.getText().trim());
			return new Vec3d(x, y, z);
		} else {
			throw new IllegalStateException();
		}
	}

	ViewingAngleRounding getViewingAngleRounding() {
		if (isPageComplete()) {
			return ViewingAngleRounding.values()[roundingCombo.getSelectionIndex()];
		} else {
			throw new IllegalStateException();
		}
	}

}
