/*
 * Copyright (c) 2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.views.layercomp;

import org.eclipse.jface.resource.ImageRegistry;
import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.TextCellEditor;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Event;

import ch.kuramo.javie.app.Activator;
import ch.kuramo.javie.app.PropertyUtil;
import ch.kuramo.javie.app.project.ModifyLayerPropertyOperation;
import ch.kuramo.javie.app.project.ProjectManager;
import ch.kuramo.javie.app.project.RenameTASelectorOperation;
import ch.kuramo.javie.app.views.LayerCompositionView;
import ch.kuramo.javie.app.widgets.WSWin32;
import ch.kuramo.javie.core.AnimatableDouble;
import ch.kuramo.javie.core.TASelector;
import ch.kuramo.javie.core.TextAnimator;
import ch.kuramo.javie.core.TextLayer;

public abstract class TASelectorElement<T extends TASelector> extends Element {

	public final T selector;

	private final SwitchGroup enableSwitch = new SwitchGroup();

	private TextCellEditor nameEditor;


	public TASelectorElement(TextAnimatorElement parent, T selector) {
		super(parent);
		this.selector = selector;

		enableSwitch.add(new EnableSwitch());

		if (WSWin32.isXPThemed()) {
			enableSwitch.setMarginLeft(11);
		}
	}

	@Override
	public String getColumnText(int columnIndex) {
		switch (columnIndex) {
			case LayerCompositionView.NAME_COL:
				return selector.getName();

			default:
				return null;
		}
	}

	@Override
	public void paintColumn(Event event) {
		switch (event.index) {
			case LayerCompositionView.SHOWHIDE_COL:
				enableSwitch.paint(event);
				break;

			default:
				super.paintColumn(event);
				break;
		}
	}

	@Override
	public void mouseDown(MouseEvent event, int columnIndex) {
		switch (columnIndex) {
			case LayerCompositionView.SHOWHIDE_COL:
				enableSwitch.mouseDown(event);
				break;

			default:
				super.mouseDown(event, columnIndex);
				break;
		}
	}

	@Override
	public boolean canEdit(int columnIndex) {
		switch (columnIndex) {
			case LayerCompositionView.NAME_COL:
				return Boolean.TRUE.equals(viewer.getData(LayerCompositionView.EDIT_ELEMENT_NAME));

			default:
				return false;
		}
	}

	@Override
	public CellEditor getCellEditor(int columnIndex) {
		switch (columnIndex) {
			case LayerCompositionView.NAME_COL:
				if (nameEditor == null) {
					nameEditor = new TextCellEditor(viewer.getTree(), SWT.SINGLE | SWT.BORDER);
					Control control = nameEditor.getControl();
					control.setBackground(control.getDisplay().getSystemColor(SWT.COLOR_WHITE));
				}
				return nameEditor;

			default:
				return null;
		}
	}

	@Override
	public Object getCellEditorValue(int columnIndex) {
		switch (columnIndex) {
			case LayerCompositionView.NAME_COL:
				return selector.getName();

			default:
				return null;
		}
	}

	@Override
	public void setCellEditorValue(int columnIndex, Object value) {
		ProjectManager pm = (ProjectManager) viewer.getData(LayerCompositionView.PROJECT_MANAGER);

		switch (columnIndex) {
			case LayerCompositionView.NAME_COL: {
				String newName = (String) value;
				if (newName.length() > 0) {
					pm.postOperation(new RenameTASelectorOperation(
							pm, getLayer(), getTextAnimator(), selector, newName));
				}
				break;
			}
		}
	}

	protected TextLayer getLayer() {
		return ((TextAnimatorElement) parent).getLayer();
	}

	protected TextAnimator getTextAnimator() {
		return ((TextAnimatorElement) parent).animator;
	}

	private String getCombinedProperty(String property) {
		TextAnimator animator = getTextAnimator();
		int animatorIndex = getLayer().getTextAnimators().indexOf(animator);
		int selectorIndex = animator.getSelectors().indexOf(selector);
		return String.format("textAnimators[%d].selectors[%d].%s", animatorIndex, selectorIndex, property);
	}

	private class TASelectorAnimatableValueElement extends LayerAnimatableValueElement {

		private final String property;

		public TASelectorAnimatableValueElement(String property, String name, String unit) {
			super(TASelectorElement.this, TASelectorElement.this.getLayer(), name, unit);
			this.property = property;
		}

		@Override
		protected String getDefaultExpression() {
			return String.format("textAnimator(\"%s\").selector(\"%s\").%s",
					getTextAnimator().getName().replaceAll("\"", "\\\\\""),
					selector.getName().replaceAll("\"", "\\\\\""), property);
		}

		@Override
		public String getProperty() {
			return getCombinedProperty(property);
		}
	}

	protected LayerAnimatableValueElement createPropertyElement(String property, String name, String unit) {
		return new TASelectorAnimatableValueElement(property, name, unit);
	}

	protected LayerAnimatableValueElement createPropertyElement(String property, String name) {
		return createPropertyElement(property, name, null);
	}

	protected LayerAnimatableValueElement createAngleElement(String property, String name, String unit) {
		return new TASelectorAnimatableValueElement(property, name, unit) {
			@Override
			protected AnimatableValueElementDelegate<?> createDelegate() {
				AnimatableDouble avalue = PropertyUtil.getProperty(layer, getProperty());
				return new AngleElementDelegate(this, name, avalue, unit);
			}
		};
	}

	protected LayerEnumPropertyElement<?> createEnumPropertyElement(final String property, String name) {
		@SuppressWarnings("unchecked")
		LayerEnumPropertyElement<?> element = new LayerEnumPropertyElement(this, getLayer(), name) {
			@Override
			public String getProperty() {
				return getCombinedProperty(property);
			}
		};
		return element;
	}


	private static final ImageRegistry imageRegistry = Activator.getDefault().getImageRegistry();

	private class EnableSwitch implements Switch {

		public boolean hasBorder() {
			return true;
		}

		public Image getImage() {
			return selector.isEnabled() ? imageRegistry.get(Activator.IMG_SWITCH_VIDEO) : null;
		}

		public void mouseDown(MouseEvent event) {
			boolean currentEnabled = selector.isEnabled();
			String label = currentEnabled ? "セレクタを無効にする" : "セレクタを有効にする";

			ProjectManager pm = (ProjectManager) viewer.getData(LayerCompositionView.PROJECT_MANAGER);
			TextLayer layer = getLayer();
			TextAnimator animator = getTextAnimator();
			int animatorIndex = layer.getTextAnimators().indexOf(animator);
			int selectorIndex = animator.getSelectors().indexOf(selector);

			String property = String.format("textAnimators[%d].selectors[%d].enabled", animatorIndex, selectorIndex);
			pm.postOperation(new ModifyLayerPropertyOperation<Boolean>(pm, layer, property, !currentEnabled, label));
		}
	}

}
