/*
 * Copyright (c) 2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.project;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;

import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.TASelector;
import ch.kuramo.javie.core.TextAnimator;
import ch.kuramo.javie.core.TextLayer;

public class RenameTASelectorOperation extends ProjectOperation {

	private final String _compId;

	private final String _layerId;

	private final int _animatorIndex;

	private final int _selectorIndex;

	private final String _oldName;

	private final String _newName;


	public RenameTASelectorOperation(
			ProjectManager pm, TextLayer layer, TextAnimator animator, TASelector selector, String newName) {

		super(pm, "セレクタの名前を変更");

		LayerComposition comp = pm.checkLayer(layer);

		int animatorIndex = layer.getTextAnimators().indexOf(animator);
		if (animatorIndex == -1) {
			throw new IllegalArgumentException("no such TextAnimator found in the layer");
		}

		int selectorIndex = animator.getSelectors().indexOf(selector);
		if (selectorIndex == -1) {
			throw new IllegalArgumentException("no such TASelector found in the TextAnimator");
		}

		_compId = comp.getId();
		_layerId = layer.getId();
		_animatorIndex = animatorIndex;
		_selectorIndex = selectorIndex;
		_oldName = selector.getName();
		_newName = newName;
		_noEffect = newName.equals(_oldName);
	}

	@Override
	protected IStatus executeOrRedo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		setSelectorName(_newName, project, pm);
		return Status.OK_STATUS;
	}

	@Override
	protected IStatus undo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		setSelectorName(_oldName, project, pm);
		return Status.OK_STATUS;
	}

	private void setSelectorName(String name, Project project, ProjectManager pm) {
		LayerComposition comp = project.getComposition(_compId);
		TextLayer layer = (TextLayer) comp.getLayer(_layerId);
		TextAnimator animator = layer.getTextAnimators().get(_animatorIndex);
		TASelector selector = animator.getSelectors().get(_selectorIndex);

		selector.setName(name);

		if (pm != null) {
			pm.fireLayerPropertyChange(layer, String.format(
					"textAnimators[%d].selectors[%d].name", _animatorIndex, _selectorIndex));
		}
	}

}
