/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.project;

import java.util.List;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;

import ch.kuramo.javie.core.Layer;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.MediaLayer;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.TrackMatte;

public class ModifyLayerTrackMatteOperation extends ProjectOperation {

	private final String _compId;

	private final String _layerId;

	private final TrackMatte _oldTrackMatte;

	private final TrackMatte _newTrackMatte;

	private final boolean _oldVideoEnabled;

	private final boolean _newVideoEnabled;


	public ModifyLayerTrackMatteOperation(ProjectManager projectManager, MediaLayer layer, TrackMatte newTrackMatte) {
		super(projectManager, "トラックマットの変更");

		LayerComposition comp = projectManager.checkLayer(layer);
		MediaLayer matteLayer = getTrackMatteLayer(comp, layer);
		if (matteLayer == null) {
			throw new IllegalArgumentException();
		}

		TrackMatte oldTrackMatte = layer.getTrackMatte();
		_noEffect = newTrackMatte.equals(oldTrackMatte);

		_compId = comp.getId();
		_layerId = layer.getId();
		_oldTrackMatte = oldTrackMatte;
		_newTrackMatte = newTrackMatte;
		_oldVideoEnabled = matteLayer.isVideoEnabled();
		_newVideoEnabled = (_oldTrackMatte == TrackMatte.NONE && !_noEffect) ? false : _oldVideoEnabled;
	}

	private MediaLayer getTrackMatteLayer(LayerComposition comp, MediaLayer layer) {
		List<Layer> layers = comp.getLayers();
		int layerIndex = layers.indexOf(layer);
		if (layerIndex + 1 < layers.size()) {
			Layer matteLayer = layers.get(layerIndex + 1);
			if (matteLayer instanceof MediaLayer) {
				return (MediaLayer) matteLayer;
			}
		}
		return null;
	}

	@Override
	protected IStatus executeOrRedo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		modifyTrackMatte(_newTrackMatte, _newVideoEnabled, project, pm);
		return Status.OK_STATUS;
	}

	@Override
	protected IStatus undo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		modifyTrackMatte(_oldTrackMatte, _oldVideoEnabled, project, pm);
		return Status.OK_STATUS;
	}

	private void modifyTrackMatte(TrackMatte trackMatte, boolean videoEnabled, Project project, ProjectManager pm) {
		LayerComposition comp = project.getComposition(_compId);
		MediaLayer layer = (MediaLayer) comp.getLayer(_layerId);
		MediaLayer matteLayer = getTrackMatteLayer(comp, layer);

		if (matteLayer.isVideoEnabled() != videoEnabled) {
			matteLayer.setVideoEnabled(videoEnabled);
			if (pm != null) {
				pm.fireLayerPropertyChange(matteLayer, "videoEnabled");
			}
		}

		layer.setTrackMatte(trackMatte);
		if (pm != null) {
			pm.fireLayerPropertyChange(layer, "trackMatte");
		}
	}

}
