/*
 * Copyright (c) 2010,2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.time;

import java.util.EnumMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import javax.media.opengl.GLUniformData;

import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IAnimatableEnum;
import ch.kuramo.javie.api.IAnimatableInteger;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.annotations.ShaderSource;
import ch.kuramo.javie.api.annotations.Effect.Categories;
import ch.kuramo.javie.api.services.IAlphaChannelSupport;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.api.services.IVideoRenderSupport;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.Echo", category=Categories.TIME)
public class Echo {

	public enum EchoOperator {
		ADD,
		MAXIMUM,
		MINIMUM,
		SCREEN,
		COMPOSITE_IN_BACK,
		COMPOSITE_IN_FRONT,
		//BLEND
	}


	@Property("-0.0333")
	private IAnimatableDouble echoTime;

	@Property(value="1", min="0")
	private IAnimatableInteger numberOfEchoes;

	@Property(value="1", min="0", max="1")
	private IAnimatableDouble startingIntensity;

	@Property(value="1", min="0")
	private IAnimatableDouble decay;

	@Property
	private IAnimatableEnum<EchoOperator> echoOperator;


	private final IVideoEffectContext context;

	private final IVideoRenderSupport support;

	private final IAlphaChannelSupport alphaSupport;

	private final Map<EchoOperator, IShaderProgram> programs = new EnumMap<EchoOperator, IShaderProgram>(EchoOperator.class);

	@Inject
	public Echo(
			IVideoEffectContext context, IVideoRenderSupport support,
			IAlphaChannelSupport alphaSupport, IShaderRegistry shaders) {

		this.context = context;
		this.support = support;
		this.alphaSupport = alphaSupport;

		programs.put(EchoOperator.ADD,					shaders.getProgram(Echo.class, "ADD"));
		programs.put(EchoOperator.MAXIMUM,				shaders.getProgram(Echo.class, "MAXIMUM"));
		programs.put(EchoOperator.MINIMUM,				shaders.getProgram(Echo.class, "MINIMUM"));
		programs.put(EchoOperator.SCREEN,				shaders.getProgram(Echo.class, "SCREEN"));
		programs.put(EchoOperator.COMPOSITE_IN_BACK,	shaders.getProgram(Echo.class, "COMPOSITE"));
		programs.put(EchoOperator.COMPOSITE_IN_FRONT,	shaders.getProgram(Echo.class, "COMPOSITE"));
	}

	public VideoBounds getVideoBounds() {
		Time time = context.getTime();
		double echoTime = context.value(this.echoTime);
		int numberOfEchoes = context.value(this.numberOfEchoes);

		VideoBounds bounds = null;
		for (int i = 0; i <= numberOfEchoes; ++i) {
			Time t = time.add(new Time(Math.round(i*echoTime*time.timeScale), time.timeScale));
			context.setTime(t);

			VideoBounds b = context.getPreviousBounds();

			if (!b.isEmpty()) {
				if (bounds == null) {
					bounds = b;
				} else {
					double left = Math.min(b.x, bounds.x);
					double top = Math.min(b.y, bounds.y);
					double right = Math.max(b.x+b.width, bounds.x+bounds.width);
					double bottom = Math.max(b.y+b.height, bounds.y+bounds.height);
					bounds = new VideoBounds(left, top, (int)Math.ceil(right-left), (int)Math.ceil(bottom-top));
				}
			}
		}

		if (bounds == null) {
			bounds = new VideoBounds(0, 0);
		}

		return bounds;
	}

	public IVideoBuffer doVideoEffect() {
		Time time = context.getTime();
		double echoTime = context.value(this.echoTime);
		int numberOfEchoes = context.value(this.numberOfEchoes);

		double startingIntensity = context.value(this.startingIntensity);
		double decay = context.value(this.decay);
		EchoOperator op = context.value(this.echoOperator);

		IVideoBuffer dstIn = null;
		try {
			for (int i = 0; i <= numberOfEchoes; ++i) {
				int echoIndex = (op == EchoOperator.COMPOSITE_IN_FRONT) ? numberOfEchoes-i : i;
				double intensity = Math.min(1.0, startingIntensity * Math.pow(decay, echoIndex));
				Time t = time.add(new Time(Math.round(echoIndex*echoTime*time.timeScale), time.timeScale));
				context.setTime(t);

				Set<IVideoBuffer> tmpBuffers = new HashSet<IVideoBuffer>();
				try {
					IVideoBuffer src = context.doPreviousEffect();
					tmpBuffers.add(src);

					if (src.getBounds().isEmpty()) {
						continue;
					}

					if (dstIn == null) {
						dstIn = context.createVideoBuffer(src.getBounds());
						dstIn.clear();
					}

					VideoBounds b1 = src.getBounds();
					VideoBounds b2 = dstIn.getBounds();
					double left = Math.min(b1.x, b2.x);
					double top = Math.min(b1.y, b2.y);
					double right = Math.max(b1.x+b1.width, b2.x+b2.width);
					double bottom = Math.max(b1.y+b1.height, b2.y+b2.height);
					VideoBounds bounds = new VideoBounds(left, top, (int)Math.ceil(right-left), (int)Math.ceil(bottom-top));

					IVideoBuffer dstOut = context.createVideoBuffer(bounds);
					tmpBuffers.add(dstOut);
					dstOut.clear();

					Set<GLUniformData> uniforms = new HashSet<GLUniformData>();
					uniforms.add(new GLUniformData("texDst", 0));
					uniforms.add(new GLUniformData("texSrc", 1));
					uniforms.add(new GLUniformData("intensity", (float)intensity));

					support.useShaderProgram(programs.get(op), uniforms, dstOut, dstIn, src);

					tmpBuffers.add(dstIn);
					tmpBuffers.remove(dstOut);
					dstIn = dstOut;

				} finally {
					for (IVideoBuffer vb : tmpBuffers) {
						vb.dispose();
					}
				}
			}

			if (dstIn == null) {
				// FIXME (0, 0) ではなく src の VideoBounds で作らないと、
				//       同じ空の VideoBounds でも後続のエフェクトの結果などが異なることがある。
				return context.createVideoBuffer(new VideoBounds(0, 0));
			}

			alphaSupport.premultiply(dstIn);

			IVideoBuffer result = dstIn;
			dstIn = null;
			return result;

		} finally {
			if (dstIn != null) dstIn.dispose();
		}
	}

	private static final String[] createProgramSource(String name) {
		return new String[] {
								"uniform sampler2D texDst;",
								"uniform sampler2D texSrc;",
								"uniform float intensity;",
								"",
				String.format(	"vec4 blend_%s(vec4 pDst, vec4 pSrc, float intensity);", name),
								"",
								"void main(void)",
								"{",
								"	vec4 dst = texture2D(texDst, gl_TexCoord[0].st);",
								"	vec4 src = texture2D(texSrc, gl_TexCoord[1].st);",
				String.format(	"	dst = blend_%s(vec4(dst.rgb*dst.a, dst.a), src, intensity);", name),
								"	gl_FragColor = (dst.a != 0.0) ? vec4(dst.rgb/dst.a, dst.a) : vec4(0.0);",
								"}"
		};
	}

	@ShaderSource(attach="ch.kuramo.javie.core.shaders.BlendModeShaders.blend_functions")
	public static final String[] ADD = createProgramSource("add");

	@ShaderSource(attach="ch.kuramo.javie.core.shaders.BlendModeShaders.blend_functions")
	public static final String[] MAXIMUM = createProgramSource("lighten");

	@ShaderSource(attach="ch.kuramo.javie.core.shaders.BlendModeShaders.blend_functions")
	public static final String[] MINIMUM = createProgramSource("darken");

	@ShaderSource(attach="ch.kuramo.javie.core.shaders.BlendModeShaders.blend_functions")
	public static final String[] SCREEN = createProgramSource("screen");

	@ShaderSource(attach="ch.kuramo.javie.core.shaders.BlendModeShaders.blend_functions")
	public static final String[] COMPOSITE = createProgramSource("normal");

}
