/*
 * Copyright (c) 2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.stylize;

import java.nio.FloatBuffer;
import java.util.HashSet;
import java.util.Set;

import javax.media.opengl.GL2;
import javax.media.opengl.GLUniformData;

import ch.kuramo.javie.api.IAnimatableBoolean;
import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IAnimatableEnum;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Resolution;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.IVideoBuffer.TextureWrapMode;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.annotations.ShaderSource;
import ch.kuramo.javie.api.annotations.Effect.Categories;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.api.services.IVideoRenderSupport;
import ch.kuramo.javie.effects.noiseGrain.NoiseTexture;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.Scatter", category=Categories.STYLIZE)
public class Scatter {

	public enum Grain { BOTH, HORIZONTAL, VERTICAL }


	@Property(min="0", max="1000")
	private IAnimatableDouble scatterAmount;

	@Property
	private IAnimatableEnum<Grain> grain;

	@Property
	private IAnimatableBoolean scatterRandomness;


	private final IVideoEffectContext context;

	private final IVideoRenderSupport support;

	private final IShaderProgram[] programs;

	@Inject
	public Scatter(IVideoEffectContext context,
			IVideoRenderSupport support, IShaderRegistry shaders) {

		this.context = context;
		this.support = support;

		programs = new IShaderProgram[3];
		for (Grain g : Grain.values()) {
			programs[g.ordinal()] = shaders.getProgram(Scatter.class, g.name());
		}
	}

	public IVideoBuffer doVideoEffect() {
		IVideoBuffer source = context.doPreviousEffect();
		final VideoBounds bounds = source.getBounds();
		if (bounds.isEmpty()) {
			return source;
		}

		Resolution resolution = context.getVideoResolution();
		double amount = resolution.scale(context.value(this.scatterAmount));
		if (amount == 0) {
			return source;
		}

		Grain grain = context.value(this.grain);
		double random = (context.getEffectName().hashCode() % 5023)
				+ (context.value(this.scatterRandomness) ? context.getTime().toSecond() : 0);

		try {
			Set<GLUniformData> uniforms = new HashSet<GLUniformData>();
			uniforms.add(new GLUniformData("texture", 0));
			uniforms.add(new GLUniformData("permTexture", 1));
			uniforms.add(new GLUniformData("size", 2, toFloatBuffer(bounds.width, bounds.height)));
			uniforms.add(new GLUniformData("amount", (float)amount));
			uniforms.add(new GLUniformData("random", (float)random));

			Runnable operation = new Runnable() {
				public void run() {
					GL2 gl = context.getGL().getGL2();
					int permTexture = NoiseTexture.getPermTexture(gl);

					gl.glActiveTexture(GL2.GL_TEXTURE1);
					gl.glBindTexture(GL2.GL_TEXTURE_2D, permTexture);

					support.ortho2D(bounds);
					support.quad2D(bounds);
				}
			};

			source.setTextureWrapMode(TextureWrapMode.MIRRORED_REPEAT);

			int pushAttribs = GL2.GL_TEXTURE_BIT;
			return support.useShaderProgram(programs[grain.ordinal()],
							uniforms, operation, pushAttribs, null, source);

		} finally {
			source.dispose();
		}
	}

	private FloatBuffer toFloatBuffer(double...values) {
		float[] farray = new float[values.length];
		for (int i = 0; i < values.length; ++i) {
			farray[i] = (float)values[i];
		}
		return FloatBuffer.wrap(farray);
	}


	// FIXME noise関数はFractalNoiseから独立させる。
	private static final String NOISE_FUNC = "ch.kuramo.javie.effects.noiseGrain.FractalNoise.noise";

	@ShaderSource(attach=NOISE_FUNC)
	public static final String[] BOTH = createProgramSource(Grain.BOTH);

	@ShaderSource(attach=NOISE_FUNC)
	public static final String[] HORIZONTAL = createProgramSource(Grain.HORIZONTAL);

	@ShaderSource(attach=NOISE_FUNC)
	public static final String[] VERTICAL = createProgramSource(Grain.VERTICAL);


	private static String[] createProgramSource(Grain grain) {
		boolean h = (grain != Grain.VERTICAL);
		boolean v = (grain != Grain.HORIZONTAL);
		return new String[] {
			h ? "#define HORIZONTAL" : "",
			v ? "#define VERTICAL" : "",
				"",
				"uniform sampler2D texture;",
				"uniform vec2 size;",
				"uniform float amount;",
				"uniform float random;",
				"",
				"float noise(vec3 P);",
				"",
				"void main(void)",
				"{",
				"	vec2 xy = gl_FragCoord.xy;",
				"	float nh, nv;",
				"",
				"#ifdef HORIZONTAL",
				"		nh = noise(vec3(xy, random)+vec3(124.543,79.777,267.203));",
				"#else",
				"		nh = 0.0;",
				"#endif",
				"",
				"#ifdef VERTICAL",
				"		nv = noise(vec3(xy, -random)+vec3(91.229,36.791,642.877));",
				"#else",
				"		nv = 0.0;",
				"#endif",
				"",
				"	vec2 tc = (xy + vec2(nh, nv)*amount) / size;",
				"	gl_FragColor = texture2D(texture, tc);",
				"}"
		};
	}

}
