/*
 * Copyright (c) 2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#import "QCCompositionSourceJNI.h"
#import <QuartzComposer/QCRenderer.h>


#define JLONG(p) ((jlong) (pointer_t) p)
#define POINTER(type, p) ((type) (pointer_t) p)


JNIEXPORT jint JNICALL JNI_OnLoad(JavaVM* vm, void* reserved)
{
	return JNI_VERSION_1_4;
}

NSString* jstringToNSString(JNIEnv* env, jstring jstr)
{
	const jchar* chars = (*env)->GetStringChars(env, jstr, NULL);
	NSString* nsstr = [NSString stringWithCharacters:(UniChar*)chars
											  length:(*env)->GetStringLength(env, jstr)];
	(*env)->ReleaseStringChars(env, jstr, chars);
	return nsstr;
}

typedef struct {
	QCRenderer* qcRenderer;
	id savedInputValues;
} QCCompositionSourceRec;

JNIEXPORT jlong JNICALL Java_ch_kuramo_javie_core_internal_QCCompositionSource_initQCRenderer
(JNIEnv* env, jobject jthis, jstring jfilename, jlong pixelFormat)
{
	jlong result = 0;
	NSAutoreleasePool* pool = [[NSAutoreleasePool alloc] init];
	
	QCComposition* qcComposition = [QCComposition compositionWithFile:jstringToNSString(env, jfilename)];
	if (qcComposition == nil) {
		NSLog(@"QCComposition compositionWithFile: failed");
		goto bail;
	}
	
	QCRenderer* qcRenderer = [[QCRenderer alloc] initWithCGLContext:CGLGetCurrentContext()
														pixelFormat:[POINTER(NSOpenGLPixelFormat*, pixelFormat) CGLPixelFormatObj]
														 colorSpace:[[NSColorSpace genericRGBColorSpace] CGColorSpace]
														composition:qcComposition];
	if (qcRenderer == nil) {
		NSLog(@"QCRenderer initWithCGLContext:pixelFormat:colorSpace:composition: failed");
		goto bail;
	}
	
	QCCompositionSourceRec* p = malloc(sizeof(QCCompositionSourceRec));
	if (p) {
		p->qcRenderer = qcRenderer;
		p->savedInputValues = [qcRenderer propertyListFromInputValues];
		CFRetain(p->savedInputValues);
		result = JLONG(p);
	} else {
		[qcRenderer release];
	}
	
bail:
	[pool release];
	return result;
}

JNIEXPORT void JNICALL Java_ch_kuramo_javie_core_internal_QCCompositionSource_disposeQCRenderer
(JNIEnv* env, jobject jthis, jlong qcRendererAddress)
{
	NSAutoreleasePool* pool = [[NSAutoreleasePool alloc] init];
	
	QCCompositionSourceRec* p = POINTER(QCCompositionSourceRec*, qcRendererAddress);
	[p->qcRenderer release];
	CFRelease(p->savedInputValues);
	free(p);
	
	[pool release];
}

JNIEXPORT jboolean JNICALL Java_ch_kuramo_javie_core_internal_QCCompositionSource_renderAtTime
(JNIEnv* env, jobject jthis, jlong qcRendererAddress, jdouble time, jobjectArray inputKeyValues)
{
	NSAutoreleasePool* pool = [[NSAutoreleasePool alloc] init];
	
	QCCompositionSourceRec* p = POINTER(QCCompositionSourceRec*, qcRendererAddress);
	QCRenderer* qcRenderer = p->qcRenderer;
	
	NSArray* inputKeys = [qcRenderer inputKeys];
	if ([inputKeys count] > 0) {
		[qcRenderer setInputValuesWithPropertyList:p->savedInputValues];
		
		jsize n = (*env)->GetArrayLength(env, inputKeyValues);
		if (n > 0) {
			jclass clsBoolean = (*env)->FindClass(env, "Ljava/lang/Boolean;");
			jclass clsInteger = (*env)->FindClass(env, "Ljava/lang/Integer;");
			jclass clsDouble = (*env)->FindClass(env, "Ljava/lang/Double;");
			jclass clsString = (*env)->FindClass(env, "Ljava/lang/String;");
			jclass clsColor = (*env)->FindClass(env, "Lch/kuramo/javie/api/Color;");
			NSColorSpace* colorSpace = nil;
			
			for (jsize i = 0; i < n; i += 2) {
				NSString* key = jstringToNSString(env, (*env)->GetObjectArrayElement(env, inputKeyValues, i));
				if ([inputKeys indexOfObject:key] == NSNotFound) {
					continue;
				}
				
				jobject obj = (*env)->GetObjectArrayElement(env, inputKeyValues, i+1);
				
				if ((*env)->IsInstanceOf(env, obj, clsBoolean)) {
					jmethodID m = (*env)->GetMethodID(env, clsBoolean, "booleanValue", "()Z");
					jboolean val = (*env)->CallBooleanMethod(env, obj, m);
					[qcRenderer setValue:[NSNumber numberWithBool:val] forInputKey:key];
					
				} else if ((*env)->IsInstanceOf(env, obj, clsInteger)) {
					jmethodID m = (*env)->GetMethodID(env, clsInteger, "intValue", "()I");
					jint val = (*env)->CallIntMethod(env, obj, m);
					[qcRenderer setValue:[NSNumber numberWithInt:val] forInputKey:key];
					
				} else if ((*env)->IsInstanceOf(env, obj, clsDouble)) {
					jmethodID m = (*env)->GetMethodID(env, clsDouble, "doubleValue", "()D");
					jdouble val = (*env)->CallDoubleMethod(env, obj, m);
					[qcRenderer setValue:[NSNumber numberWithDouble:val] forInputKey:key];
					
				} else if ((*env)->IsInstanceOf(env, obj, clsString)) {
					[qcRenderer setValue:jstringToNSString(env, obj) forInputKey:key];
					
				} else if ((*env)->IsInstanceOf(env, obj, clsColor)) {
					jfieldID fr = (*env)->GetFieldID(env, clsColor, "r", "D");
					jfieldID fg = (*env)->GetFieldID(env, clsColor, "g", "D");
					jfieldID fb = (*env)->GetFieldID(env, clsColor, "b", "D");
					jfieldID fa = (*env)->GetFieldID(env, clsColor, "a", "D");
					jdouble r = (*env)->GetDoubleField(env, obj, fr);
					jdouble g = (*env)->GetDoubleField(env, obj, fg);
					jdouble b = (*env)->GetDoubleField(env, obj, fb);
					jdouble a = (*env)->GetDoubleField(env, obj, fa);
					
					if (colorSpace == nil) {
						colorSpace = [NSColorSpace genericRGBColorSpace];
					}
					CGFloat rgba[4] = { r, g, b, a };
					NSColor* color = [NSColor colorWithColorSpace:colorSpace components:rgba count:4];
					[qcRenderer setValue:color forInputKey:key];
				}
			}
		}
	}
	
	BOOL successful = [qcRenderer renderAtTime:time arguments:nil];
	
	[pool release];
	return successful;
}
