/*
 * Copyright (c) 2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal;

import java.util.Collection;
import java.util.List;
import java.util.Set;

import ch.kuramo.javie.api.Color;
import ch.kuramo.javie.api.Vec2d;
import ch.kuramo.javie.api.Vec3d;
import ch.kuramo.javie.core.AbstractAnimatableEnum;
import ch.kuramo.javie.core.AnimatableColor;
import ch.kuramo.javie.core.AnimatableDouble;
import ch.kuramo.javie.core.AnimatableInteger;
import ch.kuramo.javie.core.AnimatableVec2d;
import ch.kuramo.javie.core.AnimatableVec3d;
import ch.kuramo.javie.core.CoreContext;
import ch.kuramo.javie.core.ExpressionScope;
import ch.kuramo.javie.core.Expressionee;
import ch.kuramo.javie.core.Keyframe;
import ch.kuramo.javie.core.TAProperty;
import ch.kuramo.javie.core.TASelector;
import ch.kuramo.javie.core.TextAnimator;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.annotations.ProjectElement;

@ProjectElement("textAnimator")
public class TextAnimatorImpl implements TextAnimator {

	private boolean enabled = true;

	private String name = "アニメータ";

	private List<TASelector> selectors = Util.newList();

	private Set<TAProperty> properties = Util.newSet();

	private AnimatableVec3d anchorPoint = new AnimatableVec3d(Vec3d.ZERO);

	private AnimatableVec3d position = new AnimatableVec3d(Vec3d.ZERO);

	private AnimatableVec3d scale = new AnimatableVec3d(new Vec3d(100, 100, 100));

	private AnimatableDouble skew = new AnimatableDouble(0d, -70d, 70d);

	private AnimatableDouble skewAxis = new AnimatableDouble(0d);

	private AnimatableDouble rotationX = new AnimatableDouble(0d);

	private AnimatableDouble rotationY = new AnimatableDouble(0d);

	private AnimatableDouble rotationZ = new AnimatableDouble(0d);

	private AnimatableDouble opacity = new AnimatableDouble(100d, 0d, 100d);

	private AnimatableColor fillRGB = new AnimatableColor(Color.WHITE);

	private AnimatableDouble fillHue = new AnimatableDouble(0d);

	private AnimatableDouble fillSaturation = new AnimatableDouble(0d, -100d, 100d);

	private AnimatableDouble fillLuminosity = new AnimatableDouble(0d, -100d, 100d);

	private AnimatableDouble fillOpacity = new AnimatableDouble(100d, 0d, 100d);

	private AnimatableColor strokeRGB = new AnimatableColor(new Color(1, 0, 0));

	private AnimatableDouble strokeHue = new AnimatableDouble(0d);

	private AnimatableDouble strokeSaturation = new AnimatableDouble(0d, -100d, 100d);

	private AnimatableDouble strokeLuminosity = new AnimatableDouble(0d, -100d, 100d);

	private AnimatableDouble strokeOpacity = new AnimatableDouble(100d, 0d, 100d);

	private AnimatableDouble strokeWidth = new AnimatableDouble(0d, -1000d, 1000d);

	private AnimatableDouble tracking = new AnimatableDouble(0d);

	private AnimatableVec2d lineSpacing = new AnimatableVec2d(Vec2d.ZERO);

	private AnimatableCharacterRange characterRange = new AnimatableCharacterRange(CharacterRange.GROUP);

	private AnimatableInteger characterValue = new AnimatableInteger(0, 0, CharacterGroup.MAX_CODE_POINT);

	private AnimatableInteger characterOffset = new AnimatableInteger(0);

	private AnimatableVec2d blur = new AnimatableVec2d(Vec2d.ZERO, Vec2d.ZERO, new Vec2d(500, 500));


	public boolean isEnabled() {
		return enabled;
	}

	public void setEnabled(boolean enabled) {
		this.enabled = enabled;
	}

	public String getName() {
		return name;
	}

	public void setName(String name) {
		this.name = name;
	}

	public List<TASelector> getSelectors() {
		return selectors;
	}

	public void setSelectors(List<TASelector> selectors) {
		this.selectors = selectors;
	}

	public Set<TAProperty> getProperties() {
		return properties;
	}

	public void setProperties(Set<TAProperty> properties) {
		this.properties = properties;
	}

	public AnimatableVec3d getAnchorPoint() {
		return anchorPoint;
	}

	public void setAnchorPoint(AnimatableVec3d anchorPoint) {
		anchorPoint.copyConfigurationFrom(this.anchorPoint);
		this.anchorPoint = anchorPoint;
	}

	public AnimatableVec3d getPosition() {
		return position;
	}

	public void setPosition(AnimatableVec3d position) {
		position.copyConfigurationFrom(this.position);
		this.position = position;
	}

	public AnimatableVec3d getScale() {
		return scale;
	}

	public void setScale(AnimatableVec3d scale) {
		scale.copyConfigurationFrom(this.scale);
		this.scale = scale;
	}

	public AnimatableDouble getSkew() {
		return skew;
	}

	public void setSkew(AnimatableDouble skew) {
		skew.copyConfigurationFrom(this.skew);
		this.skew = skew;
	}

	public AnimatableDouble getSkewAxis() {
		return skewAxis;
	}

	public void setSkewAxis(AnimatableDouble skewAxis) {
		skewAxis.copyConfigurationFrom(this.skewAxis);
		this.skewAxis = skewAxis;
	}

	public AnimatableDouble getRotationX() {
		return rotationX;
	}

	public void setRotationX(AnimatableDouble rotationX) {
		rotationX.copyConfigurationFrom(this.rotationX);
		this.rotationX = rotationX;
	}

	public AnimatableDouble getRotationY() {
		return rotationY;
	}

	public void setRotationY(AnimatableDouble rotationY) {
		rotationY.copyConfigurationFrom(this.rotationY);
		this.rotationY = rotationY;
	}

	public AnimatableDouble getRotationZ() {
		return rotationZ;
	}

	public void setRotationZ(AnimatableDouble rotationZ) {
		rotationZ.copyConfigurationFrom(this.rotationZ);
		this.rotationZ = rotationZ;
	}

	public AnimatableDouble getOpacity() {
		return opacity;
	}

	public void setOpacity(AnimatableDouble opacity) {
		opacity.copyConfigurationFrom(this.opacity);
		this.opacity = opacity;
	}

	public AnimatableColor getFillRGB() {
		return fillRGB;
	}

	public void setFillRGB(AnimatableColor fillRGB) {
		fillRGB.copyConfigurationFrom(this.fillRGB);
		this.fillRGB = fillRGB;
	}

	public AnimatableDouble getFillHue() {
		return fillHue;
	}

	public void setFillHue(AnimatableDouble fillHue) {
		fillHue.copyConfigurationFrom(this.fillHue);
		this.fillHue = fillHue;
	}

	public AnimatableDouble getFillSaturation() {
		return fillSaturation;
	}

	public void setFillSaturation(AnimatableDouble fillSaturation) {
		fillSaturation.copyConfigurationFrom(this.fillSaturation);
		this.fillSaturation = fillSaturation;
	}

	public AnimatableDouble getFillLuminosity() {
		return fillLuminosity;
	}

	public void setFillLuminosity(AnimatableDouble fillLuminosity) {
		fillLuminosity.copyConfigurationFrom(this.fillLuminosity);
		this.fillLuminosity = fillLuminosity;
	}

	public AnimatableDouble getFillOpacity() {
		return fillOpacity;
	}

	public void setFillOpacity(AnimatableDouble fillOpacity) {
		fillOpacity.copyConfigurationFrom(this.fillOpacity);
		this.fillOpacity = fillOpacity;
	}

	public AnimatableColor getStrokeRGB() {
		return strokeRGB;
	}

	public void setStrokeRGB(AnimatableColor strokeRGB) {
		strokeRGB.copyConfigurationFrom(this.strokeRGB);
		this.strokeRGB = strokeRGB;
	}

	public AnimatableDouble getStrokeHue() {
		return strokeHue;
	}

	public void setStrokeHue(AnimatableDouble strokeHue) {
		strokeHue.copyConfigurationFrom(this.strokeHue);
		this.strokeHue = strokeHue;
	}

	public AnimatableDouble getStrokeSaturation() {
		return strokeSaturation;
	}

	public void setStrokeSaturation(AnimatableDouble strokeSaturation) {
		strokeSaturation.copyConfigurationFrom(this.strokeSaturation);
		this.strokeSaturation = strokeSaturation;
	}

	public AnimatableDouble getStrokeLuminosity() {
		return strokeLuminosity;
	}

	public void setStrokeLuminosity(AnimatableDouble strokeLuminosity) {
		strokeLuminosity.copyConfigurationFrom(this.strokeLuminosity);
		this.strokeLuminosity = strokeLuminosity;
	}

	public AnimatableDouble getStrokeOpacity() {
		return strokeOpacity;
	}

	public void setStrokeOpacity(AnimatableDouble strokeOpacity) {
		strokeOpacity.copyConfigurationFrom(this.strokeOpacity);
		this.strokeOpacity = strokeOpacity;
	}

	public AnimatableDouble getStrokeWidth() {
		return strokeWidth;
	}

	public void setStrokeWidth(AnimatableDouble strokeWidth) {
		strokeWidth.copyConfigurationFrom(this.strokeWidth);
		this.strokeWidth = strokeWidth;
	}

	public AnimatableDouble getTracking() {
		return tracking;
	}

	public void setTracking(AnimatableDouble tracking) {
		tracking.copyConfigurationFrom(this.tracking);
		this.tracking = tracking;
	}

	public AnimatableVec2d getLineSpacing() {
		return lineSpacing;
	}

	public void setLineSpacing(AnimatableVec2d lineSpacing) {
		lineSpacing.copyConfigurationFrom(this.lineSpacing);
		this.lineSpacing = lineSpacing;
	}

	public AnimatableCharacterRange getCharacterRange() {
		return characterRange;
	}

	public void setCharacterRange(AnimatableCharacterRange characterRange) {
		characterRange.copyConfigurationFrom(this.characterRange);
		this.characterRange = characterRange;
	}

	public AnimatableInteger getCharacterValue() {
		return characterValue;
	}

	public void setCharacterValue(AnimatableInteger characterValue) {
		characterValue.copyConfigurationFrom(this.characterValue);
		this.characterValue = characterValue;
	}

	public AnimatableInteger getCharacterOffset() {
		return characterOffset;
	}

	public void setCharacterOffset(AnimatableInteger characterOffset) {
		characterOffset.copyConfigurationFrom(this.characterOffset);
		this.characterOffset = characterOffset;
	}

	public AnimatableVec2d getBlur() {
		return blur;
	}

	public void setBlur(AnimatableVec2d blur) {
		blur.copyConfigurationFrom(this.blur);
		this.blur = blur;
	}

	public void prepareExpression(ExpressionScope scope) {
		for (TASelector selector : selectors) {
			// scopeは同じ。
			selector.prepareExpression(scope);
		}

		scope.assignTo(anchorPoint);
		scope.assignTo(position);
		scope.assignTo(scale);
		scope.assignTo(skew);
		scope.assignTo(skewAxis);
		scope.assignTo(rotationX);
		scope.assignTo(rotationY);
		scope.assignTo(rotationZ);
		scope.assignTo(opacity);

		scope.assignTo(fillRGB);
		scope.assignTo(fillHue);
		scope.assignTo(fillSaturation);
		scope.assignTo(fillLuminosity);
		scope.assignTo(fillOpacity);
		scope.assignTo(strokeRGB);
		scope.assignTo(strokeHue);
		scope.assignTo(strokeSaturation);
		scope.assignTo(strokeLuminosity);
		scope.assignTo(strokeOpacity);
		scope.assignTo(strokeWidth);

		scope.assignTo(tracking);
		scope.assignTo(lineSpacing);

		scope.assignTo(characterRange);
		scope.assignTo(characterValue);
		scope.assignTo(characterOffset);

		scope.assignTo(blur);
	}

	public Object createExpressionElement(CoreContext context) {
		return new TextAnimatorExpressionElement(context);
	}

	public class TextAnimatorExpressionElement {

		private final CoreContext context;

		public TextAnimatorExpressionElement(CoreContext context) {
			this.context = context;
		}

		public Object getSelectors() {
			Object[] array = new Object[selectors.size()];
			for (int i = 0; i < array.length; ++i) {
				array[i] = elem(selectors.get(i));
			}
			return array;
		}

		public Object selector(int index) {
			return elem(selectors.get(index - 1));
		}

		public Object selector(String name) {
			for (TASelector selector : selectors) {
				if (selector.getName().equals(name)) {
					return elem(selector);
				}
			}
			return null;
		}

		// TODO propertiesをエクスプレッションから参照できるようにする。

		public Object getAnchorPoint()		{ return elem(anchorPoint); }
		public Object getPosition()			{ return elem(position); }
		public Object getScale()			{ return elem(scale); }
		public Object getSkew()				{ return elem(skew); }
		public Object getSkewAxis()			{ return elem(skewAxis); }
		public Object getRotationX()		{ return elem(rotationX); }
		public Object getRotationY()		{ return elem(rotationY); }
		public Object getRotationZ()		{ return elem(rotationZ); }
		public Object getRotation()			{ return getRotationZ(); }
		public Object getOpacity()			{ return elem(opacity); }
		public Object getFillRGB()			{ return elem(fillRGB); }
		public Object getFillHue()			{ return elem(fillHue); }
		public Object getFillSaturation()	{ return elem(fillSaturation); }
		public Object getFillLuminosity()	{ return elem(fillLuminosity); }
		public Object getFillOpacity()		{ return elem(fillOpacity); }
		public Object getStrokeRGB()		{ return elem(strokeRGB); }
		public Object getStrokeHue()		{ return elem(strokeHue); }
		public Object getStrokeSaturation()	{ return elem(strokeSaturation); }
		public Object getStrokeLuminosity()	{ return elem(strokeLuminosity); }
		public Object getStrokeOpacity()	{ return elem(strokeOpacity); }
		public Object getStrokeWidth()		{ return elem(strokeWidth); }
		public Object getTracking()			{ return elem(tracking); }
		public Object getLineSpacing()		{ return elem(lineSpacing); }
		public Object getCharacterRange()	{ return elem(characterRange); }
		public Object getCharacterValue()	{ return elem(characterValue); }
		public Object getCharacterOffset()	{ return elem(characterOffset); }
		public Object getBlur()				{ return elem(blur); }

		private <T> T elem(Expressionee exprnee) {
			return context.getExpressionElement(exprnee);
		}
	}

	public static class AnimatableAnchorPointGrouping
			extends AbstractAnimatableEnum<AnchorPointGrouping> {

		public AnimatableAnchorPointGrouping(
				AnchorPointGrouping staticValue,
				Collection<Keyframe<AnchorPointGrouping>> keyframes,
				String expression) {

			super(AnchorPointGrouping.class, staticValue, keyframes, expression);
		}

		public AnimatableAnchorPointGrouping(AnchorPointGrouping defaultValue) {
			super(AnchorPointGrouping.class, defaultValue);
		}

	}

	public static class AnimatableCharacterAlignment
			extends AbstractAnimatableEnum<CharacterAlignment> {

		public AnimatableCharacterAlignment(
				CharacterAlignment staticValue,
				Collection<Keyframe<CharacterAlignment>> keyframes,
				String expression) {

			super(CharacterAlignment.class, staticValue, keyframes, expression);
		}

		public AnimatableCharacterAlignment(CharacterAlignment defaultValue) {
			super(CharacterAlignment.class, defaultValue);
		}

	}

	public static class AnimatableCharacterRange
			extends AbstractAnimatableEnum<CharacterRange> {

		public AnimatableCharacterRange(
				CharacterRange staticValue,
				Collection<Keyframe<CharacterRange>> keyframes,
				String expression) {

			super(CharacterRange.class, staticValue, keyframes, expression);
		}

		public AnimatableCharacterRange(CharacterRange defaultValue) {
			super(CharacterRange.class, defaultValue);
		}

	}

}
