/*
 * Copyright (c) 2009-2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal;

import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.Arrays;

import javax.sound.sampled.AudioFileFormat;
import javax.sound.sampled.AudioFormat;
import javax.sound.sampled.AudioInputStream;
import javax.sound.sampled.AudioSystem;
import javax.sound.sampled.UnsupportedAudioFileException;
import javax.sound.sampled.AudioFileFormat.Type;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ch.kuramo.javie.api.AudioMode;
import ch.kuramo.javie.api.IArray;
import ch.kuramo.javie.api.IAudioBuffer;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.services.IArrayPools;
import ch.kuramo.javie.core.JavieRuntimeException;
import ch.kuramo.javie.core.MediaOptions;
import ch.kuramo.javie.core.MediaSource;
import ch.kuramo.javie.core.internal.Windows.Kernel32;
import ch.kuramo.javie.core.services.AudioRenderSupport;
import ch.kuramo.javie.core.services.RenderContext;

import com.google.inject.Inject;
import com.sun.jna.WString;

public class JavaSoundSource implements MediaSource {

	private static final Logger _logger = LoggerFactory.getLogger(JavaSoundSource.class);


	private File _file;

	private boolean _tmpfile;

	private Time _duration;

	private AudioMode _audioMode;

	private AudioInputStream _stream;

	private long _streamPosition;

	@Inject
	private RenderContext _context;

	@Inject
	private AudioRenderSupport _arSupport;

	@Inject
	private IArrayPools _arrayPools;


	public JavaSoundSource() {
		super();
	}

	@Override
	protected void finalize() throws Throwable {
		if (_file != null && _tmpfile) {
			_file.delete();
			_file = null;
			_tmpfile = false;
		}
		super.finalize();
	}

	public boolean initialize(File file) {
		String fileName = file.getName();
		int lastDot = fileName.indexOf('.');
		if (lastDot != -1) {
			String extension = fileName.substring(lastDot+1).toLowerCase();
			boolean matchExtension = false;
			for (Type type : AudioSystem.getAudioFileTypes()) {
				if (extension.equals(type.getExtension().toLowerCase())) {
					matchExtension = true;
					break;
				}
			}
			if (!matchExtension && !extension.equals("mp3")) {	// JavaZoomのMP3SPIが入っていてもAudioSystem.getAudioFileTypes()
				return false;									// でMP3は列挙されない。tritonus-mp3では列挙される。
			}
		}

		try {
			AudioFileFormat aff = AudioSystem.getAudioFileFormat(file);
			if ("MP3".equals(aff.getType().toString())) {
				_file = tmpWaveFromMP3(file, aff);
				_tmpfile = true;
			} else {
				_file = file;
			}

			AudioInputStream stream = AudioSystem.getAudioInputStream(_file);
			try {
				long numFrames = stream.getFrameLength();
				AudioFormat format = stream.getFormat();

				// getFrameLengthはバイト数ではなくフレーム数を返すはずなのに、Macではバイト数を返すバグがあるようである。
				// getFrameLengthが返した値が正しければ、それにgetFrameSizeの値を掛けてもファイルサイズを超えるはずはないが、
				// このバグが発生する場合はファイルサイズを超えることになる。以下ではそのチェックと対処をしている。
				if ((format.getEncoding().equals(AudioFormat.Encoding.PCM_SIGNED)
						|| format.getEncoding().equals(AudioFormat.Encoding.PCM_UNSIGNED))
						&& numFrames * format.getFrameSize() > _file.length()) {

					numFrames /= format.getFrameSize();
				}

				float frameRate = format.getFrameRate();
				int frameRateInt = (int) frameRate;
				if (frameRate - frameRateInt != 0) {
					_logger.warn("frame rate is not integer: " + frameRate);
				}
				_duration = new Time(numFrames, frameRateInt);
			} finally {
				stream.close();
			}
		} catch (UnsupportedAudioFileException e) {
			return false;
		} catch (IOException e) {
			return false;
		}

		return true;
	}

	private File tmpWaveFromMP3(File file, AudioFileFormat aff) throws UnsupportedAudioFileException, IOException {
		AudioInputStream s1 = null;
		AudioInputStream s2 = null;
		try {
			AudioFormat f1 = aff.getFormat();
			AudioFormat f2 = new AudioFormat(f1.getSampleRate(), 16, 2, true,
					ByteOrder.nativeOrder().equals(ByteOrder.BIG_ENDIAN));

			s1 = AudioSystem.getAudioInputStream(file);
			s2 = AudioSystem.getAudioInputStream(f2, s1);

			File tmpFile;
			String prefix = file.getName();
			int lastDot = prefix.lastIndexOf('.');
			if (lastDot != -1) {
				prefix = prefix.substring(0, lastDot);
			}
			prefix += "_tmp";
			try {
				tmpFile = File.createTempFile(prefix, ".wav", file.getParentFile());
			} catch (IOException e) {
				tmpFile = File.createTempFile(prefix, ".wav");
			}
			tmpFile.deleteOnExit();

			if (Windows.WINDOWS) {
				String fileName = "\\\\?\\" + tmpFile.getAbsolutePath();
				Windows.KERNEL32.SetFileAttributesW(new WString(fileName),
						/*Kernel32.FILE_ATTRIBUTE_HIDDEN |*/
						Kernel32.FILE_ATTRIBUTE_NOT_CONTENT_INDEXED |
						Kernel32.FILE_ATTRIBUTE_TEMPORARY);
			}

			AudioSystem.write(s2, Type.WAVE, tmpFile);
			return tmpFile;

		} finally {
			if (s2 != null) s2.close();
			if (s1 != null) s1.close();
		}
	}

	public void dispose() {
		closeStream();
	}

	private void closeStream() {
		if (_stream != null) {
			try {
				_stream.close();
			} catch (IOException e) {
				// ログだけ書き出して無視
				_logger.warn("failed to close AudioInputStream", e);
			}
			_stream = null;
		}
		_streamPosition = 0;
	}

	public synchronized IAudioBuffer getAudioChunk(Time mediaTime, MediaOptions options) {
		try {
			AudioMode audioMode = _context.getAudioMode();
			long frameNumber = mediaTime.toFrameNumber(audioMode.sampleDuration);

			if (-frameNumber >= _context.getAudioFrameCount() || !mediaTime.before(_duration)) {
				IAudioBuffer ab = _arSupport.createAudioBuffer();
				_arSupport.clear(ab);
				return ab;
			}

			if (audioMode != _audioMode || audioMode.frameSize*frameNumber != _streamPosition) {
				closeStream();
			}

			if (_stream == null) {
				_stream = createAudioInputStream();
			}

			return readAudioData(frameNumber);

		} catch (IOException e) {
			// TODO throw new MediaInputException(e);
			throw new JavieRuntimeException(e);
		} catch (UnsupportedAudioFileException e) {
			// TODO throw new MediaInputException(e);
			throw new JavieRuntimeException(e);
		}
	}

	private AudioInputStream createAudioInputStream() throws IOException, UnsupportedAudioFileException {
		// サンプルレートと他のパラメータを同時に変換しようとすると失敗するようなので、2段階に分けて変換する。
		//
		// 変換の必要が無い場合、AudioSystem.getAudioInputStream は元のストリームをそのまま返すようなので
		// 以下のように無条件に変換を試みても無駄な変換は発生しないようである。

		_audioMode = _context.getAudioMode();
		int bits = _audioMode.sampleSize * 8;
		int ch = _audioMode.channels;
		boolean bigEndian = ByteOrder.nativeOrder().equals(ByteOrder.BIG_ENDIAN);

		AudioInputStream s1 = AudioSystem.getAudioInputStream(_file);

		// サンプルレート以外を変換
		AudioFormat f2 = new AudioFormat(s1.getFormat().getSampleRate(), bits, ch, true, bigEndian);
		AudioInputStream s2 = AudioSystem.getAudioInputStream(f2, s1);

		// サンプルレートを変換
		AudioFormat f3 = new AudioFormat(_audioMode.sampleRate, bits, ch, true, bigEndian);
		return AudioSystem.getAudioInputStream(f3, s2);
	}

	private IAudioBuffer readAudioData(long frameNumber) throws IOException {
		IAudioBuffer ab = _arSupport.createAudioBuffer();
		AudioMode audioMode = ab.getAudioMode();
		Object data = ab.getData();
		int dataOffset = 0;
		int dataLength = ab.getDataLength();
		int lengthInBytes = dataLength * audioMode.sampleSize;

		if (_streamPosition == 0) {
			if (frameNumber > 0) {
				// tritonusかJavaSoundのバグと思われるが、サンプルレートの変換が必要なケースで
				// frameNumberぴったりの所へスキップしようとすると音がズレてしまう。
				// 元のサンプルレートとAudioModeのサンプルレートの最大公約数でAudioModeのサンプルレートを割り、
				// それを整数倍した所へスキップ、残りをreadして捨てるとうまくいく模様。
				// 例: 元のサンプルレート44100, AudioMode=48000の場合、最大公約数は300
				// 48000/300=160なので160の整数倍の所へスキップすると良い。
				// (しかし、以下では最大公約数は求めず、AudioModeのサンプルレートの整数倍の所へスキップしている)
				long skip = audioMode.frameSize * frameNumber;
				long skipped = _stream.skip((skip / audioMode.sampleRate) * audioMode.sampleRate);
				_streamPosition = skipped;
				skip -= skipped;
				if (skip > 0) {
					IArray<byte[]> array = _arrayPools.getByteArray((int)skip);
					try {
						do {
							skipped = _stream.read(array.getArray(), 0, (int)skip);
							if (skipped != -1) {
								_streamPosition += skipped;
								skip -= skipped;
							}
						} while (skipped != -1 && skip > 0);
					} finally {
						array.release();
					}
				}
				if (skip != 0) {
					_logger.warn(String.format("skip failed: frameNumber=%d, skip=%d", frameNumber, skip));
					closeStream();
					_arSupport.clear(ab);
					return ab;
				}
			} else if (frameNumber < 0) {
				_arSupport.clear(ab, 0, (int)-frameNumber);
				dataOffset = (int)-frameNumber * audioMode.channels;
				dataLength -= dataOffset;
				lengthInBytes -= dataOffset * audioMode.sampleSize;
			}
		}

		IArray<byte[]> array = _arrayPools.getByteArray(lengthInBytes);
		try {
			byte[] buffer = array.getArray();

			int readBytes = 0;
			do {
				int n = _stream.read(buffer, readBytes, lengthInBytes - readBytes);
				if (n == -1) {
					Arrays.fill(buffer, readBytes, lengthInBytes, (byte)0);
					break;
				}
				readBytes += n;
				_streamPosition += n;
			} while (readBytes < lengthInBytes);

			switch (audioMode.dataType) {
				case SHORT:
					ByteBuffer.wrap(buffer, 0, lengthInBytes).order(ByteOrder.nativeOrder())
							.asShortBuffer().get((short[]) data, dataOffset, dataLength);
					break;
				case INT:
					ByteBuffer.wrap(buffer, 0, lengthInBytes).order(ByteOrder.nativeOrder())
							.asIntBuffer().get((int[]) data, dataOffset, dataLength);
					break;
				case FLOAT:
					for (int i = dataOffset; i < dataLength; ++i) {
						int value =  (buffer[i*4  ] & 0xff)
								  | ((buffer[i*4+1] & 0xff) <<  8)
								  | ((buffer[i*4+2] & 0xff) << 16)
								  | ((buffer[i*4+3] & 0xff) << 24);
						((float[]) data)[i] = (float) value / Integer.MAX_VALUE;
					}
					break;
				default:
					throw new UnsupportedOperationException(
							"unsupported AudioMode.DataType: " + audioMode.dataType);
			}
		} finally {
			array.release();
		}

		return ab;
	}

	public boolean isVideoAvailable() {
		return false;
	}

	public boolean isAudioAvailable() {
		return true;
	}

	public Time getDuration(MediaOptions options) {
		return _duration;
	}

	public Time getVideoFrameDuration(MediaOptions options) {
		throw new UnsupportedOperationException("video is not available");
	}

	public VideoBounds getVideoFrameBounds(MediaOptions options) {
		throw new UnsupportedOperationException("video is not available");
	}

	public IVideoBuffer getVideoFrame(Time mediaTime, MediaOptions options) {
		throw new UnsupportedOperationException("video is not available");
	}

	public MediaOptions validateOptions(MediaOptions options) {
		if (options == null) {
			options = new MediaOptions();
		} else {
			options = options.clone();
			options.clearAvailableOptions();
		}
		return options;
	}

}
