/*
 * Copyright (c) 2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Random;

import ch.kuramo.javie.core.AbstractAnimatableEnum;
import ch.kuramo.javie.core.CoreContext;
import ch.kuramo.javie.core.Expressionee;
import ch.kuramo.javie.core.Keyframe;
import ch.kuramo.javie.core.TASelector;
import ch.kuramo.javie.core.Util;

public abstract class AbstractTASelector implements TASelector {

	private boolean enabled = true;

	private String name;

	private Base base = Base.CHARACTERS;


	public AbstractTASelector(String name) {
		this.name = name;
	}

	public boolean isEnabled() {
		return enabled;
	}

	public void setEnabled(boolean enabled) {
		this.enabled = enabled;
	}

	public String getName() {
		return name;
	}

	public void setName(String name) {
		this.name = name;
	}

	public Base getBase() {
		return base;
	}

	public void setBase(Base base) {
		this.base = base;
	}

	public class TASelectorExpressionElement {

		protected final CoreContext context;

		public TASelectorExpressionElement(CoreContext context) {
			this.context = context;
		}

		protected <T> T elem(Expressionee exprnee) {
			return context.getExpressionElement(exprnee);
		}

		public boolean isEnabled()	{ return enabled; }
		public String getName()		{ return name; }
		public String getBase()		{ return base.name(); }
	}

	public static class AnimatableMode extends AbstractAnimatableEnum<Mode> {

		public AnimatableMode(
				Mode staticValue,
				Collection<Keyframe<Mode>> keyframes,
				String expression) {

			super(Mode.class, staticValue, keyframes, expression);
		}

		public AnimatableMode(Mode defaultValue) {
			super(Mode.class, defaultValue);
		}

	}

	public abstract class AbstractEvaluator implements Evaluator {

		private final int[] totals;

		private final int[][] rndOrderTable;


		public AbstractEvaluator(int[] totals) {
			this(totals, null);
		}

		public AbstractEvaluator(int[] totals, Random rndOrder) {
			this.totals = totals;
			this.rndOrderTable = (rndOrder != null)
					? createRndOrderTable(totals, rndOrder) : null;
		}

		private int[][] createRndOrderTable(int[] totals, Random rndOrder) {
			int[][] table = new int[4][];

			List<Integer> list = Util.newList();
			for (int i = 0; i < 4; ++i) {
				list.clear();
				for (int j = 0; j < totals[i]; ++j) {
					list.add(j);
				}
				Collections.shuffle(list, rndOrder);

				table[i] = new int[totals[i]];
				for (int j = 0; j < totals[i]; ++j) {
					table[i][j] = list.get(j);
				}
			}

			return table;
		}

		public double[] evaluate(double[] combine, int[] indices) {
			return evaluate(combine, new int[][] { indices });
		}

		public double[] evaluate(double[] combine, int[][] indices) {
			int ordinal = getBase().ordinal();
			int index0 = -1;
			int index1 = -1;

			for (int i = 0; i < indices.length; ++i) {
				int index = indices[i][ordinal];
				if (index >= 0) {
					if (rndOrderTable != null) {
						index = rndOrderTable[ordinal][index];
						if (index0 < 0 || index0 > index) {
							index0 = index;
						}
						index1 = Math.max(index1, index + 1);
					} else {
						if (index0 < 0) {
							index0 = index;
						}
						index1 = index + 1;
					}
				}
			}
			if (index0 < 0) {
				return combine;
			}

			return evaluate(combine, index0, index1, totals[ordinal]);
		}

		protected abstract double[] evaluate(double[] combine, int index0, int index1, int total);


		protected double[] applyMode(Mode mode, double[] combine, double[] tvec) {
			if (combine != null) {
				for (int i = 0; i < 3; ++i) {
					switch (mode) {
						case ADD:
							tvec[i] += combine[i];
							break;
						case SUBTRACT:
							tvec[i] = combine[i] - tvec[i];
							break;
						case INTERSECT:
							tvec[i] *= combine[i];
							break;
						case MIN:
							tvec[i] = Math.min(tvec[i], combine[i]);
							break;
						case MAX:
							tvec[i] = Math.max(tvec[i], combine[i]);
							break;
						case DIFFERENCE:
							tvec[i] = Math.abs(tvec[i] - combine[i]);
							break;
						default:
							throw new RuntimeException("unknown Mode: " + mode.name());
					}
					tvec[i] = Math.min(Math.max(tvec[i], -1), 1);
				}
			} else if (mode == Mode.SUBTRACT) {
				for (int i = 0; i < 3; ++i) {
					tvec[i] = 1 - tvec[i];
				}
			}
			return tvec;
		}

	}

}
