/*
 * Copyright (c) 2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.project;

import java.util.Collection;
import java.util.List;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;

import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.TAProperty;
import ch.kuramo.javie.core.TextAnimator;
import ch.kuramo.javie.core.TextLayer;
import ch.kuramo.javie.core.Util;

public class AddRemoveTAPropertiesOperation extends ProjectOperation {

	private static class Entry {
		final String layerId;
		final int animatorIndex;
		final TAProperty property;

		private Entry(String layerId, int animatorIndex, TAProperty property) {
			this.layerId = layerId;
			this.animatorIndex = animatorIndex;
			this.property = property;
		}
	}


	private final boolean add;

	private final String compId;

	private final List<Entry> entries = Util.newList();


	public AddRemoveTAPropertiesOperation(
			ProjectManager pm, Collection<Object[]> properties, boolean add) {

		super(pm, add ? "プロパティの追加" : "プロパティの削除");
		this.add = add;

		LayerComposition comp = pm.checkLayer((TextLayer) properties.iterator().next()[0]);
		compId = comp.getId();

		for (Object[] o : properties) {
			TextLayer layer = (TextLayer) o[0];
			TextAnimator animator = (TextAnimator) o[1];
			TAProperty property = (TAProperty) o[2];

			if (pm.checkLayer(layer) != comp) {
				throw new IllegalArgumentException();
			}

			int animatorIndex = layer.getTextAnimators().indexOf(animator);
			if (animatorIndex == -1) {
				throw new IllegalArgumentException();
			}

			if (add ^ animator.getProperties().contains(property)) {
				entries.add(new Entry(layer.getId(), animatorIndex, property));
			}
		}

		_noEffect = entries.isEmpty();
	}

	@Override
	protected IStatus executeOrRedo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		addRemove(project, pm, add);
		return Status.OK_STATUS;
	}

	@Override
	protected IStatus undo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		addRemove(project, pm, !add);
		return Status.OK_STATUS;
	}

	private void addRemove(Project project, ProjectManager pm, boolean add) {
		LayerComposition comp = project.getComposition(compId);

		List<Object[]> data = null;
		if (pm != null) {
			data = Util.newList();
		}

		for (Entry entry : entries) {
			TextLayer layer = (TextLayer) comp.getLayer(entry.layerId);
			TextAnimator animator = layer.getTextAnimators().get(entry.animatorIndex);

			if (add) {
				animator.getProperties().add(entry.property);
			} else {
				animator.getProperties().remove(entry.property);
			}

			if (pm != null) {
				data.add(new Object[] { layer, entry.animatorIndex, entry.property.name() });
			}
		}

		if (pm != null) {
			if (add) {
				pm.fireTAPropertiesAdd(comp, data.toArray(new Object[data.size()][]));
			} else {
				pm.fireTAPropertiesRemove(comp, data.toArray(new Object[data.size()][]));
			}
		}
	}

}
