/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.player;

import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.ActionContributionItem;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IMenuCreator;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.MenuItem;
import org.eclipse.swt.widgets.ToolItem;

import ch.kuramo.javie.api.Resolution;
import ch.kuramo.javie.app.ImageUtil;

public class PlayerResolutionAction extends Action implements IMenuCreator {

	private static final Resolution[] RESOLUTIONS = {
		Resolution.FULL,
		Resolution.HALF,
		Resolution.ONETHIRD,
		Resolution.QUARTER
	};


	private final MediaPlayer mediaPlayer;

	private Menu menu;


	public PlayerResolutionAction(MediaPlayer mediaPlayer) {
		super("解像度");
		this.mediaPlayer = mediaPlayer;

		setImageDescriptor(ImageUtil.getPlayerResolutionIcon());
		setMenuCreator(this);
		setEnabled(mediaPlayer.hasVideo());
	}

	public void dispose() {
		disposeMenu();
	}

	private void disposeMenu() {
		if (menu != null) {
			menu.dispose();
			menu = null;
		}
	}

	public Menu getMenu(Control parent) {
		disposeMenu();

		if (!isEnabled()) {
			return null;
		}

		menu = new Menu(parent);


		addAction(menu, null);
		new MenuItem(menu, SWT.SEPARATOR);

		for (Resolution resolution : RESOLUTIONS) {
			addAction(menu, resolution);
		}

		return menu;
	}

	public Menu getMenu(Menu parent) {
		return null;
	}

	private void addAction(Menu parent, final Resolution resolution) {
		String text = (resolution == null) ? "自動" :
					  resolution.equals(Resolution.FULL) ? "フル画質" :
					  resolution.equals(Resolution.HALF) ? "1/2 画質" :
					  resolution.equals(Resolution.ONETHIRD) ? "1/3 画質" :
					  resolution.equals(Resolution.QUARTER) ? "1/4 画質" :
							String.valueOf(resolution.scale) + " 画質";

		Action action = new Action(text, IAction.AS_RADIO_BUTTON) {
			public void run() {
				mediaPlayer.setVideoResolution(resolution);
			}
		};

		if (resolution == null) {
			action.setChecked(mediaPlayer.getVideoResolution() == null);
		} else {
			action.setChecked(resolution.equals(mediaPlayer.getVideoResolution()));
		}

		ActionContributionItem item = new ActionContributionItem(action);
		item.fill(parent, -1);
	}

	public void runWithEvent(Event event) {
		if (event.widget instanceof ToolItem) {
			ToolItem toolItem = (ToolItem) event.widget;
			Control control = toolItem.getParent();
			Menu menu = getMenu(control);

			Rectangle bounds = toolItem.getBounds();
			Point topLeft = new Point(bounds.x, bounds.y + bounds.height);
			menu.setLocation(control.toDisplay(topLeft));
			menu.setVisible(true);
		}
	}

}
