/*
 * Copyright (c) 2009-2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app;

import java.lang.reflect.Array;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Collection;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;

import ch.kuramo.javie.core.JavieRuntimeException;
import ch.kuramo.javie.core.Layer;
import ch.kuramo.javie.core.TASelector;
import ch.kuramo.javie.core.TextAnimator;
import ch.kuramo.javie.core.TextLayer;
import ch.kuramo.javie.core.Util;

public class PropertyUtil {

	public static Collection<String> getPropertyNames(Object object) {
		List<String> names = Util.newList();

		Class<?> clazz = object.getClass();
		for (Method m : clazz.getMethods()) {
			String name = m.getName();
			boolean is;
			if (((is = name.startsWith("is")) || name.startsWith("get"))
					&& m.getParameterTypes().length == 0) {

				if (is && !m.getReturnType().equals(boolean.class)) {
					continue;
				}

				name = name.substring(is ? 2 : 3);
				name = Character.toLowerCase(name.charAt(0)) + name.substring(1);
				names.add(name);
			}
		}

		return names;
	}

	public static Collection<String> getLayerPropertyNames(Layer layer) {
		return (layer instanceof TextLayer) ? getTextLayerPropertyNames((TextLayer) layer)
											: getPropertyNames(layer);
	}

	public static Collection<String> getTextLayerPropertyNames(TextLayer layer) {
		List<String> names = Util.newList();

		List<TextAnimator> animators = layer.getTextAnimators();
		for (ListIterator<TextAnimator> it1 = animators.listIterator(); it1.hasNext(); ) {
			int i = it1.nextIndex();
			TextAnimator animator = it1.next();

			List<TASelector> selectors = animator.getSelectors();
			for (ListIterator<TASelector> it2 = selectors.listIterator(); it2.hasNext(); ) {
				int j = it2.nextIndex();
				TASelector selector = it2.next();

				for (String p : getPropertyNames(selector)) {
					names.add(String.format("textAnimators[%d].selectors[%d].%s", i, j, p));
				}
			}

			for (String p : getPropertyNames(animator)) {
				if (!p.equals("selectors")) {
					names.add(String.format("textAnimators[%d].%s", i, p));
				}
			}
		}

		for (String p : getPropertyNames(layer)) {
			if (!p.equals("textAnimators")) {
				names.add(p);
			}
		}

		return names;
	}

	public static <T> T getProperty(Object object, String property) {
		int index = property.indexOf('.');
		if (index != -1) {
			String p0 = property.substring(0, index);
			String p1 = property.substring(index+1);
			return getProperty(getProperty(object, p0), p1);
		}

		index = property.indexOf('[');
		if (index != -1) {
			int index2 = property.indexOf(']', index+1);
			if (index2 != -1) {
				String p = property.substring(0, index);
				int n = Integer.parseInt(property.substring(index+1, index2));
				Object listOrArray = getProperty(object, p);
				if (listOrArray instanceof List<?>) {
					@SuppressWarnings("unchecked")
					T value = ((List<T>) listOrArray).get(n);
					return value;
				} else if (listOrArray.getClass().isArray()) {
					@SuppressWarnings("unchecked")
					T value = (T) Array.get(listOrArray, n);
					return value;
				}
			}
		}

		index = property.indexOf('(');
		if (index != -1) {
			int index2 = property.indexOf(')', index+1);
			if (index2 != -1) {
				String p = property.substring(0, index);
				String key = property.substring(index+1, index2);
				@SuppressWarnings("unchecked")
				Map<String, T> map = (Map<String, T>) getProperty(object, p);
				return map.get(key);
			}
		}

		Class<?> clazz = object.getClass();
		String upperCamelCase = Character.toUpperCase(property.charAt(0)) + property.substring(1);

		Method method = null;
		try {
			method = clazz.getMethod("get" + upperCamelCase);
			if (method.getReturnType().equals(void.class)) {
				method = null;
			}
		} catch (NoSuchMethodException e) {
		}

		if (method == null) {
			try {
				method = clazz.getMethod("is" + upperCamelCase);
				if (!method.getReturnType().equals(boolean.class)) {
					method = null;
				}
			} catch (NoSuchMethodException e) {
			}
		}

		if (method == null) {
			throw new IllegalArgumentException(String.format(
					"no such property found: class=%s, property=%s",
					clazz.getName(), property));
		}

		try {
			@SuppressWarnings("unchecked")
			T value = (T) method.invoke(object);
			return value;
		} catch (IllegalAccessException e) {
			throw new JavieRuntimeException(e);
		} catch (InvocationTargetException e) {
			throw new JavieRuntimeException(e);
		}
	}

	public static <T> void setProperty(Object object, String property, T newValue) {
		int index = property.indexOf('.');
		if (index != -1) {
			String p0 = property.substring(0, index);
			String p1 = property.substring(index+1);
			setProperty(getProperty(object, p0), p1, newValue);
			return;
		}

		index = property.indexOf('[');
		if (index != -1) {
			int index2 = property.indexOf(']', index+1);
			if (index2 != -1) {
				String p = property.substring(0, index);
				int n = Integer.parseInt(property.substring(index+1, index2));
				Object listOrArray = getProperty(object, p);
				if (listOrArray instanceof List<?>) {
					@SuppressWarnings("unchecked")
					List<T> list = (List<T>) listOrArray;
					list.set(n, newValue);
					return;
				} else if (listOrArray.getClass().isArray()) {
					Array.set(listOrArray, n, newValue);
					return;
				}
			}
		}

		index = property.indexOf('(');
		if (index != -1) {
			int index2 = property.indexOf(')', index+1);
			if (index2 != -1) {
				String p = property.substring(0, index);
				String key = property.substring(index+1, index2);
				@SuppressWarnings("unchecked")
				Map<String, T> map = (Map<String, T>) getProperty(object, p);
				map.put(key, newValue);
				return;
			}
		}

		Class<?> clazz = object.getClass();
		String methodName = "set" + Character.toUpperCase(property.charAt(0)) + property.substring(1);

		Method method = null;
		for (Method m : clazz.getMethods()) {
			if (m.getName().equals(methodName) && m.getParameterTypes().length == 1
					&& m.getReturnType().equals(void.class)) {

				Class<?> paramType = toWrapperType(m.getParameterTypes()[0]);
				if (paramType.isAssignableFrom(newValue.getClass())) {
					method = m;
					break;
				}
			}
		}

		if (method == null) {
			throw new IllegalArgumentException(String.format(
					"no such property found: class=%s, property=%s",
					clazz.getName(), property));
		}

		try {
			method.invoke(object, newValue);
		} catch (IllegalAccessException e) {
			throw new JavieRuntimeException(e);
		} catch (InvocationTargetException e) {
			throw new JavieRuntimeException(e);
		}
	}

	private static Class<?> toWrapperType(Class<?> type) {
		return type.equals(Boolean.TYPE)   ? Boolean.class
			 : type.equals(Character.TYPE) ? Character.class
			 : type.equals(Byte.TYPE)      ? Byte.class
			 : type.equals(Short.TYPE)     ? Short.class
			 : type.equals(Integer.TYPE)   ? Integer.class
			 : type.equals(Long.TYPE)      ? Long.class
			 : type.equals(Float.TYPE)     ? Float.class
			 : type.equals(Double.TYPE)    ? Double.class
			 : type.equals(Void.TYPE)      ? Void.class
			 : type;
	}

}
