/*
 * Copyright (c) 2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.util.Date;
import java.util.Iterator;
import java.util.List;

import net.arnx.jsonic.JSON;

import org.eclipse.core.runtime.IPath;

import ch.kuramo.javie.app.project.ProjectManager;
import ch.kuramo.javie.core.JavieRuntimeException;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.ProjectDecodeException;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.output.Output;
import ch.kuramo.javie.core.services.ProjectDecoder;
import ch.kuramo.javie.core.services.ProjectEncoder;

public class BatchStore {

	public enum BatchState { READY, IN_PROGRESS, CANCELED, PENDING, ERROR, DONE, REMOVED }

	public static class BatchEntry {
		public String id;
		public BatchState state;
		public String projectName;
		public String compositionName;
		public String compositionItemId;
		public File file;
		public Date registeredTime;
		public Date startTime;
		public Date endTime;

		public BatchEntry() { }

		public BatchEntry(String prjName, String compName, String compItemId, File file) {
			this.id = Util.randomId();
			this.state = isReserved(file) ? BatchState.PENDING : BatchState.READY;
			this.projectName = prjName;
			this.compositionName = compName;
			this.compositionItemId = compItemId;
			this.file = file;
			this.registeredTime = new Date();
		}
	}

	public static class BatchEntries {
		public List<BatchEntry> entries;
		public BatchEntries() { }
		public BatchEntries(List<BatchEntry> entries) { this.entries = entries; }
	}

	private static File getBatchLocation() throws IOException {
		IPath stateLocation = Activator.getDefault().getStateLocation();
		IPath batchLocation = stateLocation.append("batch");

		File file = batchLocation.toFile();
		if (!file.exists()) {
			if (!file.mkdir()) {
				throw new IOException("can't make directory");
			}
		} else if (!file.isDirectory()) {
			throw new IOException("not directory");
		}

		return file;
	}

	private static void writeData(Project project, Output output, String id) throws IOException {
		ProjectEncoder encoder = InjectorHolder.getInjector().getInstance(ProjectEncoder.class);
		String data = encoder.encodeElement(new Object[] { project, output });

		OutputStreamWriter w = null;
		try {
			w = new OutputStreamWriter(new FileOutputStream(new File(getBatchLocation(), id)), "UTF-8");
			w.write(data);
		} finally {
			if (w != null) w.close();
		}
	}

	public static Object[] loadData(BatchEntry entry) {
		ProjectDecoder decoder = InjectorHolder.getInjector().getInstance(ProjectDecoder.class);
		try {
			File file = new File(getBatchLocation(), entry.id);
			BufferedReader br = new BufferedReader(new InputStreamReader(new FileInputStream(file), "UTF-8"));
			try {
				StringBuilder sb = new StringBuilder();
				String line;
				while ((line = br.readLine()) != null) {
					sb.append(line).append("\n");
				}

				Object[] data = decoder.decodeElement(sb.toString(), Object[].class);
				((Project)data[0]).afterDecode();
				return data;

			} finally {
				br.close();
			}
		} catch (ProjectDecodeException e) {
			throw new JavieRuntimeException(e);
		} catch (IOException e) {
			throw new JavieRuntimeException(e);
		}
	}

	public static void addBatch(ProjectManager pm, String compItemId, Output output) {
		try {
			File file = pm.getFile();
			String prjName = file != null ? file.getName() : ProjectManager.DEFAULT_NAME;
			String compName = pm.getProject().getItem(compItemId).getName();
			BatchEntry entry = new BatchEntry(prjName, compName, compItemId, output.getFile());

			writeData(pm.getProject(), output, entry.id);

			List<BatchEntry> entries = listEntries();
			entries.add(entry);
			saveEntries(entries);

		} catch (IOException e) {
			throw new JavieRuntimeException(e);
		}
	}

	public static boolean isReserved(File file) {
		try {
			file = file.getCanonicalFile();
		} catch (IOException e) {
			throw new JavieRuntimeException(e);
		}

		List<BatchEntry> entries = listEntries();
		for (BatchEntry entry : entries) {
			try {
				if (entry.file.getCanonicalFile().equals(file)) {
					return true;
				}
			} catch (IOException e) {
				e.printStackTrace();
			}
		}
		return false;
	}

	public static List<BatchEntry> listEntries() {
		try {
			File file = new File(getBatchLocation(), "entries");
			if (!file.exists()) {
				return Util.newList();
			}

			FileInputStream in = new FileInputStream(file);
			try {
				return JSON.decode(in, BatchEntries.class).entries;
			} finally {
				in.close();
			}
		} catch (IOException e) {
			throw new JavieRuntimeException(e);
		}
	}

	public static void saveEntries(List<BatchEntry> entries) {
		File tmp = null;
		File bak = null;
		try {
			File dir = getBatchLocation();

			tmp = File.createTempFile("entries", ".tmp", dir);
			//tmp.deleteOnExit();

			bak = File.createTempFile("entries", ".bak", dir);
			//bak.deleteOnExit();


			FileOutputStream out = new FileOutputStream(tmp);
			try {
				JSON.encode(new BatchEntries(entries), out);
			} finally {
				out.close();
			}

			File file = new File(dir, "entries");
			if (file.exists()) {
				bak.delete();
				file.renameTo(bak);
			}
			if (!tmp.renameTo(file)) {
				throw new IOException(String.format("can't rename %s to %s.",
						tmp.getName(), file.getName()));
			}

		} catch (IOException e) {
			throw new JavieRuntimeException(e);

		} finally {
			if (tmp != null) tmp.delete();
			if (bak != null) bak.delete();
		}
	}

	public static void cleanup() {
		File dir;
		try {
			dir = getBatchLocation();
		} catch (IOException e) {
			throw new JavieRuntimeException(e);
		}

		List<BatchEntry> entries = listEntries();
		List<File> files = Util.newList();

		for (Iterator<BatchEntry> it = entries.iterator(); it.hasNext(); ) {
			BatchEntry entry = it.next();
			switch (entry.state) {
				case DONE:
				case REMOVED:
					it.remove();
					files.add(new File(dir, entry.id));
					break;
				case IN_PROGRESS:
					entry.state = BatchState.ERROR;
					break;
			}
		}

		saveEntries(entries);

		for (File file : files) {
			file.delete();
		}
	}

}
