/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.api.services;

import ch.kuramo.javie.api.ColorMode;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.VideoBounds;

/**
 * ぼかしの処理を支援するサービスです。
 */
public interface IBlurSupport {

	/**
	 * ぼかしの方向を表す列挙型定数。
	 */
	public enum BlurDimensions {

		/**
		 * 水平および垂直
		 */
		BOTH,

		/**
		 * 水平
		 */
		HORIZONTAL,

		/**
		 * 垂直
		 */
		VERTICAL

	}


	/**
	 * <code>inputBounds</code> の領域にガウスぼかしを行った場合、結果の画像の位置とサイズがどうなるかを返します。
	 * 
	 * @param inputBounds ガウスぼかしを行う領域
	 * @param radius ガウスぼかしの量
	 * @param dimensions ガウスぼかしの方向
	 * @param fast 高速モードの場合は <code>true</code>、そうでない場合は <code>false</code>
	 * @return <code>inputBounds</code> の領域にガウスぼかしを行った場合の、結果の画像の位置とサイズ
	 * 
	 * @see #gaussianBlur(IVideoBuffer, double, BlurDimensions, boolean, boolean)
	 * @see #gaussianBlur(IVideoBuffer, ColorMode, double, BlurDimensions, boolean, boolean)
	 */
	VideoBounds calcGaussianBlurredBounds(
			VideoBounds inputBounds, double radius, BlurDimensions dimensions,
			boolean fast);

	/**
	 * 半径が <code>radius</code> であるガウスぼかしのカーネルを作成します。
	 * 戻り値は、サイズが <code>((int)Math.ceil(radius))*2+1</code> の <code>float</code> 配列です。
	 * 
	 * @param radius ガウスぼかしの半径
	 * @return 半径が <code>radius</code> であるガウスぼかしのカーネル
	 * 
	 * @since 0.3.14
	 */
	float[] createGaussianBlurKernel(double radius);

	/**
	 * <code>input</code> の画像にガウスぼかしを行い、結果の画像を返します。
	 * 結果の画像の色深度が <code>input</code> の色深度と同じになること以外は
	 * {@link #gaussianBlur(IVideoBuffer, ColorMode, double, BlurDimensions, boolean, boolean)}
	 * と同じです。
	 * 
	 * @param input ガウスぼかしを行う画像
	 * @param radius ガウスぼかしの量
	 * @param dimensions ガウスぼかしの方向
	 * @param repeatEdgePixels 端のピクセルを繰り返す場合は <code>true</code>、そうでない場合は <code>false</code>
	 * @param fast 高速モードの場合は <code>true</code>、そうでない場合は <code>false</code>
	 * @return <code>input</code> の画像にガウスぼかしを行った結果の画像
	 */
	IVideoBuffer gaussianBlur(
			IVideoBuffer input, double radius, BlurDimensions dimensions,
			boolean repeatEdgePixels, boolean fast);

	/**
	 * <code>input</code> の画像にガウスぼかしを行い、結果の画像を返します。
	 * 結果の画像の色深度は <code>colorMode</code> で指定した色深度となります。
	 * <code>repeatEdgePixels</code> が <code>true</code> の場合、結果の画像の位置とサイズは
	 * <code>input</code> と等しくなります。<code>repeatEdgePixels</code> が <code>false</code> の場合、
	 * 結果の画像は <code>calcGaussianBlurredBounds</code> メソッドが返す位置とサイズとなります。
	 * 
	 * @param input ガウスぼかしを行う画像
	 * @param colorMode 結果の画像の色深度
	 * @param radius ガウスぼかしの量
	 * @param dimensions ガウスぼかしの方向
	 * @param repeatEdgePixels 端のピクセルを繰り返す場合は <code>true</code>、そうでない場合は <code>false</code>
	 * @param fast 高速モードの場合は <code>true</code>、そうでない場合は <code>false</code>
	 * @return <code>input</code> の画像にガウスぼかしを行った結果の画像
	 */
	IVideoBuffer gaussianBlur(
			IVideoBuffer input, ColorMode colorMode, double radius, BlurDimensions dimensions,
			boolean repeatEdgePixels, boolean fast);

	VideoBounds calcBoxBlurredBounds(
			VideoBounds inputBounds, double radius, BlurDimensions dimensions,
			boolean fast, int boxIterations);

	/**
	 * 半径が <code>radius</code> であるボックスブラーのカーネルを作成します。
	 * 戻り値は、サイズが <code>((int)Math.ceil(radius))*2+1</code> の <code>float</code> 配列です。
	 * 
	 * @param radius ボックスブラーの半径
	 * @return 半径が <code>radius</code> であるボックスブラーのカーネル
	 * 
	 * @since 0.3.14
	 */
	float[] createBoxBlurKernel(double radius);

	IVideoBuffer boxBlur(
			IVideoBuffer input, double radius, BlurDimensions dimensions,
			boolean repeatEdgePixels, boolean fast, int boxIterations);

	IVideoBuffer boxBlur(
			IVideoBuffer input, ColorMode colorMode, double radius, BlurDimensions dimensions,
			boolean repeatEdgePixels, boolean fast, int boxIterations);

}
