/*
 * Copyright (c) 2009-2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.api.annotations;

import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

/**
 * この注釈を付けたクラスは、エフェクトクラスとして認識されます。
 */
@Target(ElementType.TYPE)
@Retention(RetentionPolicy.RUNTIME)
public @interface Effect {

	/**
	 * エフェクトのIDです。他のエフェクトと重複しない値を指定する必要があります。
	 * デフォルトでは、この注釈を付けられたクラスの完全修飾クラス名がIDとなりますが、
	 * クラス名の変更やパッケージの移動でIDが変化してしまうことを避けるために、明示的に指定することを推奨します。
	 * 
	 * @return エフェクトのID
	 */
	String id() default "";

	/**
	 * <p>エフェクトのカテゴリです。デフォルトは「カテゴリなし」です。
	 * ここで指定したカテゴリに従ってUI上の表示位置などが分類されます。</p>
	 * 
	 * <p>{@link Categories} にある定義済みのカテゴリまたは独自のカテゴリを指定することができます。
	 * 独自のカテゴリを指定するには、<code>ch.kuramo.javie.api.effectCategory</code>
	 * 拡張ポイントに拡張を追加する必要があります。</p>
	 * 
	 * @return エフェクトのカテゴリ
	 */
	String category() default "";

	/**
	 * エフェクトIDが {@link #id()} のエフェクトを、{@link #convertTo()}
	 * に指定したIDのエフェクトに変換します。
	 * 
	 * 例えば、<code>Foo</code> エフェクトの改良版として <code>Foo2</code> エフェクトを作成したため
	 * <code>Foo</code> エフェクトを <code>Foo2</code> エフェクトに自動的に変換したい、という場合に次のように使用します。
	 * 
	 * <p>
	 * <blockquote><pre>
	 *  &#64;Effect(id="Foo", convertTo="Foo2")
	 *  public class Foo {
	 *  
	 *  	// <b>シグネチャが <code>public void convert(Map&lt;String, Object&gt;)</code></b> 
	 *  	// <b>のメソッドを実装し、エフェクトのプロパティを変換します。</b> 
	 *  	// このメソッドが無い場合はエフェクトIDが付け替えられるだけとなります。
	 *  	public void convert(Map&lt;String, Object&gt; map) {
	 *  		// プロパティの変換処理
	 *  		// TODO 変換処理の詳細を書く
	 *  	}
	 *  }
	 * </pre></blockquote>
	 * この場合、<code>Effect</code> アノテーションを付けたクラスは
	 * エフェクトクラスではなくエフェクトのコンバータクラスとして動作することになります。
	 * </p>
	 * 
	 * <p>
	 * なお、旧版のエフェクトを残したまま新版のエフェクトと併存させても構いません。
	 * 変換と併存のどちらを選ぶかは利用者の利便性などを考慮して決定してください。
	 * </p>
	 * 
	 * @return 変換先のエフェクトID
	 */
	String convertTo() default "";


	/**
	 * {@link Effect#category()} に指定することができる定義済みカテゴリの定数です。
	 * カテゴリの追加は ch.kuramo.javie.api.effectCategory 拡張ポイントで行うことができます。 
	 */
	public static final class Categories {

		private Categories() { }

		/**
		 * Quartz Composer 制御
		 */
		public static final String QUARTZ_COMPOSER_CONTROLS = "ch.kuramo.javie.api.effectCategory.quartzComposerControls";

		/**
		 * エクスプレッション制御
		 */
		public static final String EXPRESSION_CONTROLS = "ch.kuramo.javie.api.effectCategory.expressionControls";

		/**
		 * オーディオ
		 */
		public static final String AUDIO = "ch.kuramo.javie.api.effectCategory.audio";

		/**
		 * キーイング
		 */
		public static final String KEYING = "ch.kuramo.javie.api.effectCategory.keying";

		/**
		 * シミュレーション
		 */
		public static final String SIMULATION = "ch.kuramo.javie.api.effectCategory.simulation";

		/**
		 * スタイライズ
		 */
		public static final String STYLIZE = "ch.kuramo.javie.api.effectCategory.stylize";

		/**
		 * チャンネル
		 */
		public static final String CHANNEL = "ch.kuramo.javie.api.effectCategory.channel";

		/**
		 * ディストーション
		 */
		public static final String DISTORT = "ch.kuramo.javie.api.effectCategory.distort";

		/**
		 * トランジション
		 */
		public static final String TRANSITION = "ch.kuramo.javie.api.effectCategory.transition";

		/**
		 * ノイズ＆グレイン
		 */
		public static final String NOISE_AND_GRAIN = "ch.kuramo.javie.api.effectCategory.noiseGrain";

		/**
		 * ブラー＆シャープ
		 */
		public static final String BLUR_AND_SHARPEN = "ch.kuramo.javie.api.effectCategory.blurAndSharpen";

		/**
		 * ユーティリティ
		 */
		public static final String UTILITY = "ch.kuramo.javie.api.effectCategory.utility";

		/**
		 * 遠近
		 */
		public static final String PERSPECTIVE = "ch.kuramo.javie.api.effectCategory.perspective";

		/**
		 * 時間
		 */
		public static final String TIME = "ch.kuramo.javie.api.effectCategory.time";

		/**
		 * 色調補正
		 */
		public static final String COLOR_CORRECTION = "ch.kuramo.javie.api.effectCategory.colorCorrection";

		/**
		 * 描画
		 */
		public static final String GENERATE = "ch.kuramo.javie.api.effectCategory.generate";

	}

}
