/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal.services;

import ch.kuramo.javie.core.services.FontManager;

import com.google.inject.Inject;
import com.google.inject.Injector;

import ftgl.FTGLfont;

public class FontManagerProxy implements FontManager {

	private static final ThreadLocal<FontManagerImpl>
			_threadLocal = new ThreadLocal<FontManagerImpl>();


	static class Lifecycle {

		private final Injector _injector;

		@Inject
		Lifecycle(Injector injector) {
			_injector = injector;
		}

		void createImpl() {
			if (_threadLocal.get() != null) {
				throw new IllegalStateException("FontManagerImpl already exists on this thread.");
			}
			FontManagerImpl impl = _injector.getInstance(FontManagerImpl.class);
			_threadLocal.set(impl);
		}

		void disposeImpl() {
			FontManagerImpl impl = _threadLocal.get();
			if (impl != null) {
				impl.dispose();
				_threadLocal.set(null);
			}
		}
	}


	private FontManagerImpl getImpl() {
		FontManagerImpl impl = _threadLocal.get();
		if (impl == null) {
			throw new IllegalStateException("No FontManagerImpl exist on this thread.");
		}
		return impl;
	}

	public FTGLfont getPolygonFont(String filename, int faceIndex, int size) {
		return getImpl().getPolygonFont(filename, faceIndex, size);
	}

	public FTGLfont getOutlineFont(String filename, int faceIndex, int size) {
		return getImpl().getOutlineFont(filename, faceIndex, size);
	}

	public FTGLfont getExtrudeFont(String filename, int faceIndex, int size) {
		return getImpl().getExtrudeFont(filename, faceIndex, size);
	}

	public FTGLfont getTextureFont(String filename, int faceIndex, int size) {
		return getImpl().getTextureFont(filename, faceIndex, size);
	}

	public FTGLfont getBufferFont(String filename, int faceIndex, int size) {
		return getImpl().getBufferFont(filename, faceIndex, size);
	}

}
