/*
 * Copyright (c) 2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.voodoo2javie;

import java.util.List;

import org.eclipse.jface.dialogs.IMessageProvider;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;

import ch.kuramo.javie.api.Size2i;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.app.project.ProjectManager;
import ch.kuramo.javie.app.widgets.FrameRateComboUtil;
import ch.kuramo.javie.app.widgets.GridBuilder;
import ch.kuramo.javie.app.wizards.CompositionWizardPage;
import ch.kuramo.javie.core.Composition;
import ch.kuramo.javie.core.CompositionItem;
import ch.kuramo.javie.core.FrameDuration;
import ch.kuramo.javie.core.Item;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.TimeCode;
import ch.kuramo.javie.core.Util;

class Voodoo2JavieWizardPage extends WizardPage {

	private final Voodoo2Javie v2j;

	private final ProjectManager pm;

	private Text scaleText;

	private Label sizeLabel;

	private Combo fpsCombo;

	private Label durationLabel;

	private Button newCompRadio;

	private Text newCompText;

	private Button addCameraRadio;

	private Combo addCameraCombo;

	private List<LayerComposition> compositions;


	Voodoo2JavieWizardPage(Voodoo2Javie v2j, ProjectManager pm) {
		super("Voodoo2JavieWizardPage", "読み込み設定", null);
		this.v2j = v2j;
		this.pm = pm;
	}

	public void createControl(Composite parent) {
		GridBuilder gb = new GridBuilder(parent, 1, false);
		((GridLayout) gb.getComposite().getLayout()).verticalSpacing = 0;

		createCameraDataGroup(gb);
		gb.size(10, 10).composite(SWT.NULL);
		createCompositionGroup(gb);

		Composite grid = gb.getComposite();
		grid.setTabList(gb.getTabList());

		setControl(grid);
		doValidate();
	}

	private void createCameraDataGroup(GridBuilder gb) {
		Group group = gb.hAlign(SWT.FILL).hGrab().group(SWT.NONE, "カメラデータ");
		FillLayout layout = new FillLayout(SWT.HORIZONTAL);
		layout.marginHeight = 5;
		group.setLayout(layout);
		GridBuilder gb2 = new GridBuilder(group, 6, false);

						gb2.hSpan(1).hAlign(SWT.RIGHT).label(SWT.NULL, "サイズ:");
						gb2.hSpan(5).hAlign(SWT.LEFT).label(SWT.NULL,
							String.format("%d x %d", v2j.getWidth(), v2j.getHeight()));

						gb2.hSpan(1).hAlign(SWT.RIGHT).label(SWT.NULL, "スケール:");
		scaleText =		gb2.hSpan(1).hAlign(SWT.FILL).width(40).text(SWT.BORDER, "100");
						gb2.hSpan(1).hAlign(SWT.LEFT).label(SWT.NULL, "%");
		sizeLabel =		gb2.hSpan(3).hAlign(SWT.FILL).hGrab().label(SWT.NULL, "");

						gb2.hSpan(6).size(10, 10).composite(SWT.NULL);

						gb2.hSpan(1).hAlign(SWT.RIGHT).label(SWT.NULL, "フレーム数:");
						gb2.hSpan(5).hAlign(SWT.LEFT).label(SWT.NULL, String.valueOf(v2j.getFrameCount()));

						gb2.hSpan(1).hAlign(SWT.RIGHT).label(SWT.NULL, "フレームレート:");
		fpsCombo =		gb2.hSpan(1).hAlign(SWT.LEFT).combo(SWT.NULL, 0,
							FrameRateComboUtil.getComboItems(), FrameRateComboUtil.toComboItem(FrameDuration.FPS_29_97));
						gb2.hSpan(4).hAlign(SWT.LEFT).label(SWT.NULL, "フレーム／秒");

						gb2.hSpan(1).hAlign(SWT.RIGHT).label(SWT.NULL, "デュレーション:");
		durationLabel =	gb2.hSpan(5).hAlign(SWT.FILL).hGrab().label(SWT.NULL, "");

		ModifyListener modifyListener = new ModifyListener() {
			public void modifyText(ModifyEvent e) { doValidate(); }
		};
		SelectionListener selectionListener = new SelectionListener() {
			public void widgetSelected(SelectionEvent e)		{ doValidate(); }
			public void widgetDefaultSelected(SelectionEvent e)	{ doValidate(); }
		};
		scaleText.addModifyListener(modifyListener);
		fpsCombo.addModifyListener(modifyListener);
		fpsCombo.addSelectionListener(selectionListener);

		Composite grid2 = gb2.getComposite();
		grid2.setTabList(gb2.getTabList());
	}

	private void createCompositionGroup(GridBuilder gb) {
		List<String> compNames = Util.newList();
		compositions = Util.newList(); 
		for (Item item : pm.getProject().getItems()) {
			if (item instanceof CompositionItem) {
				Composition comp = ((CompositionItem) item).getComposition();
				if (comp instanceof LayerComposition) {
					Size2i size = comp.getSize();
					compNames.add(String.format("%s (%d x %d)", item.getName(), size.width, size.height));
					compositions.add((LayerComposition)comp);
				}
			}
		}

		Group group = gb.hAlign(SWT.FILL).hGrab().group(SWT.NONE, "コンポジション");
		FillLayout layout = new FillLayout(SWT.HORIZONTAL);
		layout.marginHeight = 5;
		group.setLayout(layout);
		GridBuilder gb2 = new GridBuilder(group, 3, false);

		newCompRadio = gb2.hSpan(1).hAlign(SWT.LEFT).button(SWT.RADIO, "新規コンポジション:");
		newCompText = gb2.hSpan(2).hAlign(SWT.FILL).hGrab().text(SWT.BORDER, pm.getUnusedItemName("コンポ"));
		newCompRadio.setSelection(true);

		if (compNames.isEmpty()) {
			addCameraRadio = gb2.hSpan(2).hAlign(SWT.LEFT).button(SWT.RADIO, "コンポジションにカメラを追加:");
			addCameraCombo = gb2.hSpan(1).hAlign(SWT.LEFT).combo(SWT.READ_ONLY, new String[] { "" });
			addCameraRadio.setEnabled(false);
			addCameraCombo.setEnabled(false);
		} else {
			addCameraRadio = gb2.hSpan(2).hAlign(SWT.LEFT).button(SWT.RADIO, "コンポジションにカメラを追加:");
			addCameraCombo = gb2.hSpan(1).hAlign(SWT.LEFT).combo(SWT.READ_ONLY, 20,
								compNames.toArray(new String[compNames.size()]), compNames.get(0));
			addCameraCombo.setEnabled(false);
		}

		newCompRadio.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				newCompText.setEnabled(true);
				addCameraCombo.setEnabled(false);
				doValidate();
			}
		});
		addCameraRadio.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				newCompText.setEnabled(false);
				addCameraCombo.setEnabled(true);
				doValidate();
			}
		});

		newCompText.addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent e) { doValidate(); }
		});
		addCameraCombo.addSelectionListener(new SelectionListener() {
			public void widgetSelected(SelectionEvent e)		{ doValidate(); }
			public void widgetDefaultSelected(SelectionEvent e)	{ doValidate(); }
		});

		Composite grid2 = gb2.getComposite();
		grid2.setTabList(gb2.getTabList());
	}

	private void doValidate() {
		setPageComplete(false);

		double scale;
		try {
			scale = Double.parseDouble(scaleText.getText().trim());
		} catch (NumberFormatException e) {
			scale = 0;
		}

		String fpsStr = fpsCombo.getText().trim();
		double fps;
		try {
			fps = Double.parseDouble(fpsStr);
		} catch (NumberFormatException e) {
			fps = 0;
		}

		int width = (int)Math.round(v2j.getWidth() * scale * 0.01);
		int height = (int)Math.round(v2j.getHeight() * scale * 0.01);
		if (scale > 0) {
			sizeLabel.setText(String.format("= %d x %d", width, height));
		} else {
			sizeLabel.setText("");
		}

		if (fps >= 1) {
			Time frameDuration = FrameRateComboUtil.toFrameDuration(fpsStr);
			String timeCode = TimeCode.toTimeCode(
					Time.fromFrameNumber(v2j.getFrameCount(), frameDuration), frameDuration);
			int base = (int)Math.round((double)frameDuration.timeScale / frameDuration.timeValue);
			boolean drop = (timeCode.indexOf(';') != -1);
			durationLabel.setText(String.format("%s (ベース %d%s)", timeCode, base, drop ? "ドロップ" : ""));
		} else {
			durationLabel.setText("");
		}


		String warnMsg = null;
		String errMsg = null;

		if (newCompRadio.getSelection()) {
			if (errMsg == null && (width < 1 || width > CompositionWizardPage.MAX_SIZE
								|| height < 1 || height > CompositionWizardPage.MAX_SIZE)) {
				errMsg = String.format(
						"サイズが範囲外です。幅および高さが1から%dとなるようにスケールの値を入力してください。",
						CompositionWizardPage.MAX_SIZE);
			}
		} else {
			LayerComposition comp = compositions.get(addCameraCombo.getSelectionIndex());
			Size2i size = comp.getSize();
			if (warnMsg == null && (width != size.width || height != size.height)) {
				warnMsg = "サイズがコンポジションと一致していません。";
			}
		}

		if (errMsg == null && fpsStr.length() == 0) {
			errMsg = "フレームレートが入力されていません。";
		}
		// TODO フレームレートの上限はどうする？
		if (errMsg == null && fps < 1) {
			errMsg = "フレームレートには1以上の数値を入力してください。";
		}

		if (errMsg == null && newCompRadio.getSelection() && newCompText.getText().trim().length() == 0) {
			errMsg = "コンポジション名が入力されていません。";
		}

		setMessage(warnMsg, IMessageProvider.WARNING);
		setErrorMessage(errMsg);
		setPageComplete(errMsg == null);
	}

	double getScale() {
		if (isPageComplete()) {
			return Double.parseDouble(scaleText.getText().trim()) * 0.01;
		} else {
			throw new IllegalStateException();
		}
	}

	Time getFrameDuration() {
		if (isPageComplete()) {
			return FrameRateComboUtil.toFrameDuration(fpsCombo.getText().trim());
		} else {
			throw new IllegalStateException();
		}
	}

	String getNewCompositionName() {
		if (isPageComplete()) {
			return newCompRadio.getSelection() ? newCompText.getText().trim() : null;
		} else {
			throw new IllegalStateException();
		}
	}

	LayerComposition getComposition() {
		if (isPageComplete()) {
			return newCompRadio.getSelection() ? null : compositions.get(addCameraCombo.getSelectionIndex());
		} else {
			throw new IllegalStateException();
		}
	}

}
