/*
 * Copyright (c) 2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.mmd2javie;

import java.util.List;

import org.eclipse.jface.dialogs.IMessageProvider;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;

import ch.kuramo.javie.api.Size2i;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.api.Vec3d;
import ch.kuramo.javie.app.project.ProjectManager;
import ch.kuramo.javie.app.widgets.FrameRateComboUtil;
import ch.kuramo.javie.app.widgets.GridBuilder;
import ch.kuramo.javie.app.wizards.CompositionWizardPage;
import ch.kuramo.javie.core.Composition;
import ch.kuramo.javie.core.CompositionItem;
import ch.kuramo.javie.core.FrameDuration;
import ch.kuramo.javie.core.Item;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.TimeCode;
import ch.kuramo.javie.core.Util;

class MMD2Javie2WizardPage extends WizardPage {

	private final MMD2Javie2 m2j2;

	private final ProjectManager pm;

	private Combo fpsCombo;

	private Label durationLabel;

	private Text scaleText;

	private Combo originCombo;

	private Text xoffText;

	private Text yoffText;

	private Text zoffText;

	private Button newCompRadio;

	private Text newCompText;

	private Button addCameraRadio;

	private Combo addCameraCombo;

	private Text widthText;

	private Text heightText;

	private Button freezeRatioCheck;

	private List<LayerComposition> compositions;

	private int widthRatio;

	private int heightRatio;


	private final ModifyListener modifyListener = new ModifyListener() {
		public void modifyText(ModifyEvent e) { doValidate((Control)e.getSource()); }
	};

	private final SelectionListener selectionListener = new SelectionListener() {
		public void widgetSelected(SelectionEvent e)		{ doValidate((Control)e.getSource()); }
		public void widgetDefaultSelected(SelectionEvent e)	{ doValidate((Control)e.getSource()); }
	};


	MMD2Javie2WizardPage(MMD2Javie2 m2j2, ProjectManager pm) {
		super("MMD2Javie2WizardPage", "読み込み設定", null);
		this.m2j2 = m2j2;
		this.pm = pm;
	}

	public void createControl(Composite parent) {
		GridBuilder gb = new GridBuilder(parent, 1, false);
		((GridLayout) gb.getComposite().getLayout()).verticalSpacing = 0;

		createCameraDataGroup(gb);
		gb.size(10, 10).composite(SWT.NULL);
		createCompositionGroup(gb);

		Composite grid = gb.getComposite();
		grid.setTabList(gb.getTabList());

		setControl(grid);
		doValidate(null);
	}

	private void createCameraDataGroup(GridBuilder gb) {
		String[] originComboItems = new String[] { "原点", "コンポジションの中心", "カスタム" };

		Group group = gb.hAlign(SWT.FILL).hGrab().group(SWT.NONE, "カメラデータ");
		FillLayout layout = new FillLayout(SWT.HORIZONTAL);
		layout.marginHeight = 5;
		group.setLayout(layout);
		GridBuilder gb2 = new GridBuilder(group, 9, false);

						gb2.hSpan(1).hAlign(SWT.RIGHT).label(SWT.NULL, "フレーム数:");
						gb2.hSpan(8).hAlign(SWT.LEFT).label(SWT.NULL, String.valueOf(m2j2.getFrameCount()));

						gb2.hSpan(1).hAlign(SWT.RIGHT).label(SWT.NULL, "フレームレート:");
		fpsCombo =		gb2.hSpan(2).hAlign(SWT.FILL).combo(SWT.NULL, 0,
							FrameRateComboUtil.getComboItems(), FrameRateComboUtil.toComboItem(FrameDuration.FPS_30));
						gb2.hSpan(6).hAlign(SWT.LEFT).label(SWT.NULL, "フレーム／秒");

						gb2.hSpan(1).hAlign(SWT.RIGHT).label(SWT.NULL, "デュレーション:");
		durationLabel =	gb2.hSpan(8).hAlign(SWT.FILL).hGrab().label(SWT.NULL, "");

		gb2.hSpan(9).size(10, 10).composite(SWT.NULL);

						gb2.hSpan(1).hAlign(SWT.RIGHT).label(SWT.NULL, "スケール:");
		scaleText =		gb2.hSpan(2).hAlign(SWT.FILL).width(70).text(SWT.BORDER, "2000");
						gb2.hSpan(6).hAlign(SWT.LEFT).label(SWT.NULL, "%");

						gb2.hSpan(1).hAlign(SWT.RIGHT).label(SWT.NULL, "位置の基準:");
		originCombo =	gb2.hSpan(8).hAlign(SWT.LEFT).combo(SWT.READ_ONLY, originComboItems, originComboItems[1]);

						gb2.hSpan(1).size(10, 10).composite(SWT.NULL);
						gb2.hSpan(1).hAlign(SWT.LEFT).label(SWT.NULL, "x =");
		xoffText =		gb2.hSpan(2).hAlign(SWT.FILL).width(70).text(SWT.BORDER, "");
						gb2.hSpan(1).hAlign(SWT.LEFT).label(SWT.NULL, "  y =");
		yoffText =		gb2.hSpan(1).hAlign(SWT.FILL).width(70).text(SWT.BORDER, "");
						gb2.hSpan(1).hAlign(SWT.LEFT).label(SWT.NULL, "  z =");
		zoffText =		gb2.hSpan(1).hAlign(SWT.FILL).width(70).text(SWT.BORDER, "");
						gb2.hSpan(1).hAlign(SWT.LEFT).label(SWT.NULL, "");

		fpsCombo.addModifyListener(modifyListener);
		fpsCombo.addSelectionListener(selectionListener);
		scaleText.addModifyListener(modifyListener);
		originCombo.addSelectionListener(selectionListener);
		xoffText.addModifyListener(modifyListener);
		yoffText.addModifyListener(modifyListener);
		zoffText.addModifyListener(modifyListener);

		Composite grid2 = gb2.getComposite();
		grid2.setTabList(gb2.getTabList());
	}

	private void createCompositionGroup(GridBuilder gb) {
		List<String> compNames = Util.newList();
		compositions = Util.newList(); 
		for (Item item : pm.getProject().getItems()) {
			if (item instanceof CompositionItem) {
				Composition comp = ((CompositionItem) item).getComposition();
				if (comp instanceof LayerComposition) {
					compNames.add(item.getName());
					compositions.add((LayerComposition)comp);
				}
			}
		}

		Group group = gb.hAlign(SWT.FILL).hGrab().group(SWT.NONE, "コンポジション");
		FillLayout layout = new FillLayout(SWT.HORIZONTAL);
		layout.marginHeight = 5;
		group.setLayout(layout);
		GridBuilder gb2 = new GridBuilder(group, 4, false);

		newCompRadio = gb2.hSpan(1).hAlign(SWT.LEFT).button(SWT.RADIO, "新規コンポジション:");
		newCompText = gb2.hSpan(3).hAlign(SWT.FILL).hGrab().text(SWT.BORDER, pm.getUnusedItemName("コンポ"));
		newCompRadio.setSelection(true);

		if (compNames.isEmpty()) {
			addCameraRadio = gb2.hSpan(2).hAlign(SWT.LEFT).button(SWT.RADIO, "コンポジションにカメラを追加:");
			addCameraCombo = gb2.hSpan(2).hAlign(SWT.LEFT).combo(SWT.READ_ONLY, new String[] { "" });
			addCameraRadio.setEnabled(false);
			addCameraCombo.setEnabled(false);
		} else {
			addCameraRadio = gb2.hSpan(2).hAlign(SWT.LEFT).button(SWT.RADIO, "コンポジションにカメラを追加:");
			addCameraCombo = gb2.hSpan(2).hAlign(SWT.LEFT).combo(SWT.READ_ONLY, 20,
								compNames.toArray(new String[compNames.size()]), compNames.get(0));
			addCameraCombo.setEnabled(false);
		}

		gb2.hSpan(4).size(10, 10).composite(SWT.NULL);

							gb2.hSpan(1).hAlign(SWT.RIGHT).label(SWT.NULL, "幅:");
		widthText =			gb2.hSpan(2).hAlign(SWT.FILL).width(70).text(SWT.BORDER, "512");
		freezeRatioCheck =	gb2.span(1, 2).hAlign(SWT.LEFT).hGrab().button(SWT.CHECK, "縦横比を固定");

							gb2.hSpan(1).hAlign(SWT.RIGHT).label(SWT.NULL, "高さ:");
		heightText =		gb2.hSpan(2).hAlign(SWT.FILL).width(70).tabAfter(widthText).text(SWT.BORDER, "384");

		newCompRadio.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				newCompText.setEnabled(true);
				addCameraCombo.setEnabled(false);
				widthText.setEnabled(true);
				heightText.setEnabled(true);
				freezeRatioCheck.setEnabled(true);
				doValidate((Control)e.getSource());
			}
		});
		addCameraRadio.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				newCompText.setEnabled(false);
				addCameraCombo.setEnabled(true);
				widthText.setEnabled(false);
				heightText.setEnabled(false);
				freezeRatioCheck.setEnabled(false);
				doValidate((Control)e.getSource());
			}
		});

		newCompText.addModifyListener(modifyListener);
		addCameraCombo.addSelectionListener(selectionListener);
		widthText.addModifyListener(modifyListener);
		heightText.addModifyListener(modifyListener);
		freezeRatioCheck.addSelectionListener(selectionListener);

		Composite grid2 = gb2.getComposite();
		grid2.setTabList(gb2.getTabList());
	}

	private void doValidate(Control control) {
		setPageComplete(false);

		String fpsStr = fpsCombo.getText().trim();
		double fps;
		try {
			fps = Double.parseDouble(fpsStr);
		} catch (NumberFormatException e) {
			fps = 0;
		}

		if (fps >= 1) {
			Time frameDuration = FrameRateComboUtil.toFrameDuration(fpsStr);
			String timeCode = TimeCode.toTimeCode(
					Time.fromFrameNumber(m2j2.getFrameCount(), frameDuration), frameDuration);
			int base = (int)Math.round((double)frameDuration.timeScale / frameDuration.timeValue);
			boolean drop = (timeCode.indexOf(';') != -1);
			durationLabel.setText(String.format("%s (ベース %d%s)", timeCode, base, drop ? "ドロップ" : ""));
		} else {
			durationLabel.setText("");
		}

		String scaleStr = scaleText.getText().trim();
		double scale;
		try {
			scale = Double.parseDouble(scaleStr);
		} catch (NumberFormatException e) {
			scale = 0;
		}

		String wStr = null, hStr = null;
		int width = 0, height = 0;
		if (newCompRadio.getSelection()) {
			wStr = widthText.getText().trim();
			try {
				width = Integer.parseInt(wStr);
			} catch (NumberFormatException e) {
				width = 0;
			}

			hStr = heightText.getText().trim();
			try {
				height = Integer.parseInt(hStr);
			} catch (NumberFormatException e) {
				height = 0;
			}

			boolean frozen = freezeRatioCheck.getSelection();
			if (frozen) {
				if (control == widthText && width > 0) {
					height = Math.max(1, width * heightRatio / widthRatio);
					heightText.removeModifyListener(modifyListener);
					heightText.setText(String.valueOf(height));
					heightText.addModifyListener(modifyListener);
				} else if (control == heightText && height > 0) {
					width = Math.max(1, height * widthRatio / heightRatio);
					widthText.removeModifyListener(modifyListener);
					widthText.setText(String.valueOf(width));
					widthText.addModifyListener(modifyListener);
				}
			} else if ((control == null || control == widthText || control == heightText) && width > 0 && height > 0) {
				int gcd = gcd(width, height);
				widthRatio = width / gcd;
				heightRatio = height / gcd;
				freezeRatioCheck.setText(String.format("縦横比を%d:%dに固定", widthRatio, heightRatio));
				freezeRatioCheck.getParent().layout(new Control[] { freezeRatioCheck });
			}
		} else {
			LayerComposition comp = compositions.get(addCameraCombo.getSelectionIndex());
			Size2i size = comp.getSize();
			width = size.width;
			height = size.height;

			widthText.removeModifyListener(modifyListener);
			widthText.setText(String.valueOf(width));
			widthText.addModifyListener(modifyListener);

			heightText.removeModifyListener(modifyListener);
			heightText.setText(String.valueOf(height));
			heightText.addModifyListener(modifyListener);

			int gcd = gcd(width, height);
			widthRatio = width / gcd;
			heightRatio = height / gcd;
			freezeRatioCheck.setText(String.format("縦横比を%d:%dに固定", widthRatio, heightRatio));
			freezeRatioCheck.getParent().layout(new Control[] { freezeRatioCheck });
		}

		xoffText.removeModifyListener(modifyListener);
		yoffText.removeModifyListener(modifyListener);
		zoffText.removeModifyListener(modifyListener);
		switch (originCombo.getSelectionIndex()) {
			case 0:
				xoffText.setEnabled(false); xoffText.setText("0");
				yoffText.setEnabled(false); yoffText.setText("0");
				zoffText.setEnabled(false); zoffText.setText("0");
				break;
			case 1:
				xoffText.setEnabled(false); xoffText.setText(width > 0 ? String.valueOf(width * 0.5) : "");
				yoffText.setEnabled(false); yoffText.setText(height > 0 ? String.valueOf(height * 0.5) : "");
				zoffText.setEnabled(false); zoffText.setText("0");
				break;
			case 2:
				boolean focus = !xoffText.isEnabled();
				xoffText.setEnabled(true);
				yoffText.setEnabled(true);
				zoffText.setEnabled(true);
				if (focus) xoffText.setFocus();
				break;
		}
		xoffText.addModifyListener(modifyListener);
		yoffText.addModifyListener(modifyListener);
		zoffText.addModifyListener(modifyListener);


		String warnMsg = null;
		String errMsg = null;

		if (errMsg == null && fpsStr.length() == 0) {
			errMsg = "フレームレートが入力されていません。";
		}
		// TODO フレームレートの上限はどうする？
		if (errMsg == null && fps < 1) {
			errMsg = "フレームレートには1以上の数値を入力してください。";
		}

		if (errMsg == null && scaleStr.length() == 0) {
			errMsg = "スケールが入力されていません。";
		}
		if (errMsg == null && scale <= 0) {
			errMsg = "スケールには0より大きい数値を入力してください。";
		}

		if (newCompRadio.getSelection()) {
			if (errMsg == null && newCompText.getText().trim().length() == 0) {
				errMsg = "コンポジション名が入力されていません。";
			}

			if (errMsg == null && wStr.length() == 0) {
				errMsg = "幅が入力されていません。";
			}
			if (errMsg == null && (width < 1 || width > CompositionWizardPage.MAX_SIZE)) {
				errMsg = String.format("幅には1から%dまでの整数を入力してください。", CompositionWizardPage.MAX_SIZE);
			}
			if (errMsg == null && hStr.length() == 0) {
				errMsg = "高さが入力されていません。";
			}
			if (errMsg == null && (height < 1 || height > CompositionWizardPage.MAX_SIZE)) {
				errMsg = String.format("高さには1から%dまでの整数を入力してください。", CompositionWizardPage.MAX_SIZE);
			}
		}

		if (originCombo.getSelectionIndex() == 2) {
			String xStr = xoffText.getText().trim();
			String yStr = yoffText.getText().trim();
			String zStr = zoffText.getText().trim();
			double x = Double.NaN, y = Double.NaN, z = Double.NaN;
			try {
				x = Double.parseDouble(xStr);
				y = Double.parseDouble(yStr);
				z = Double.parseDouble(zStr);
			} catch (NumberFormatException e) {
			}
			if (errMsg == null && (xStr.length() == 0 || yStr.length() == 0 || zStr.length() == 0)) {
				errMsg = "位置の基準が入力されていません。";
			}
			if (errMsg == null && (Double.isNaN(x) || Double.isNaN(y) || Double.isNaN(z))) {
				errMsg = "位置の基準には数値を入力してください。";
			}
		}

		setMessage(warnMsg, IMessageProvider.WARNING);
		setErrorMessage(errMsg);
		setPageComplete(errMsg == null);
	}

	private int gcd(int x, int y) {
		while (y != 0) {
			int t = x % y;
			x = y;
			y = t;
		}
		return x;
	}

	Time getFrameDuration() {
		if (isPageComplete()) {
			return FrameRateComboUtil.toFrameDuration(fpsCombo.getText().trim());
		} else {
			throw new IllegalStateException();
		}
	}

	double getScale() {
		if (isPageComplete()) {
			return Double.parseDouble(scaleText.getText().trim()) * 0.01;
		} else {
			throw new IllegalStateException();
		}
	}

	Vec3d getOffset() {
		if (isPageComplete()) {
			double x = Double.parseDouble(xoffText.getText().trim());
			double y = Double.parseDouble(yoffText.getText().trim());
			double z = Double.parseDouble(zoffText.getText().trim());
			return new Vec3d(x, y, z);
		} else {
			throw new IllegalStateException();
		}
	}

	String getNewCompositionName() {
		if (isPageComplete()) {
			return newCompRadio.getSelection() ? newCompText.getText().trim() : null;
		} else {
			throw new IllegalStateException();
		}
	}

	LayerComposition getComposition() {
		if (isPageComplete()) {
			return newCompRadio.getSelection() ? null : compositions.get(addCameraCombo.getSelectionIndex());
		} else {
			throw new IllegalStateException();
		}
	}

	Size2i getCompositionSize() {
		if (isPageComplete()) {
			if (newCompRadio.getSelection()) {
				return new Size2i(Integer.parseInt(widthText.getText().trim()),
								Integer.parseInt(heightText.getText().trim()));
			} else {
				return getComposition().getSize();
			}
		} else {
			throw new IllegalStateException();
		}
	}

}
