/*
 * Copyright (c) 2009 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.project;

import java.io.File;
import java.lang.reflect.InvocationTargetException;
import java.util.Collection;
import java.util.Collections;
import java.util.Map;
import java.util.Set;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.dialogs.ProgressMonitorDialog;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.swt.SWT;

import ch.kuramo.javie.app.InjectorHolder;
import ch.kuramo.javie.core.FileItem;
import ch.kuramo.javie.core.Folder;
import ch.kuramo.javie.core.MediaItem;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.ProjectDecodeException;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.services.ProjectDecoder;
import ch.kuramo.javie.core.services.ProjectElementFactory;
import ch.kuramo.javie.core.services.ProjectEncoder;

import com.google.inject.Inject;

public class NewFileItemsOperation extends ProjectOperation {

	private final String _parentId;

	private Set<File> _files;

	private Map<String, String> _data;

	@Inject
	private ProjectElementFactory _elementFactory;

	@Inject
	private ProjectEncoder _encoder;

	@Inject
	private ProjectDecoder _decoder;


	public NewFileItemsOperation(ProjectManager projectManager, Folder parent, Collection<File> files) {
		super(projectManager, "ファイルの読み込み");
		projectManager.checkItem(parent);

		InjectorHolder.getInjector().injectMembers(this);

		_parentId = (parent != null) ? parent.getId() : null;
		_files = Util.newSet(files);
	}

	@Override
	protected IStatus executeOrRedo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		Set<FileItem> fileItems;

		if (_data == null) {
			Folder parent = project.getItem(_parentId);

			fileItems = newFileItems();
			if (fileItems == null || fileItems.size() == 0) {
				return Status.CANCEL_STATUS;
			}

			_data = Util.newLinkedHashMap();

			for (FileItem item : fileItems) {
				item.setParent(parent);
				_data.put(item.getId(), _encoder.encodeElement(item));
			}

			// _data 作成後は不要。
			_files = null;

		} else {
			fileItems = decodeFileItems(project, (pm == null));
			if (fileItems == null) {
				return Status.CANCEL_STATUS;
			}
		}

		project.getItems().addAll(fileItems);

		if (pm != null) {
			fireItemsAdd(fileItems, project, pm);
		}

		return Status.OK_STATUS;
	}

	@Override
	protected IStatus undo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		Set<FileItem> fileItems = Util.newSet();

		for (String id : _data.keySet()) {
			FileItem item = project.getItem(id);
			project.getItems().remove(item);
			fileItems.add(item);
		}

		if (pm != null) {
			fireItemsRemove(fileItems, project, pm);
		}

		disposeAll(fileItems);
		return Status.OK_STATUS;
	}

	private Set<FileItem> newFileItems() throws ExecutionException {
		final Set<FileItem> fileItems = Util.newLinkedHashSet();
		final Set<File> errorFiles = Util.newSet();

		IRunnableWithProgress runnable = new IRunnableWithProgress() {
			public void run(IProgressMonitor monitor) throws InvocationTargetException, InterruptedException {
				monitor.beginTask("ファイルを読み込み中...", _files.size());
				for (File file : _files) {
					monitor.subTask(file.getName());
					FileItem item = _elementFactory.newFileItem(file);
					if (item != null) {
						fileItems.add(item);
					} else {
						errorFiles.add(file);
					}
					if (monitor.isCanceled()) {
						throw new InterruptedException("canceled");
					}
					monitor.worked(1);
				}
				monitor.done();
			}
		};

		try {
			ProgressMonitorDialog dialog = new ProgressMonitorDialog(getShell()) {
				protected int getShellStyle() {
					return super.getShellStyle() | SWT.SHEET;
				}
			};
			dialog.run(true, true, runnable);
		} catch (InvocationTargetException e) {
			disposeAll(fileItems);
			throw new ExecutionException("error reading files", e.getCause());
		} catch (InterruptedException e) {
			disposeAll(fileItems);
			return null;
		}

		if (errorFiles.size() > 0) {
			// いくつかのファイルが、対応する入力プラグインが存在せず読み込めなかった。
			// TODO メッセージダイアログの表示
			for (File file : errorFiles) {
				System.err.println("error reading file: " + file.getName());
			}
		}

		return fileItems;
	}

	private Set<FileItem> decodeFileItems(final Project project, boolean shadow) throws ExecutionException {
		final Set<FileItem> fileItems = Util.newLinkedHashSet();
		final Set<File> errorFiles = Util.newSet();

		IRunnableWithProgress runnable = new IRunnableWithProgress() {
			public void run(IProgressMonitor monitor) throws InvocationTargetException, InterruptedException {
				try {
					monitor.beginTask("ファイルの読み込み...", _data.size());
					for (String data : _data.values()) {
						FileItem item = _decoder.decodeElement(data, FileItem.class);
						monitor.subTask(item.getFile().getName());

						fileItems.add(item);
						item.afterDecode(project);
						if (item instanceof MediaItem && ((MediaItem) item).getMediaInput() == null) {
							errorFiles.add(item.getFile());
						}

						if (monitor.isCanceled()) {
							throw new InterruptedException("canceled");
						}
						monitor.worked(1);
					}
					monitor.done();
				} catch (ProjectDecodeException e) {
					throw new InvocationTargetException(e);
				}
			}
		};

		try {
			if (shadow) {
				runnable.run(new NullProgressMonitor());
			} else {
				ProgressMonitorDialog dialog = new ProgressMonitorDialog(getShell()) {
					protected int getShellStyle() {
						return super.getShellStyle() | SWT.SHEET;
					}
				};
				dialog.run(true, true, runnable);
			}
		} catch (InvocationTargetException e) {
			disposeAll(fileItems);
			throw new ExecutionException("error decoding FileItem data", e.getCause());
		} catch (InterruptedException e) {
			disposeAll(fileItems);
			return null;
		}

		if (errorFiles.size() > 0) {
			// いくつかのファイルが読み込めなかった場合。
			// execute時には読み込めていたのだから、ファイルを移動したり削除したりしてしまった場合か？
			// TODO 無視して継続するかキャンセルするかを問うダイアログ
			//      (ファイルの置き換え機能があれば、継続したのちファイルを置き換えることができる)
			for (File file : errorFiles) {
				System.err.println("error reading file: " + file.getName());
			}
		}

		return fileItems;
	}

	private void fireItemsAdd(Set<FileItem> fileItems, Project project, ProjectManager pm) {
		if (_parentId != null) {
			Folder parent = project.getItem(_parentId);
			pm.fireItemsAdd(fileItems, Collections.singleton(parent));
		} else {
			pm.fireItemsAdd(fileItems, null);
		}
	}

	private void fireItemsRemove(Set<FileItem> fileItems, Project project, ProjectManager pm) {
		if (_parentId != null) {
			Folder parent = project.getItem(_parentId);
			pm.fireItemsRemove(fileItems, Collections.singleton(parent));
		} else {
			pm.fireItemsRemove(fileItems, null);
		}
	}

	private void disposeAll(Collection<FileItem> fileItems) {
		for (FileItem item : fileItems) {
			item.dispose();
		}
	}

}
