/*
 * Copyright (c) 2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.distort;

import java.nio.FloatBuffer;
import java.util.HashSet;
import java.util.Set;

import javax.media.opengl.GLUniformData;

import ch.kuramo.javie.api.IAnimatableBoolean;
import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IAnimatableVec2d;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Quality;
import ch.kuramo.javie.api.Resolution;
import ch.kuramo.javie.api.Vec2d;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.IVideoBuffer.TextureFilter;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.annotations.ShaderSource;
import ch.kuramo.javie.api.annotations.Effect.Categories;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.api.services.IVideoRenderSupport;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.Twirl", category=Categories.DISTORT)
public class Twirl {

	@Property
	private IAnimatableDouble angle;

	@Property(value="30", min="0")
	private IAnimatableDouble twirlRadius;

	@Property
	private IAnimatableVec2d twirlCenter;

	@Property(value="0", min="0")
	private IAnimatableDouble ease;

	@Property("true")
	private IAnimatableBoolean interpolation;


	private final IVideoEffectContext context;

	private final IVideoRenderSupport support;

	private final IShaderProgram program;

	@Inject
	public Twirl(IVideoEffectContext context,
			IVideoRenderSupport support, IShaderRegistry shaders) {

		this.context = context;
		this.support = support;
		program = shaders.getProgram(Twirl.class, "TWIRL");
	}

	public IVideoBuffer doVideoEffect() {
		IVideoBuffer source = context.doPreviousEffect();
		VideoBounds bounds = source.getBounds();
		if (bounds.isEmpty()) {
			return source;
		}

		double angle = Math.toRadians(context.value(this.angle));
		double radius = context.value(this.twirlRadius) / 100 * Math.max(bounds.width, bounds.height);
		if (angle == 0 || radius == 0) {
			return source;
		}

		Resolution resolution = context.getVideoResolution();
		Vec2d center = resolution.scale(context.value(this.twirlCenter));
		double ease = context.value(this.ease) + 1;
		boolean interpolation = context.value(this.interpolation);

		TextureFilter filter = (context.getQuality() == Quality.DRAFT || resolution.scale < 1)
					? TextureFilter.NEAREST : interpolation ? TextureFilter.MIPMAP : TextureFilter.LINEAR;

		try {
			Set<GLUniformData> uniforms = new HashSet<GLUniformData>();
			uniforms.add(new GLUniformData("texture", 0));
			uniforms.add(new GLUniformData("size", 2, toFloatBuffer(bounds.width, bounds.height)));
			uniforms.add(new GLUniformData("center", 2, toFloatBuffer(center.x-bounds.x, center.y-bounds.y)));
			uniforms.add(new GLUniformData("radius", (float)radius));
			uniforms.add(new GLUniformData("angle", (float)angle));
			uniforms.add(new GLUniformData("ease", (float)ease));

			source.setTextureFilter(filter);

			return support.useShaderProgram(program, uniforms, null, source);

		} finally {
			source.dispose();
		}
	}

	private FloatBuffer toFloatBuffer(double...values) {
		float[] farray = new float[values.length];
		for (int i = 0; i < values.length; ++i) {
			farray[i] = (float)values[i];
		}
		return FloatBuffer.wrap(farray);
	}


	@ShaderSource
	public static final String[] TWIRL = {
		"uniform sampler2D texture;",
		"uniform vec2 size;",
		"uniform vec2 center;",
		"uniform float radius;",
		"uniform float angle;",
		"uniform float ease;",
		"",
		"void main(void)",
		"{",
		"	vec2 xy = gl_FragCoord.xy;",
		"	float d = distance(xy, center);",
		"	float a = angle * pow(max(0.0, radius-d)/radius, ease);",
		"	float cos = cos(a);",
		"	float sin = sin(a);",
		"	vec2 tc = (mat2(cos,-sin,sin,cos)*(xy-center)+center) / size;",
		"",
		"	gl_FragColor = texture2D(texture, tc);",
		"}"
	};

}
