/*
 * Copyright (c) 2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.distort;

import java.nio.FloatBuffer;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.media.opengl.GL2;
import javax.media.opengl.GLUniformData;
import javax.vecmath.Point2d;
import javax.vecmath.Vector3d;

import ch.kuramo.javie.api.IAnimatableBoolean;
import ch.kuramo.javie.api.IAnimatableVec2d;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Quality;
import ch.kuramo.javie.api.Resolution;
import ch.kuramo.javie.api.Vec2d;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.IVideoBuffer.TextureFilter;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.annotations.ShaderSource;
import ch.kuramo.javie.api.annotations.Effect.Categories;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.api.services.IVideoRenderSupport;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.CornerPin", category=Categories.DISTORT)
public class CornerPin {

	@Property
	private IAnimatableVec2d upperLeft;

	@Property
	private IAnimatableVec2d upperRight;

	@Property
	private IAnimatableVec2d lowerRight;

	@Property
	private IAnimatableVec2d lowerLeft;

	@Property("true")
	private IAnimatableBoolean perspective;

	@Property("true")
	private IAnimatableBoolean interpolation;


	private final IVideoEffectContext context;

	private final IVideoRenderSupport support;

	private final IShaderProgram[] programs;

	@Inject
	public CornerPin(IVideoEffectContext context,
			IVideoRenderSupport support, IShaderRegistry shaders) {

		this.context = context;
		this.support = support;

		programs = new IShaderProgram[7];
		programs[0] = shaders.getProgram(CornerPin.class, "PERSPECTIVE");
		programs[1] = shaders.getProgram(CornerPin.class, "PERSPECTIVE_H");
		programs[2] = shaders.getProgram(CornerPin.class, "PERSPECTIVE_V");
		programs[3] = shaders.getProgram(CornerPin.class, "PROPORTIONAL");
		programs[4] = shaders.getProgram(CornerPin.class, "PROPORTIONAL_H");
		programs[5] = shaders.getProgram(CornerPin.class, "PROPORTIONAL_V");
		programs[6] = shaders.getProgram(CornerPin.class, "PARALLELOGRAM");
	}

	private Vec2d[] expand(Vec2d p0, Vec2d p1, Vec2d p2, Vec2d p3) {
		List<Vec2d> list = new ArrayList<Vec2d>();

		double signum = Double.NaN;
		double expand = 3;

		for (Vec2d[] p : new Vec2d[][] {{p3,p0,p1},{p0,p1,p2},{p1,p2,p3},{p2,p3,p0}}) {
			Vector3d v1 = new Vector3d(p[1].x-p[0].x, p[1].y-p[0].y, 0);
			Vector3d v2 = new Vector3d(p[2].x-p[1].x, p[2].y-p[1].y, 0);
			Vector3d c = new Vector3d();
			c.cross(v1, v2);

			if (c.lengthSquared() == 0) {
				return null;
			}

			if (Double.isNaN(signum)) {
				// 初回の符号でポリゴンの向きを判定
				signum = Math.signum(c.z);
			} else if (Math.signum(c.z) != signum) {
				// 符号が異なる場合は凹型ポリゴン
				return null;
			}

			Vector3d v3 = new Vector3d();
			v3.cross(v1, c);
			v3.normalize();

			Vector3d v4 = new Vector3d();
			v4.cross(v2, c);
			v4.normalize();

			list.add(new Vec2d(p[1].x+expand*v3.x, p[1].y+expand*v3.y));
			list.add(new Vec2d(p[1].x+expand*v4.x, p[1].y+expand*v4.y));
		}

		return list.toArray(new Vec2d[list.size()]);
	}

	private VideoBounds calcBounds(Vec2d ... vertices) {
		double left, upper, right, lower;
		left = upper = Double.POSITIVE_INFINITY;
		right = lower = Double.NEGATIVE_INFINITY;

		for (Vec2d v : vertices) {
			left  = Math.min(v.x, left);
			upper = Math.min(v.y, upper);
			right = Math.max(v.x, right);
			lower = Math.max(v.y, lower);
		}

		return new VideoBounds(left, upper,
				(int)Math.ceil(right-left),
				(int)Math.ceil(lower-upper));
	}

	private Vector3d calcLine(Vec2d p1, Vec2d p2) {
		double x2_x1 = p2.x - p1.x;
		double y2_y1 = p2.y - p1.y;
		return new Vector3d(y2_y1, -x2_x1, x2_x1*p1.y - y2_y1*p1.x);
	}

	private Point2d calcIntersection(Vector3d line1, Vector3d line2) {
		double a1b2_a2b1 = line1.x*line2.y - line2.x*line1.y;
		if (a1b2_a2b1 == 0) {
			return null;
		}
		return new Point2d(
				(line1.y*line2.z - line2.y*line1.z) / a1b2_a2b1,
				(line2.x*line1.z - line1.x*line2.z) / a1b2_a2b1);
	}

	public VideoBounds getVideoBounds() {
		Resolution resolution = context.getVideoResolution();
		Vec2d upperLeft = resolution.scale(context.value(this.upperLeft));
		Vec2d upperRight = resolution.scale(context.value(this.upperRight));
		Vec2d lowerRight = resolution.scale(context.value(this.lowerRight));
		Vec2d lowerLeft = resolution.scale(context.value(this.lowerLeft));

		Vec2d[] expanded = expand(upperLeft, upperRight, lowerRight, lowerLeft);
		if (expanded != null) {
			return calcBounds(expanded);
		} else {
			return calcBounds(upperLeft, upperRight, lowerRight, lowerLeft);
		}
	}

	public IVideoBuffer doVideoEffect() {
		Resolution resolution = context.getVideoResolution();
		Vec2d upperLeft = resolution.scale(context.value(this.upperLeft));
		Vec2d upperRight = resolution.scale(context.value(this.upperRight));
		Vec2d lowerRight = resolution.scale(context.value(this.lowerRight));
		Vec2d lowerLeft = resolution.scale(context.value(this.lowerLeft));

		final Vec2d[] expanded = expand(upperLeft, upperRight, lowerRight, lowerLeft);
		if (expanded == null) {
			return drawPolygon(upperLeft, upperRight, lowerRight, lowerLeft);
		}

		final VideoBounds outputBounds = calcBounds(expanded);
		if (outputBounds.isEmpty()) {
			return support.createVideoBuffer(outputBounds);
		}

		IVideoBuffer source = context.doPreviousEffect();
		if (source.getBounds().isEmpty()) {
			return source;
		}

		IVideoBuffer buffer = null;
		try {
			buffer = support.createVideoBuffer(outputBounds);
			buffer.clear();

			boolean perspective = context.value(this.perspective);
			boolean interpolation = context.value(this.interpolation);

			Vector3d upperLine = calcLine(upperLeft, upperRight);
			Vector3d rightLine = calcLine(upperRight, lowerRight);
			Vector3d lowerLine = calcLine(lowerRight, lowerLeft);
			Vector3d leftLine = calcLine(lowerLeft, upperLeft);

			Point2d hInf = calcIntersection(upperLine, lowerLine);
			Point2d vInf = calcIntersection(leftLine, rightLine);

			Set<GLUniformData> uniforms = new HashSet<GLUniformData>();
			uniforms.add(new GLUniformData("texture", 0));
			uniforms.add(new GLUniformData("fragCoordOffset", 2, toFloatBuffer(outputBounds.x, outputBounds.y)));
			uniforms.add(new GLUniformData("upperLine", 3, toFloatBuffer(upperLine.x, upperLine.y, upperLine.z)));
			uniforms.add(new GLUniformData("rightLine", 3, toFloatBuffer(rightLine.x, rightLine.y, rightLine.z)));
			uniforms.add(new GLUniformData("lowerLine", 3, toFloatBuffer(lowerLine.x, lowerLine.y, lowerLine.z)));
			uniforms.add(new GLUniformData("leftLine", 3, toFloatBuffer(leftLine.x, leftLine.y, leftLine.z)));

			IShaderProgram program;
			if (hInf == null && vInf == null) {
				program = programs[6];
			} else if (hInf == null) {
				program = perspective ? programs[2] : programs[5];
				uniforms.add(new GLUniformData("vInf", 2, toFloatBuffer(vInf.x, vInf.y)));
			} else if (vInf == null) {
				program = perspective ? programs[1] : programs[4];
				uniforms.add(new GLUniformData("hInf", 2, toFloatBuffer(hInf.x, hInf.y)));
			} else {
				program = perspective ? programs[0] : programs[3];
				uniforms.add(new GLUniformData("hInf", 2, toFloatBuffer(hInf.x, hInf.y)));
				uniforms.add(new GLUniformData("vInf", 2, toFloatBuffer(vInf.x, vInf.y)));
			}

			Runnable operation = new Runnable() {
				public void run() {
					support.ortho2D(outputBounds);

					GL2 gl = context.getGL().getGL2();
					gl.glColor4f(1, 1, 1, 1);

					gl.glBegin(GL2.GL_POLYGON);
					for (Vec2d v : expanded) {
						gl.glVertex2f((float)v.x, (float)v.y);
					}
					gl.glEnd();
				}
			};

			source.setTextureFilter((context.getQuality() == Quality.DRAFT || resolution.scale < 1)
					? TextureFilter.NEAREST : interpolation ? TextureFilter.MIPMAP : TextureFilter.LINEAR);

			int pushAttribs = GL2.GL_CURRENT_BIT;
			support.useShaderProgram(program, uniforms, operation, pushAttribs, buffer, source);

			IVideoBuffer result = buffer;
			buffer = null;
			return result;

		} finally {
			if (buffer != null) buffer.dispose();
			source.dispose();
		}
	}

	private FloatBuffer toFloatBuffer(double...values) {
		float[] farray = new float[values.length];
		for (int i = 0; i < values.length; ++i) {
			farray[i] = (float)values[i];
		}
		return FloatBuffer.wrap(farray);
	}

	@ShaderSource
	public static final String[] PERSPECTIVE = createShaderSource(true, true, true);

	@ShaderSource
	public static final String[] PERSPECTIVE_H = createShaderSource(true, true, false);

	@ShaderSource
	public static final String[] PERSPECTIVE_V = createShaderSource(true, false, true);

	@ShaderSource
	public static final String[] PROPORTIONAL = createShaderSource(false, true, true);

	@ShaderSource
	public static final String[] PROPORTIONAL_H = createShaderSource(false, true, false);

	@ShaderSource
	public static final String[] PROPORTIONAL_V = createShaderSource(false, false, true);

	@ShaderSource
	public static final String[] PARALLELOGRAM = createShaderSource(false, false, false);


	private static String[] createShaderSource(boolean perspective, boolean horizontal, boolean vertical) {
		boolean p = perspective;
		boolean h = horizontal;
		boolean v = vertical;
		return new String[] {
			p ? "#define PERSPECTIVE" : "",
			h ? "#define HORIZONTAL" : "",
			v ? "#define VERTICAL" : "",
				"",
				"uniform sampler2D texture;",
				"uniform vec2 fragCoordOffset;",
				"uniform vec3 upperLine;",
				"uniform vec3 rightLine;",
				"uniform vec3 lowerLine;",
				"uniform vec3 leftLine;",
				"",
				"#ifdef HORIZONTAL",
				"	uniform vec2 hInf;",
				"#endif",
				"",
				"#ifdef VERTICAL",
				"	uniform vec2 vInf;",
				"#endif",
				"",
				"vec3 calcLine(vec2 p1, vec2 p2)",
				"{",
				"	vec2 v = p2 - p1;",
				"	return vec3(v.y, -v.x, v.x*p1.y - v.y*p1.x);",
				"}",
				"",
				"vec2 calcIntersection(vec3 line1, vec3 line2)",
				"{",
				"	return vec2(line1.y*line2.z - line2.y*line1.z,",
				"				line2.x*line1.z - line1.x*line2.z)",
				"			/ (line1.x*line2.y - line2.x*line1.y);",
				"}",
				"",
				"void main(void)",
				"{",
				"	vec2 xy = gl_FragCoord.xy + fragCoordOffset;",
				"	vec3 line;",
				"	vec2 ip1, ip2;",
				"	float d1, d2, d;",
				"	float s, t;",
				"",
				"#ifdef HORIZONTAL",
				"	line = calcLine(xy, hInf);",
				"#else",
				"	line = vec3(upperLine.xy, -dot(upperLine.xy, xy));",
				"#endif",
				"	ip1 = calcIntersection(line, leftLine);",
				"	ip2 = calcIntersection(line, rightLine);",
				"#if defined(PERSPECTIVE) && defined(HORIZONTAL)",
				"	d1 = distance(ip1, hInf);",
				"	d2 = distance(ip2, hInf);",
				"	d = distance(xy, hInf);",
				"	s = (1.0/d-1.0/d1) / (1.0/d2-1.0/d1);",
				"#else",
				"	s = sign(dot(leftLine, vec3(xy, 1.0)) * dot(leftLine, vec3(ip2, 1.0)))",
				"			* distance(xy, ip1) / distance(ip2, ip1);",
				"#endif",
				"",
				"#ifdef VERTICAL",
				"	line = calcLine(xy, vInf);",
				"#else",
				"	line = vec3(leftLine.xy, -dot(leftLine.xy, xy));",
				"#endif",
				"	ip1 = calcIntersection(line, upperLine);",
				"	ip2 = calcIntersection(line, lowerLine);",
				"#if defined(PERSPECTIVE) && defined(VERTICAL)",
				"	d1 = distance(ip1, vInf);",
				"	d2 = distance(ip2, vInf);",
				"	d = distance(xy, vInf);",
				"	t = (1.0/d-1.0/d1) / (1.0/d2-1.0/d1);",
				"#else",
				"	t = sign(dot(upperLine, vec3(xy, 1.0)) * dot(upperLine, vec3(ip2, 1.0)))",
				"			* distance(xy, ip1) / distance(ip2, ip1);",
				"#endif",
				"",
				"	gl_FragColor = texture2D(texture, vec2(s, t));",
				"}"
		};
	}

	private IVideoBuffer drawPolygon(final Vec2d ... vertices) {
		final VideoBounds outputBounds = calcBounds(vertices);
		if (outputBounds.isEmpty()) {
			return support.createVideoBuffer(outputBounds);
		}

		IVideoBuffer buffer = null;
		try {
			buffer = support.createVideoBuffer(outputBounds);
			buffer.clear();

			Runnable operation = new Runnable() {
				public void run() {
					support.ortho2D(outputBounds);

					GL2 gl = context.getGL().getGL2();
					gl.glColor4f(1, 0, 0, 1);
					gl.glLineWidth(2);

					gl.glBegin(GL2.GL_LINE_LOOP);
					for (Vec2d v : vertices) {
						gl.glVertex2f((float)v.x, (float)v.y);
					}
					gl.glEnd();
				}
			};

			int pushAttribs = GL2.GL_CURRENT_BIT | GL2.GL_LINE_BIT;
			support.useFramebuffer(operation, pushAttribs, buffer);

			IVideoBuffer result = buffer;
			buffer = null;
			return result;

		} finally {
			if (buffer != null) buffer.dispose();
		}
	}

}
