/*
 * Copyright (c) 2010,2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.blurSharpen;

import java.nio.FloatBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.media.opengl.GLUniformData;

import ch.kuramo.javie.api.IAnimatableBoolean;
import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IAnimatableEnum;
import ch.kuramo.javie.api.IAnimatableInteger;
import ch.kuramo.javie.api.IAnimatableVec2d;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Resolution;
import ch.kuramo.javie.api.Vec2d;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.IVideoBuffer.TextureFilter;
import ch.kuramo.javie.api.IVideoBuffer.TextureWrapMode;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.annotations.ShaderSource;
import ch.kuramo.javie.api.annotations.Effect.Categories;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.api.services.IVideoRenderSupport;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.ZoomBlur", category=Categories.BLUR_AND_SHARPEN)
public class ZoomBlur {

	public enum Direction { ZOOM_IN, ZOOM_OUT, ZOOM_BOTH } 


	@Property(value="0", min="0", max="1000")
	private IAnimatableDouble amount;

	@Property
	private IAnimatableVec2d center;

	@Property("ZOOM_IN")
	private IAnimatableEnum<Direction> direction;

	@Property(value="100", min="0")
	private IAnimatableInteger maxSamples;

	@Property("true")
	private IAnimatableBoolean fast;


	private final IVideoEffectContext context;

	private final IVideoRenderSupport support;

	private final IShaderProgram zoomInProgram;

	private final IShaderProgram zoomOutProgram;

	private final IShaderProgram zoomBothProgram;

	@Inject
	public ZoomBlur(IVideoEffectContext context, IVideoRenderSupport support, IShaderRegistry shaders) {
		this.context = context;
		this.support = support;
		zoomInProgram = shaders.getProgram(ZoomBlur.class, "ZOOM_IN");
		zoomOutProgram = shaders.getProgram(ZoomBlur.class, "ZOOM_OUT");
		zoomBothProgram = shaders.getProgram(ZoomBlur.class, "ZOOM_BOTH");
	}

	public IVideoBuffer doVideoEffect() {
		IVideoBuffer source = context.doPreviousEffect();
		VideoBounds bounds = source.getBounds();
		if (bounds.isEmpty()) {
			return source;
		}

		Resolution resolution = context.getVideoResolution();

		double amount = resolution.scale(context.value(this.amount));
		int maxSamples = context.value(this.maxSamples);
		if (amount == 0 || maxSamples == 0) {
			return source;
		}

		Vec2d center = resolution.scale(context.value(this.center));
		Direction direction = context.value(this.direction);
		boolean fast = context.value(this.fast);

		Set<IVideoBuffer> tmpBuffers = new HashSet<IVideoBuffer>();
		tmpBuffers.add(source);

		try {
			Set<GLUniformData> uniforms = new HashSet<GLUniformData>();

			if (fast) {
				double amountIn, amountOut;
				switch (direction) {
					case ZOOM_IN:
						amountIn = amountOut = Math.min(amount/resolution.scale(200.0), 1) / maxSamples * 5;
						break;

					case ZOOM_OUT:
						amountIn = amountOut = amount/resolution.scale(200.0) / maxSamples * 5;
						break;

					case ZOOM_BOTH:
						amountIn = Math.min(amount/resolution.scale(400.0), 1) / maxSamples * 2.5;
						amountOut = amount/resolution.scale(400.0) / maxSamples * 2.5;
						break;

					default:
						throw new RuntimeException("unknown zoom direction: " + direction);
				}
				uniforms.add(new GLUniformData("amountIn", (float)Math.min(amountIn, 1)));
				uniforms.add(new GLUniformData("amountOut", (float)amountOut));
				uniforms.add(new GLUniformData("source", 0));
				uniforms.add(new GLUniformData("center", 2, toFloatBuffer(center.x-bounds.x, center.y-bounds.y)));
				uniforms.add(new GLUniformData("size", 2, toFloatBuffer(bounds.width, bounds.height)));
				uniforms.add(new GLUniformData("maxSamples", (float)5));

				for (int i = 0; i < 2; ++i) {
					source.setTextureFilter(TextureFilter.LINEAR);
					source.setTextureWrapMode(TextureWrapMode.CLAMP_TO_EDGE);
					source = support.useShaderProgram(zoomBothProgram, uniforms, null, source);
					tmpBuffers.add(source);
				}

				uniforms.clear();
				maxSamples = (int) Math.ceil(maxSamples / 5.0);
			}

			IShaderProgram program;
			switch (direction) {
				case ZOOM_IN:
					program = zoomInProgram;
					amount /= resolution.scale(200.0);
					uniforms.add(new GLUniformData("amountIn", (float)Math.min(amount, 1)));
					break;

				case ZOOM_OUT:
					program = zoomOutProgram;
					amount /= resolution.scale(200.0);
					uniforms.add(new GLUniformData("amountOut", (float)amount));
					break;

				case ZOOM_BOTH:
					program = zoomBothProgram;
					amount /= resolution.scale(400.0);
					uniforms.add(new GLUniformData("amountIn", (float)Math.min(amount, 1)));
					uniforms.add(new GLUniformData("amountOut", (float)amount));
					break;

				default:
					throw new RuntimeException("unknown zoom direction: " + direction);
			}

			uniforms.add(new GLUniformData("source", 0));
			uniforms.add(new GLUniformData("center", 2, toFloatBuffer(center.x-bounds.x, center.y-bounds.y)));
			uniforms.add(new GLUniformData("size", 2, toFloatBuffer(bounds.width, bounds.height)));
			uniforms.add(new GLUniformData("maxSamples", (float)maxSamples));

			source.setTextureFilter(TextureFilter.LINEAR);
			source.setTextureWrapMode(TextureWrapMode.CLAMP_TO_EDGE);

			return support.useShaderProgram(program, uniforms, null, source);

		} finally {
			for (IVideoBuffer vb : tmpBuffers) {
				vb.dispose();
			}
		}
	}

	private FloatBuffer toFloatBuffer(double... values) {
		float[] array = new float[values.length];
		for (int i = 0; i < values.length; ++i) {
			array[i] = (float)values[i];
		}
		return FloatBuffer.wrap(array);
	}

	private static String[] createShaderSource(Direction direction) {
		boolean in = (direction != Direction.ZOOM_OUT);
		boolean out = (direction != Direction.ZOOM_IN);

		List<String> list = new ArrayList<String>(Arrays.asList(new String[] {
					"uniform sampler2D source;",
			in ?	"uniform float amountIn;" : "",
			out ?	"uniform float amountOut;" : "",
					"uniform vec2 center;",
					"uniform vec2 size;",
					"uniform float maxSamples;",
					"",
					"void main(void)",
					"{",
					"	vec2 coord = gl_TexCoord[0].st;",
					"	vec2 v = center - coord * size;",
					"	float d = length(v);",
					"",
					"	vec4 color = texture2D(source, coord);",
					"",
					"	if (d > 0.0) {",
					"		v /= size;",
					"		int n = 0;"
		}));

		if (in) {
			list.addAll(Arrays.asList(new String[] {
					"		int   n1 = int(min(amountIn*d+1.0, maxSamples));",
					"		vec2  v1 = v*amountIn/float(n1);",
					"		for (int i = 1; i <= n1; ++i) {",
					"			color += texture2D(source, coord + v1*float(i));",
					"		}",
					"		n += n1;"
			}));
		}

		if (out) {
			list.addAll(Arrays.asList(new String[] {
					"		int   n2 = int(min(amountOut*d+1.0, maxSamples));",
					"		vec2  v2 = v*amountOut/float(n2);",
					"		for (int i = 1; i <= n2; ++i) {",
					"			color += texture2D(source, coord - v2*float(i));",
					"		}",
					"		n += n2;"
			}));
		}

		list.addAll(Arrays.asList(new String[] {
					"",
					"		color /= float(n+1);",
					"	}",
					"",
					"	gl_FragColor = color;",
					"}"
		}));

		return list.toArray(new String[list.size()]);
	}

	@ShaderSource
	public static final String[] ZOOM_IN = createShaderSource(Direction.ZOOM_IN);

	@ShaderSource
	public static final String[] ZOOM_OUT = createShaderSource(Direction.ZOOM_OUT);

	@ShaderSource
	public static final String[] ZOOM_BOTH = createShaderSource(Direction.ZOOM_BOTH);

}
