/*
 * Copyright (c) 2010,2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#include "stdafx.h"
#include "DirectShowOutputJNI.h"
#include "DirectShowOutput.h"


class PrimitiveArrayCritical
{
public:
	PrimitiveArrayCritical(JNIEnv* env, jarray array) : mEnv(env), mArray(array), mBuffer(NULL) { }
	~PrimitiveArrayCritical() { Release(); }

	operator void* ()
	{
		if (mBuffer == NULL) {
			mBuffer = mEnv->GetPrimitiveArrayCritical(mArray, NULL);
		}
		return mBuffer;
	}

	void Release()
	{
		if (mBuffer != NULL) {
			mEnv->ReleasePrimitiveArrayCritical(mArray, mBuffer, 0);
		}
	}

private:
	JNIEnv*	mEnv;
	jarray	mArray;
	void*	mBuffer;
};

class JStringToWSTR
{
public:
	JStringToWSTR(JNIEnv* env, jstring jstr) : mWSTR(NULL)
	{
		jsize jstrLen = env->GetStringLength(jstr);
		mWSTR = (LPWSTR) malloc(sizeof(jchar) * (jstrLen + 1));

		const jchar* jchars = env->GetStringChars(jstr, NULL);
		memcpy(mWSTR, jchars, sizeof(jchar) * jstrLen);
		mWSTR[jstrLen] = 0;

		env->ReleaseStringChars(jstr, jchars);
	}

	~JStringToWSTR()
	{
		if (mWSTR != NULL) {
			free(mWSTR);
			mWSTR = NULL;
		}
	}

	operator LPWSTR () { return mWSTR; }

private:
	LPWSTR	mWSTR;
};


JNIEXPORT jint JNICALL JNI_OnLoad(JavaVM* vm, void* reserved)
{
	return JNI_VERSION_1_4;
}

JNIEXPORT jint JNICALL Java_ch_kuramo_javie_core_output_WindowsDirectShowOutput_initialize
  (JNIEnv* env, jclass, jlongArray nativeObjPointer, jstring filename,
   jint vWidth, jint vHeight, jlong vFrameDuration, jboolean vHasAlpha,
   jstring vcName, jdouble vcQuality, jint vcKeyFrameRate, jint vcPFramesPerKey,
   jlong vcBitRate, jlong vcWindowSize, jbyteArray vcConfig,
   jint aChannels, jint aSampleRate, jint aSampleSize, jboolean aFloat)
{
	bool addVideo = (vWidth > 0 && vHeight > 0 && vFrameDuration > 0);
	bool addAudio = (aChannels > 0 && aSampleRate > 0 && aSampleSize > 0);

	if (!addVideo && !addAudio) {
		return E_INVALIDARG;
	}


	HRESULT hr;

	hr = CoInitializeEx(NULL, COINIT_MULTITHREADED);
	if (FAILED(hr)) return hr;


	DirectShowOutput* dsOut = new DirectShowOutput();
	if (dsOut == NULL) {
		hr = E_OUTOFMEMORY;
		goto bail;
	}

	hr = dsOut->Initialize(JStringToWSTR(env, filename));
	if (FAILED(hr)) goto bail;

	if (addVideo) {
		hr = dsOut->AddVideoSource(vWidth, vHeight, vFrameDuration, vHasAlpha,
				(vcName != NULL) ? (LPWSTR)JStringToWSTR(env, vcName) : NULL,
				vcQuality, vcKeyFrameRate, vcPFramesPerKey, vcBitRate, vcWindowSize,
				(vcConfig != NULL) ? (void*)PrimitiveArrayCritical(env, vcConfig) : NULL,
				(vcConfig != NULL) ? env->GetArrayLength(vcConfig) : 0);
		if (FAILED(hr)) goto bail;
	}

	if (addAudio) {
		hr = dsOut->AddAudioSource(aChannels, aSampleRate, aSampleSize, aFloat);
		if (FAILED(hr)) goto bail;
	}

	if (addVideo && addAudio) {
		hr = dsOut->SetMasterStreamAndInterleaving(vFrameDuration);
		if (FAILED(hr)) goto bail;
	}

	hr = dsOut->RunGraph();
	if (FAILED(hr)) goto bail;

	jlong buf[1] = { reinterpret_cast<jlong>(dsOut) };
	env->SetLongArrayRegion(nativeObjPointer, 0, 1, buf);
	return S_OK;

bail:
	delete dsOut;
	CoUninitialize();
	return hr;
}

JNIEXPORT jint JNICALL Java_ch_kuramo_javie_core_output_WindowsDirectShowOutput_finish
  (JNIEnv *, jclass, jlong nativeObjPointer)
{
	DirectShowOutput* dsOut = reinterpret_cast<DirectShowOutput*>(nativeObjPointer);
	delete dsOut;

	CoUninitialize();
	return S_OK;
}

JNIEXPORT jint JNICALL Java_ch_kuramo_javie_core_output_WindowsDirectShowOutput_writeVideo
  (JNIEnv* env, jclass, jlong nativeObjPointer,
   jlong dsTimeStart, jlong dsTimeEnd, jbyteArray buffer)
{
	DirectShowOutput* dsOut = reinterpret_cast<DirectShowOutput*>(nativeObjPointer);
	dsOut->WriteVideo(dsTimeStart, dsTimeEnd, PrimitiveArrayCritical(env, buffer));
	return S_OK;
}

JNIEXPORT jint JNICALL Java_ch_kuramo_javie_core_output_WindowsDirectShowOutput_writeAudio
  (JNIEnv* env, jclass, jlong nativeObjPointer,
   jlong dsTime, jint audioFrameCount, jobject buffer)
{
	DirectShowOutput* dsOut = reinterpret_cast<DirectShowOutput*>(nativeObjPointer);
	dsOut->WriteAudio(dsTime, audioFrameCount, PrimitiveArrayCritical(env, (jarray)buffer));
	return S_OK;
}

JNIEXPORT jint JNICALL Java_ch_kuramo_javie_core_output_WindowsDirectShowOutput_coInitializeEx
  (JNIEnv *, jclass)
{
	HRESULT hr = CoInitializeEx(NULL, COINIT_MULTITHREADED);
	if (FAILED(hr)) return hr;
	return S_OK;
}

JNIEXPORT jint JNICALL Java_ch_kuramo_javie_core_output_WindowsDirectShowOutput_coUninitialize
  (JNIEnv *, jclass)
{
	CoUninitialize();
	return S_OK;
}

JNIEXPORT jint JNICALL Java_ch_kuramo_javie_core_output_WindowsDirectShowOutput_getVideoCompressors
  (JNIEnv* env, jclass, jobject list,
   jint width, jint height, jlong frameDuration, jboolean hasAlpha)
{
	jclass descClass = env->FindClass("Lch/kuramo/javie/core/output/WindowsDirectShowOutput$VideoCompressorDescriptor;");
	if (descClass == NULL) return -1;

	jmethodID descCnstr = env->GetMethodID(descClass, "<init>", "(Ljava/lang/String;Ljava/lang/String;IZZDIIJ)V");
	if (descCnstr == NULL) return -1;

	jmethodID addMethod = env->GetMethodID(env->GetObjectClass(list), "add", "(Ljava/lang/Object;)Z");
	if (addMethod == NULL) return -1;

	HRESULT hr;
	VideoCompressorEnumerator vcompEnum(width, height, frameDuration, hasAlpha);

	while ((hr = vcompEnum.Next()) == S_OK) {
		LPCWSTR displayName = vcompEnum.GetDisplayName();
		jstring jdisplayName = env->NewString((jchar*)displayName, wcslen(displayName));

		LPCWSTR friendlyName = vcompEnum.GetFriendlyName();
		jstring jfriendlyName = env->NewString((jchar*)friendlyName, wcslen(friendlyName));

		jint capabilities = vcompEnum.GetCapabilities();
		jboolean canConfigDialog = vcompEnum.GetCanConfigDialog();
		jboolean canAboutDialog = vcompEnum.GetCanAboutDialog();
		jdouble defaultQuality = vcompEnum.GetDefaultQuality();
		jint defaultKeyFrameRate = vcompEnum.GetDefaultKeyFrameRate();
		jint defaultPFramesPerKey = vcompEnum.GetDefaultPFramesPerKey();
		jlong defaultBitRate = vcompEnum.GetDefaultBitRate();

		env->CallBooleanMethod(list, addMethod, env->NewObject(descClass, descCnstr,
			jdisplayName, jfriendlyName, capabilities, canConfigDialog, canAboutDialog,
			defaultQuality, defaultKeyFrameRate, defaultPFramesPerKey, defaultBitRate));
	}

	return FAILED(hr) ? hr : S_OK;
}

JNIEXPORT jint JNICALL Java_ch_kuramo_javie_core_output_WindowsDirectShowOutput_openVideoCompressorConfigDialog
  (JNIEnv* env, jclass,
   jint width, jint height, jlong frameDuration, jboolean hasAlpha,
   jstring jdisplayName, jobjectArray ioConfig, jlong hwnd)
{
	jbyteArray configArray = (jbyteArray) env->GetObjectArrayElement(ioConfig, 0);
	jbyte* configElems = NULL;
	long configLen = 0;

	void* config = NULL;

	if (configArray != NULL) {
		configElems = env->GetByteArrayElements(configArray, NULL);
		configLen = env->GetArrayLength(configArray);
		config = configElems;
	}

	HRESULT hr = VideoCompressorUtil::OpenConfigDialog(
		width, height, frameDuration, hasAlpha,
		JStringToWSTR(env, jdisplayName), &config, &configLen, (HWND)hwnd);

	if (configArray != NULL) {
		env->ReleaseByteArrayElements(configArray, configElems, JNI_ABORT);
	}

	if (hr == S_OK) {
		jbyteArray newArray = env->NewByteArray(configLen);
		if (newArray != NULL) {
			memcpy(PrimitiveArrayCritical(env, newArray), config, configLen);
			env->SetObjectArrayElement(ioConfig, 0, newArray);
		} else {
			hr = E_OUTOFMEMORY;
		}
	}

	if (config != NULL) {
		free(config);
	}

	return hr;
}

JNIEXPORT jint JNICALL Java_ch_kuramo_javie_core_output_WindowsDirectShowOutput_openVideoCompressorAboutDialog
  (JNIEnv* env, jclass, jstring jdisplayName, jlong hwnd)
{
	return VideoCompressorUtil::OpenAboutDialog(JStringToWSTR(env, jdisplayName), (HWND)hwnd);
}
