package ch.kuramo.javie.core.internal;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.Arrays;

import javax.media.opengl.GL2;
import javax.media.opengl.glu.GLU;
import javax.sound.sampled.AudioFormat;
import javax.sound.sampled.AudioInputStream;
import javax.sound.sampled.AudioSystem;
import javax.sound.sampled.UnsupportedAudioFileException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ch.kuramo.javie.api.AudioMode;
import ch.kuramo.javie.api.Color;
import ch.kuramo.javie.api.ColorMode;
import ch.kuramo.javie.api.IArray;
import ch.kuramo.javie.api.IAudioBuffer;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Resolution;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.services.IArrayPools;
import ch.kuramo.javie.api.services.IVideoRenderSupport;
import ch.kuramo.javie.core.FrameDuration;
import ch.kuramo.javie.core.JavieRuntimeException;
import ch.kuramo.javie.core.MediaOptions;
import ch.kuramo.javie.core.MediaSource;
import ch.kuramo.javie.core.MediaOptions.AlphaType;
import ch.kuramo.javie.core.MediaOptions.Option;
import ch.kuramo.javie.core.services.AudioRenderSupport;
import ch.kuramo.javie.core.services.RenderContext;
import ch.kuramo.javie.core.services.SynchronousTaskThread;
import ch.kuramo.javie.core.services.SynchronousTaskThread.Task;
import ch.kuramo.javie.core.services.SynchronousTaskThread.TaskWithoutResult;

import com.google.inject.Inject;
import com.google.inject.Injector;

public class AviSynthSource implements MediaSource {

	private static final Logger logger = LoggerFactory.getLogger(AviSynthSource.class);


	private SynchronousTaskThread thread;

	private long envPointer;

	private long clipPointer;


	private VideoBounds bounds;

	private Time duration;

	private Time videoFrameDuration;


	private AudioFormat audioFormat;

	private boolean audioFloat;

	private Time audioDuration;

	private AudioMode audioMode;

	private AudioInputStream audioStream;

	private long audioStreamPosition;


	private final RenderContext context;

	private final IVideoRenderSupport vrSupport;

	private final AudioRenderSupport arSupport;

	private final IArrayPools arrayPools;

	private final Injector injector;


	@Inject
	public AviSynthSource(
			RenderContext context, IVideoRenderSupport vrSupport,
			AudioRenderSupport arSupport, IArrayPools arrayPools,
			Injector injector) {

		super();
		this.context = context;
		this.vrSupport = vrSupport;
		this.arSupport = arSupport;
		this.arrayPools = arrayPools;
		this.injector = injector;
	}

	public boolean initialize(File file) {
		if (!hasExtension(file, "avs")) {
			return false;
		}

		final byte[] filename = filenameToBytes(file.getAbsolutePath());

		SynchronousTaskThread thread = injector.getInstance(SynchronousTaskThread.class);
		thread.setName(getClass().getSimpleName() + ": " + file.getName());
		thread.start();
		this.thread = thread;

		thread.invoke(new TaskWithoutResult() {
			protected void runWithoutResult() throws Exception {
				envPointer = createScriptEnvironment();
				if (envPointer == 0) return;

				long[] result = openAVS(envPointer, filename);
				if (result == null || result[0] == 0) return;

				clipPointer = result[0];

				if (result[1] > 0) {
					bounds = new VideoBounds((int)result[1], (int)result[2]);
					videoFrameDuration = new Time(result[4], (int)result[3]);
					duration = Time.fromFrameNumber(result[5], videoFrameDuration);
				}

				if (result[6] > 0) {
					audioFormat = new AudioFormat(result[6], (int)result[7]*8, (int)result[8], true, false);
					audioFloat = (result[10] != 0);
					audioDuration = Time.fromFrameNumber(result[9], new Time(1, (int)result[6]));
					if (duration == null) {
						duration = audioDuration;
					}
				}
			}
		});

		if (isVideoAvailable() || isAudioAvailable()) {
			return true;
		} else {
			dispose();
			return false;
		}
	}

	protected boolean hasExtension(File file, String extension) {
		if (extension.length() > 0 && extension.charAt(0) == '.') {
			extension = extension.substring(1);
		}

		String fileName = file.getName();
		int lastDot = fileName.lastIndexOf('.');
		if (lastDot != -1) {
			return extension.toLowerCase().equals(
					fileName.substring(lastDot+1).toLowerCase());
		}

		return false;
	}

	private byte[] filenameToBytes(String filename) {
		filename += '\0';
		try {
			return filename.getBytes("Windows-31J");
		} catch (UnsupportedEncodingException e) {
			try {
				return filename.getBytes("ISO-8859-1");
			} catch (UnsupportedEncodingException e2) {
				return filename.getBytes();
			}
		}
	}

	public void dispose() {
		closeAudioStream();

		if (thread != null) {
			thread.exit(new TaskWithoutResult() {
				protected void runWithoutResult() throws Exception {
					if (clipPointer != 0) {
						closeAVS(clipPointer);
						clipPointer = 0;
					}
					if (envPointer != 0) {
						deleteScriptEnvironment(envPointer);
						envPointer = 0;
					}
				}
			});
		}

		bounds = null;
		duration = null;
		videoFrameDuration = null;

		audioFormat = null;
		audioDuration = null;
		audioMode = null;
	}

	public boolean isVideoAvailable() {
		return (bounds != null);
	}

	public boolean isAudioAvailable() {
		return (audioFormat != null);
	}

	public Time getDuration(MediaOptions options) {
		return duration;
	}

	public Time getVideoFrameDuration(MediaOptions options) {
		return videoFrameDuration;
	}

	public VideoBounds getVideoFrameBounds(MediaOptions options) {
		return bounds;
	}

	public IVideoBuffer getVideoFrame(Time mediaTime, MediaOptions options) {
		if (!isVideoAvailable()) {
			return null;
		}

		IVideoBuffer buffer = null;
		try {
			buffer = vrSupport.createVideoBuffer(bounds, ColorMode.RGBA8);

			if (mediaTime.timeValue < 0 || !mediaTime.before(getDuration(options))) {
				buffer.clear();
				IVideoBuffer result = buffer;
				buffer = null;
				return result;
			}

			final int frameNumber = (int) mediaTime.toFrameNumber(videoFrameDuration);
			final byte[] array = (byte[]) buffer.getArray();

			boolean successful = thread.invoke(new Task<Boolean>() {
				public Boolean run() throws Exception {
					boolean successful = getFrame(envPointer, clipPointer, frameNumber, array);
					if (!successful) {
						logger.error("getFrame failed.");
					}
					return successful;
				}
			});

			if (!successful) {
				return null;
			}

			return convertByResolutionAndOptions(buffer, options);

		} finally {
			if (buffer != null) buffer.dispose();
		}
	}

	private IVideoBuffer convertByResolutionAndOptions(
			IVideoBuffer srcBuffer, MediaOptions options) {

		Resolution resolution = context.getVideoResolution();

		final VideoBounds srcBounds = srcBuffer.getBounds();
		final VideoBounds dstBounds = resolution.scale(bounds);

		final float scale = (float)resolution.scale;
		final boolean flipVertical = options.isFlipVertical();
		final AlphaType alphaType = options.getAlphaType();
		final Color colorMatte = options.getColorMatte();

		Runnable operation = new Runnable() {
			public void run() {
				GL2 gl = context.getGL().getGL2();
				GLU glu = context.getGLU();

				gl.glViewport(0, 0, dstBounds.width, dstBounds.height);

				gl.glMatrixMode(GL2.GL_PROJECTION);
				gl.glLoadIdentity();
				if (flipVertical) {
					glu.gluOrtho2D(0, dstBounds.width, 0, dstBounds.height);
				} else {
					glu.gluOrtho2D(0, dstBounds.width, dstBounds.height, 0);
				}

				gl.glMatrixMode(GL2.GL_MODELVIEW);
				gl.glLoadIdentity();
				gl.glScalef(scale, scale, 1);

				switch (alphaType) {
					case IGNORE:
						gl.glClearColor(0, 0, 0, 1);
						gl.glClear(GL2.GL_COLOR_BUFFER_BIT);

						gl.glEnable(GL2.GL_BLEND);
						gl.glBlendFuncSeparate(GL2.GL_ONE, GL2.GL_ZERO, GL2.GL_ZERO, GL2.GL_ONE);
						gl.glBlendEquation(GL2.GL_FUNC_ADD);
						break;

					case STRAIGHT:
						gl.glClearColor(0, 0, 0, 0);
						gl.glClear(GL2.GL_COLOR_BUFFER_BIT);

						gl.glEnable(GL2.GL_BLEND);
						gl.glBlendFuncSeparate(GL2.GL_SRC_ALPHA, GL2.GL_ZERO, GL2.GL_ONE, GL2.GL_ZERO);
						gl.glBlendEquation(GL2.GL_FUNC_ADD);
						break;

					case PREMULTIPLIED:
						if (colorMatte.r == 0 && colorMatte.g == 0 && colorMatte.b == 0) {
							gl.glDisable(GL2.GL_BLEND);

						} else {
							gl.glClearColor((float)colorMatte.r, (float)colorMatte.g, (float)colorMatte.b, 1);
							gl.glClear(GL2.GL_COLOR_BUFFER_BIT);

							gl.glEnable(GL2.GL_BLEND);
							gl.glBlendFuncSeparate(GL2.GL_ONE, GL2.GL_ONE_MINUS_SRC_ALPHA, GL2.GL_ONE, GL2.GL_ZERO);
							gl.glBlendEquation(GL2.GL_FUNC_SUBTRACT);
						}
						break;
				}

				// テクスチャは useFramebuffer によってすでにバインドされている。
				//gl.glBindTexture(GL2.GL_TEXTURE_2D, srcTexture);
				//gl.glEnable(GL2.GL_TEXTURE_2D);

				gl.glColor4f(1, 1, 1, 1);

				gl.glBegin(GL2.GL_QUADS);
				gl.glTexCoord2f(0, 0); gl.glVertex2f(0, 0);
				gl.glTexCoord2f(1, 0); gl.glVertex2f(srcBounds.width, 0);
				gl.glTexCoord2f(1, 1); gl.glVertex2f(srcBounds.width, srcBounds.height);
				gl.glTexCoord2f(0, 1); gl.glVertex2f(0, srcBounds.height);
				gl.glEnd();
			}
		};

		int pushAttribs = GL2.GL_COLOR_BUFFER_BIT
						| GL2.GL_ENABLE_BIT
						| GL2.GL_CURRENT_BIT;

		IVideoBuffer buffer = null;
		try {
			buffer = vrSupport.createVideoBuffer(dstBounds);

			vrSupport.useFramebuffer(operation, pushAttribs, buffer, srcBuffer);

			IVideoBuffer result = buffer;
			buffer = null;
			return result;

		} finally {
			if (buffer != null) buffer.dispose();
		}
	}

	public synchronized IAudioBuffer getAudioChunk(Time mediaTime, MediaOptions options) {
		if (!isAudioAvailable()) {
			return null;
		}

		try {
			AudioMode audioMode = context.getAudioMode();
			long frameNumber = mediaTime.toFrameNumber(audioMode.sampleDuration);

			if (-frameNumber >= context.getAudioFrameCount() || !mediaTime.before(getDuration(options))) {
				IAudioBuffer ab = arSupport.createAudioBuffer();
				arSupport.clear(ab);
				return ab;
			}

			if (audioMode != this.audioMode || audioMode.frameSize*frameNumber != audioStreamPosition) {
				closeAudioStream();
			}

			if (audioStream == null) {
				audioStream = createAudioStream();
			}

			return readAudioData(frameNumber);

		} catch (IOException e) {
			// TODO throw new MediaInputException(e);
			throw new JavieRuntimeException(e);
		} catch (UnsupportedAudioFileException e) {
			// TODO throw new MediaInputException(e);
			throw new JavieRuntimeException(e);
		}
	}

	private AudioInputStream createAudioStream() throws IOException, UnsupportedAudioFileException {
		// サンプルレートと他のパラメータを同時に変換しようとすると失敗するようなので、2段階に分けて変換する。
		//
		// 変換の必要が無い場合、AudioSystem.getAudioInputStream は元のストリームをそのまま返すようなので
		// 以下のように無条件に変換を試みても無駄な変換は発生しないようである。

		this.audioMode = context.getAudioMode();
		int bits = this.audioMode.sampleSize * 8;
		int ch = this.audioMode.channels;

		long length = audioDuration.toFrameNumber(new Time(1, (int)audioFormat.getFrameRate()));
		AudioInputStream s1 = AudioSystem.getAudioInputStream(audioFormat, new AudioInputStream(
				new AviSynthAudioInputStream(audioFormat), audioFormat, length));

		// サンプルレート以外を変換
		AudioFormat f2 = new AudioFormat(s1.getFormat().getSampleRate(), bits, ch, true, false);
		AudioInputStream s2 = AudioSystem.getAudioInputStream(f2, s1);

		// サンプルレートを変換
		AudioFormat f3 = new AudioFormat(this.audioMode.sampleRate, bits, ch, true, false);
		return AudioSystem.getAudioInputStream(f3, s2);
	}

	private void closeAudioStream() {
		if (audioStream != null) {
			try {
				audioStream.close();
			} catch (IOException e) {
				// ログだけ書き出して無視
				logger.warn("failed to close AudioInputStream", e);
			}
			audioStream = null;
		}
		audioStreamPosition = 0;
	}

	private IAudioBuffer readAudioData(long frameNumber) throws IOException {
		IAudioBuffer ab = arSupport.createAudioBuffer();
		AudioMode audioMode = ab.getAudioMode();
		Object data = ab.getData();
		int dataOffset = 0;
		int dataLength = ab.getDataLength();
		int lengthInBytes = dataLength * audioMode.sampleSize;

		if (audioStreamPosition == 0) {
			if (frameNumber > 0) {
				long skip = audioMode.frameSize * frameNumber;
				long skipped = audioStream.skip((skip / audioMode.sampleRate) * audioMode.sampleRate);
				audioStreamPosition = skipped;
				skip -= skipped;
				if (skip > 0) {
					IArray<byte[]> array = arrayPools.getByteArray((int)skip);
					try {
						do {
							skipped = audioStream.read(array.getArray(), 0, (int)skip);
							if (skipped != -1) {
								audioStreamPosition += skipped;
								skip -= skipped;
							}
						} while (skipped != -1 && skip > 0);
					} finally {
						array.release();
					}
				}
				if (skip != 0) {
					logger.warn(String.format("skip failed: frameNumber=%d, skip=%d", frameNumber, skip));
					closeAudioStream();
					arSupport.clear(ab);
					return ab;
				}
			} else if (frameNumber < 0) {
				arSupport.clear(ab, 0, (int)-frameNumber);
				dataOffset = (int)-frameNumber * audioMode.channels;
				dataLength -= dataOffset;
				lengthInBytes -= dataOffset * audioMode.sampleSize;
			}
		}

		IArray<byte[]> array = arrayPools.getByteArray(lengthInBytes);
		try {
			byte[] buffer = array.getArray();

			int readBytes = 0;
			do {
				int n = audioStream.read(buffer, readBytes, lengthInBytes - readBytes);
				if (n == -1) {
					Arrays.fill(buffer, readBytes, lengthInBytes, (byte)0);
					break;
				}
				readBytes += n;
				audioStreamPosition += n;
			} while (readBytes < lengthInBytes);

			switch (audioMode.dataType) {
				case SHORT:
					ByteBuffer.wrap(buffer, 0, lengthInBytes).order(ByteOrder.nativeOrder())
							.asShortBuffer().get((short[]) data, dataOffset, dataLength);
					break;
				case INT:
					ByteBuffer.wrap(buffer, 0, lengthInBytes).order(ByteOrder.nativeOrder())
							.asIntBuffer().get((int[]) data, dataOffset, dataLength);
					break;
				case FLOAT:
					for (int i = dataOffset; i < dataLength; ++i) {
						int value =  (buffer[i*4  ] & 0xff)
								  | ((buffer[i*4+1] & 0xff) <<  8)
								  | ((buffer[i*4+2] & 0xff) << 16)
								  | ((buffer[i*4+3] & 0xff) << 24);
						((float[]) data)[i] = (float) value / Integer.MAX_VALUE;
					}
					break;
				default:
					throw new UnsupportedOperationException(
							"unsupported AudioMode.DataType: " + audioMode.dataType);
			}
		} finally {
			array.release();
		}

		return ab;
	}

	public MediaOptions validateOptions(MediaOptions options) {
		Option[] availableOptions;

		if (isVideoAvailable()) {
			availableOptions = new Option[] {
					Option.IGNORE_ALPHA, Option.STRAIGHT_ALPHA, Option.PREMULTIPLIED_ALPHA,
					Option.FLIP_VERTICAL
			};
		} else {
			availableOptions = new Option[0];
		}

		if (options == null) {
			options = new MediaOptions(availableOptions);

			if (isVideoAvailable()) {
				options.setAlphaType(AlphaType.IGNORE);
			}

		} else {
			options = options.clone();
			options.setAvailableOptions(availableOptions);
		}

		return options;
	}


	private class AviSynthAudioInputStream extends InputStream {

		private final AudioFormat format;

		private long positionInBytes;


		private AviSynthAudioInputStream(AudioFormat format) {
			super();
			this.format = format;
		}

		@Override
		public int read() throws IOException {
			byte[] b = new byte[1];
			return (read(b, 0, 0) != -1) ? b[0] : -1;
		}

		@Override
		public int read(final byte[] b, final int off, int len) throws IOException {
			final int frameSize = format.getFrameSize();

			if (len % frameSize != 0) {
				throw new IOException("can't read the length bytes that is not multiples of frameSize");
			}

			Time frameDuration = isVideoAvailable() ? videoFrameDuration : FrameDuration.FPS_29_97;
			final long count1 = (int)Math.round(format.getSampleRate()*frameDuration.toSecond());

			final long start = positionInBytes / frameSize;
			final long count = len / frameSize;

			boolean successful = thread.invoke(new Task<Boolean>() {
				public Boolean run() throws Exception {
					long s = start;
					long c = count;
					int o = off;
					while (c > 0) {
						long c2 = Math.min(count1, c);
						boolean successful = getAudio(envPointer, clipPointer, s, c2, b, o);
						if (!successful) {
							logger.error("getAudio failed.");
							return false;
						}
						s += c2;
						c -= c2;
						o += c2 * frameSize;
					}
					return true;
				}
			});
			if (!successful) {
				throw new IOException("error");
			}

			if (audioFloat) {
				for (int i = off; i < len; i += 4) {
					int bits =  (b[i  ] & 0xff)
							 | ((b[i+1] & 0xff) <<  8)
							 | ((b[i+2] & 0xff) << 16)
							 | ((b[i+3] & 0xff) << 24);
					int intValue = (int)(Float.intBitsToFloat(bits) * Integer.MAX_VALUE);

					b[i  ] = (byte)( intValue         & 0xff);
					b[i+1] = (byte)((intValue >>>  8) & 0xff);
					b[i+2] = (byte)((intValue >>> 16) & 0xff);
					b[i+3] = (byte)((intValue >>> 24) & 0xff);
				}
			}

			positionInBytes += len;
			return len;
		}

		@Override
		public long skip(long n) throws IOException {
			long actualSkip = 0;
			if (n > 0) {
				int frameSize = format.getFrameSize();
				actualSkip = (n / frameSize) * frameSize; 
				positionInBytes += actualSkip;
			}
			return actualSkip;
		}

	}


	static { System.loadLibrary("AviSynthSource"); }

	private native long createScriptEnvironment();

	private native void deleteScriptEnvironment(long envPointer);

	private native long[] openAVS(long envPointer, byte[] filename);

	private native void closeAVS(long clipPointer);

	private native boolean getFrame(
			long envPointer, long clipPointer,
			int frameNumber, byte[] buffer);

	private native boolean getAudio(
			long envPointer, long clipPointer,
			long start, long count, byte[] buffer, int offset);

}
