/*
 * Copyright (c) 2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.views.layercomp;

import java.util.List;
import java.util.Map;

import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.ActionContributionItem;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.resource.ImageRegistry;
import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.TextCellEditor;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.graphics.Cursor;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.MenuItem;
import org.eclipse.swt.widgets.TreeItem;

import ch.kuramo.javie.app.Activator;
import ch.kuramo.javie.app.EnumLabels;
import ch.kuramo.javie.app.PropertyUtil;
import ch.kuramo.javie.app.project.AddRemoveTAPropertiesOperation;
import ch.kuramo.javie.app.project.AddTASelectorOperation;
import ch.kuramo.javie.app.project.ModifyLayerPropertyOperation;
import ch.kuramo.javie.app.project.PerCharacter3DOperation;
import ch.kuramo.javie.app.project.ProjectManager;
import ch.kuramo.javie.app.project.RenameTextAnimatorOperation;
import ch.kuramo.javie.app.views.LayerCompositionView;
import ch.kuramo.javie.app.widgets.WSWin32;
import ch.kuramo.javie.core.AnimatableDouble;
import ch.kuramo.javie.core.AnimatableVec2d;
import ch.kuramo.javie.core.AnimatableVec3d;
import ch.kuramo.javie.core.LayerNature;
import ch.kuramo.javie.core.TAProperty;
import ch.kuramo.javie.core.TARangeSelector;
import ch.kuramo.javie.core.TASelector;
import ch.kuramo.javie.core.TAWigglySelector;
import ch.kuramo.javie.core.TextAnimator;
import ch.kuramo.javie.core.TextLayer;
import ch.kuramo.javie.core.Util;

public class TextAnimatorElement extends Element {

	private static final int[] UNDERLINE_DASH = new int[] { 2, 2 };

	private static final boolean COCOA = SWT.getPlatform().equals("cocoa");

	private static final boolean WIN32 = SWT.getPlatform().equals("win32");


	public final TextAnimator animator;

	private final List<Element> children = Util.newList();

	private final Map<Object, Element> childrenMap = Util.newMap();

	private boolean perCharacter3D;

	private final SwitchGroup enableSwitch = new SwitchGroup();

	private Rectangle valueArea;

	private TextCellEditor nameEditor;


	public TextAnimatorElement(TextAnimatorsElement parent, TextAnimator animator) {
		super(parent);
		this.animator = animator;

		enableSwitch.add(new EnableSwitch());

		if (WSWin32.isXPThemed()) {
			enableSwitch.setMarginLeft(11);
		}
	}

	@Override
	public String getColumnText(int columnIndex) {
		switch (columnIndex) {
			case LayerCompositionView.NAME_COL:
				return animator.getName();

			default:
				return null;
		}
	}

	@Override
	public boolean hasChildren() {
		return !(animator.getSelectors().isEmpty() && animator.getProperties().isEmpty());
	}

	@Override
	public Element[] getChildren() {
		prepareChildren();
		return children.toArray(new Element[children.size()]);
	}

	@Override
	public void paintColumn(Event event) {
		switch (event.index) {
			case LayerCompositionView.VALUE_COL: {
				String valueStr = ">";

				GC gc = event.gc;
				int x = event.x + 5;
				int y = event.y;
				int height = ((TreeItem) event.item).getBounds(event.index).height;
				boolean focused = ((event.detail & SWT.SELECTED) != 0 && ((Control) event.widget).isFocusControl());


				gc.setForeground(gc.getDevice().getSystemColor(
						focused ? SWT.COLOR_WHITE : SWT.COLOR_DARK_BLUE));

				Point extent = gc.textExtent(valueStr, SWT.DRAW_TRANSPARENT);

				y += (height - extent.y) / 2;
				gc.drawString(valueStr, x, y, true);

				valueArea = new Rectangle(x, y, extent.x, extent.y);

				y += extent.y;
				gc.setLineDash(UNDERLINE_DASH);
				if (COCOA) {
					gc.drawLine(x, y - 1, x + extent.x, y - 1);
				} else if (WIN32) {
					gc.drawLine(x, y, x + extent.x - 2, y);
				} else {
					gc.drawLine(x, y, x + extent.x, y);
				}
				break;
			}

			case LayerCompositionView.SHOWHIDE_COL:
				enableSwitch.paint(event);
				break;

			default:
				super.paintColumn(event);
				break;
		}
	}

	@Override
	public void updateCursor(MouseEvent event, int columnIndex) {
		switch (columnIndex) {
			case LayerCompositionView.VALUE_COL:
				Cursor cursor = null;
				if (valueArea != null && valueArea.contains(event.x, event.y)) {
					cursor = event.display.getSystemCursor(SWT.CURSOR_HAND);
				}
				viewer.getTree().setCursor(cursor);
				break;

			default:
				super.updateCursor(event, columnIndex);
				break;
		}
	}

	@Override
	public void mouseDown(MouseEvent event, int columnIndex) {
		switch (columnIndex) {
			case LayerCompositionView.VALUE_COL:
				if (event.button == 1 && valueArea != null && valueArea.contains(event.x, event.y)) {
					showMenu();
				}
				break;

			case LayerCompositionView.SHOWHIDE_COL:
				enableSwitch.mouseDown(event);
				break;

			default:
				super.mouseDown(event, columnIndex);
				break;
		}
	}

	private void showMenu() {
		Menu menu = createPopUpMenu(valueArea.x-5, valueArea.y/*+valueArea.height+2*/);

		MenuItem propItem = new MenuItem(menu, SWT.CASCADE);
		Menu propMenu = new Menu(propItem);
		propItem.setMenu(propMenu);
		propItem.setText("プロパティ");

		MenuItem selectorItem = new MenuItem(menu, SWT.CASCADE);
		Menu selectorMenu = new Menu(selectorItem);
		selectorItem.setMenu(selectorMenu);
		selectorItem.setText("セレクタ");


		final TextLayer layer = getLayer();
		boolean perCharacter3D = LayerNature.isThreeD(layer) && layer.isPerCharacter3D();

		Action action = new Action("文字単位の3D化", IAction.AS_CHECK_BOX) {
			public void run() {
				ProjectManager pm = (ProjectManager) viewer.getData(LayerCompositionView.PROJECT_MANAGER);
				pm.postOperation(PerCharacter3DOperation.toggle(pm, layer));
			}
		};
		action.setChecked(perCharacter3D);
		ActionContributionItem item = new ActionContributionItem(action);
		item.fill(propMenu, -1);

		new MenuItem(propMenu, SWT.SEPARATOR);


		Map<TAProperty, Action> propActions = Util.newMap();
		for (TAProperty p : TAProperty.values()) {
			final List<Object[]> data = Util.newList();
			switch (p) {
				case rotationX:
				case rotationY:
				case skewAxis:
				case characterRange:
					continue;
				case rotationZ:
					if (perCharacter3D) {
						data.add(new Object[] { layer, animator, TAProperty.rotationX });
						data.add(new Object[] { layer, animator, TAProperty.rotationY });
					}
					data.add(new Object[] { layer, animator, TAProperty.rotationZ });
					break;
				case skew:
					data.add(new Object[] { layer, animator, TAProperty.skew });
					data.add(new Object[] { layer, animator, TAProperty.skewAxis });
					break;
				case characterValue:
				case characterOffset:
					data.add(new Object[] { layer, animator, TAProperty.characterRange });
					data.add(new Object[] { layer, animator, p });
					break;
				default:
					data.add(new Object[] { layer, animator, p });
					break;
			}
			action = new Action(EnumLabels.getLabel(p)) {
				public void run() {
					ProjectManager pm = (ProjectManager) viewer.getData(LayerCompositionView.PROJECT_MANAGER);
					pm.postOperation(new AddRemoveTAPropertiesOperation(pm, data, true));
				}
			};
			propActions.put(p, action);
		}
		createPropertyMenuItems(propMenu, propActions);


		action = new Action("範囲") {
			public void run() {
				ProjectManager pm = (ProjectManager) viewer.getData(LayerCompositionView.PROJECT_MANAGER);
				pm.postOperation(new AddTASelectorOperation(pm, layer, animator, TARangeSelector.class));
			}
		};
		item = new ActionContributionItem(action);
		item.fill(selectorMenu, -1);

		action = new Action("ウィグリー") {
			public void run() {
				ProjectManager pm = (ProjectManager) viewer.getData(LayerCompositionView.PROJECT_MANAGER);
				pm.postOperation(new AddTASelectorOperation(pm, layer, animator, TAWigglySelector.class));
			}
		};
		item = new ActionContributionItem(action);
		item.fill(selectorMenu, -1);

//		action = new Action("エクスプレッション") {
//			public void run() {
//				ProjectManager pm = (ProjectManager) viewer.getData(LayerCompositionView.PROJECT_MANAGER);
//				pm.postOperation(new AddTASelectorOperation(pm, layer, animator, TAExpressionSelector.class));
//			}
//		};
//		item = new ActionContributionItem(action);
//		item.fill(selectorMenu, -1);


		menu.setVisible(true);
	}

	static void createPropertyMenuItems(Menu propMenu, Map<TAProperty, Action> actions) {
		Menu fillColorMenu = null;
		Menu strokeColorMenu = null;

		for (TAProperty p : TAProperty.values()) {
			Action action = actions.get(p);
			if (action == null) {
				continue;
			}

			Menu parentMenu = propMenu;

			switch (p) {
				case rotationZ:
					action.setText("回転");
					break;

				case fillRGB:
					new MenuItem(propMenu, SWT.SEPARATOR);
					MenuItem fillColorItem = new MenuItem(propMenu, SWT.CASCADE);
					fillColorMenu = new Menu(fillColorItem);
					fillColorItem.setMenu(fillColorMenu);
					fillColorItem.setText("塗りのカラー");
					parentMenu = fillColorMenu;
					action.setText("RGB");
					break;

				case fillHue:
				case fillSaturation:
				case fillLuminosity:
				case fillOpacity:
					action.setText(action.getText().replaceFirst("塗りの", "")); // TODO 国際化する際はリソースから取得するようにする
					parentMenu = fillColorMenu;
					break;

				case strokeRGB:
					MenuItem strokeColorItem = new MenuItem(propMenu, SWT.CASCADE);
					strokeColorMenu = new Menu(strokeColorItem);
					strokeColorItem.setMenu(strokeColorMenu);
					strokeColorItem.setText("線のカラー");
					parentMenu = strokeColorMenu;
					action.setText("RGB");
					break;

				case strokeHue:
				case strokeSaturation:
				case strokeLuminosity:
				case strokeOpacity:
					parentMenu = strokeColorMenu;
					action.setText(action.getText().replaceFirst("線の", "")); // TODO 国際化する際はリソースから取得するようにする
					break;

				case tracking:
				case characterValue:
				case blur:
					new MenuItem(propMenu, SWT.SEPARATOR);
					break;
			}

			ActionContributionItem item = new ActionContributionItem(action);
			item.fill(parentMenu, -1);
		}
	}

	@Override
	public boolean canEdit(int columnIndex) {
		switch (columnIndex) {
			case LayerCompositionView.NAME_COL:
				return Boolean.TRUE.equals(viewer.getData(LayerCompositionView.EDIT_ELEMENT_NAME));

			default:
				return false;
		}
	}

	@Override
	public CellEditor getCellEditor(int columnIndex) {
		switch (columnIndex) {
			case LayerCompositionView.NAME_COL:
				if (nameEditor == null) {
					nameEditor = new TextCellEditor(viewer.getTree(), SWT.SINGLE | SWT.BORDER);
					Control control = nameEditor.getControl();
					control.setBackground(control.getDisplay().getSystemColor(SWT.COLOR_WHITE));
				}
				return nameEditor;

			default:
				return null;
		}
	}

	@Override
	public Object getCellEditorValue(int columnIndex) {
		switch (columnIndex) {
			case LayerCompositionView.NAME_COL:
				return animator.getName();

			default:
				return null;
		}
	}

	@Override
	public void setCellEditorValue(int columnIndex, Object value) {
		ProjectManager pm = (ProjectManager) viewer.getData(LayerCompositionView.PROJECT_MANAGER);

		switch (columnIndex) {
			case LayerCompositionView.NAME_COL: {
				String newName = (String) value;
				if (newName.length() > 0) {
					pm.postOperation(new RenameTextAnimatorOperation(pm, getLayer(), animator, newName));
				}
				break;
			}
		}
	}

	TextLayer getLayer() {
		return ((TextAnimatorsElement) parent).getLayer();
	}

	private Element getSelectorElement(TASelector selector) {
		Element element = childrenMap.get(selector);
		if (element == null) {
			if (selector instanceof TARangeSelector) {
				element = new TARangeSelectorElement(this, (TARangeSelector)selector);

			} else if (selector instanceof TAWigglySelector) {
				element = new TAWigglySelectorElement(this, (TAWigglySelector)selector);

//			} else if (selector instanceof TAExpressionSelector) {
//				element = new TAExpressionSelectorElement(this, (TAExpressionSelector)selector);

			} else {
				throw new RuntimeException("unknown TASelector: " + selector.getClass().getName());
			}
			childrenMap.put(selector, element);
		}
		return element;
	}

	private Element getPropertyElement(TAProperty property) {
		if (!perCharacter3D) {
			switch (property) {
				case rotationX:
				case rotationY:
					return null;
			}
		}

		Element element = childrenMap.get(property);
		if (element != null) {
			return element;
		}

		String label = EnumLabels.getLabel(property);
		switch (property) {
			case anchorPoint:
			case position:
				element = perCharacter3D ? createAnimatableVec3dElement(property.name(), label, null)
										 : createTiny2DManipVec3dElement(property.name(), label);
				break;

			case scale:
				element = createScaleElement(property.name(), label, "%");
				break;

			case rotationZ:
				if (!perCharacter3D) {
					label = "回転";
				}
				// fall through
			case rotationX:
			case rotationY:
			case skewAxis:
			case fillHue:
			case strokeHue:
				element = createAngleElement(property.name(), label, "°");
				break;

			case opacity:
			case fillSaturation:
			case fillLuminosity:
			case fillOpacity:
			case strokeSaturation:
			case strokeLuminosity:
			case strokeOpacity:
				element = createPropertyElement(property.name(), label, "%");
				break;

			case skew:
			case fillRGB:
			case strokeRGB:
			case strokeWidth:
			case tracking:
			case lineSpacing:
			case characterRange:
			case characterValue:
			case characterOffset:
				element = createPropertyElement(property.name(), label);
				break;

			case blur:
				element = createBlurElement(property.name(), label);
				break;

			default:
				throw new RuntimeException("unknown TAProperty: " + property);
		}
		childrenMap.put(property, element);
		return element;
	}

	private class TAPropertyAnimatableValueElement extends LayerAnimatableValueElement {

		private final String property;

		public TAPropertyAnimatableValueElement(String property, String name, String unit) {
			super(TextAnimatorElement.this, TextAnimatorElement.this.getLayer(), name, unit);
			this.property = property;
		}

		@Override
		protected String getDefaultExpression() {
			return String.format("textAnimator(\"%s\").%s",
					animator.getName().replaceAll("\"", "\\\\\""), property);
		}

		@Override
		public String getProperty() {
			return String.format("textAnimators[%d].%s",
					((TextLayer)layer).getTextAnimators().indexOf(animator), property);
		}
	}

	private LayerAnimatableValueElement createPropertyElement(String property, String name, String unit) {
		return new TAPropertyAnimatableValueElement(property, name, unit);
	}

	private LayerAnimatableValueElement createPropertyElement(String property, String name) {
		return createPropertyElement(property, name, null);
	}

	private LayerAnimatableValueElement createTiny2DManipVec3dElement(String property, String name) {
		return new TAPropertyAnimatableValueElement(property, name, null) {
			@Override
			protected AnimatableValueElementDelegate<?> createDelegate() {
				AnimatableVec3d avalue = PropertyUtil.getProperty(layer, getProperty());
				return new Tiny2DManipVec3dElementDelegate(this, name, avalue);
			}
		};
	}

	private LayerAnimatableValueElement createAnimatableVec3dElement(String property, String name, String unit) {
		return new TAPropertyAnimatableValueElement(property, name, unit) {
			@Override
			protected AnimatableValueElementDelegate<?> createDelegate() {
				AnimatableVec3d avalue = PropertyUtil.getProperty(layer, getProperty());
				return new AnimatableVec3dElementDelegate(this, name, avalue, perCharacter3D, unit);
			}
		};
	}

	private LayerAnimatableValueElement createScaleElement(String property, String name, String unit) {
		return new TAPropertyAnimatableValueElement(property, name, unit) {
			@Override
			protected AnimatableValueElementDelegate<?> createDelegate() {
				AnimatableVec3d avalue = PropertyUtil.getProperty(layer, getProperty());
				return new ScaleElementDelegate(this, name, avalue, perCharacter3D, unit);
			}
		};
	}

	private LayerAnimatableValueElement createAngleElement(String property, String name, String unit) {
		return new TAPropertyAnimatableValueElement(property, name, unit) {
			@Override
			protected AnimatableValueElementDelegate<?> createDelegate() {
				AnimatableDouble avalue = PropertyUtil.getProperty(layer, getProperty());
				return new AngleElementDelegate(this, name, avalue, unit);
			}
		};
	}

	private LayerAnimatableValueElement createBlurElement(String property, String name) {
		return new TAPropertyAnimatableValueElement(property, name, null) {
			@Override
			protected AnimatableValueElementDelegate<?> createDelegate() {
				AnimatableVec2d avalue = PropertyUtil.getProperty(layer, getProperty());
				return new XYLinkableVec2dElementDelegate(this, name, avalue, 1, 1, 4, unit);
			}
		};
	}

	private void prepareChildren() {
		children.clear();

		TextLayer layer = getLayer();
		boolean currentPerCharacter3D = LayerNature.isThreeD(layer) && layer.isPerCharacter3D();
		if (perCharacter3D != currentPerCharacter3D) {
			childrenMap.remove(TAProperty.anchorPoint);
			childrenMap.remove(TAProperty.position);
			childrenMap.remove(TAProperty.scale);
			childrenMap.remove(TAProperty.rotationX);
			childrenMap.remove(TAProperty.rotationY);
			childrenMap.remove(TAProperty.rotationZ);
			perCharacter3D = currentPerCharacter3D;
		}

		for (TASelector selector : animator.getSelectors()) {
			children.add(getSelectorElement(selector));
		}

		for (TAProperty property : TAProperty.values()) {
			if (animator.getProperties().contains(property)) {
				Element element = getPropertyElement(property);
				if (element != null) {
					children.add(element);
				}
			}
		}

		childrenMap.values().retainAll(children);
	}


	private static final ImageRegistry imageRegistry = Activator.getDefault().getImageRegistry();

	private class EnableSwitch implements Switch {

		public boolean hasBorder() {
			return true;
		}

		public Image getImage() {
			return animator.isEnabled() ? imageRegistry.get(Activator.IMG_SWITCH_VIDEO) : null;
		}

		public void mouseDown(MouseEvent event) {
			boolean currentEnabled = animator.isEnabled();
			String label = currentEnabled ? "テキストアニメータを無効にする" : "テキストアニメータを有効にする";

			ProjectManager pm = (ProjectManager) viewer.getData(LayerCompositionView.PROJECT_MANAGER);
			TextLayer layer = getLayer();
			String property = String.format("textAnimators[%d].enabled", layer.getTextAnimators().indexOf(animator));
			pm.postOperation(new ModifyLayerPropertyOperation<Boolean>(
					pm, layer, property, !currentEnabled, label));
		}
	}

}
