/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.distort;

import java.nio.FloatBuffer;
import java.util.HashSet;
import java.util.Set;

import javax.media.opengl.GLUniformData;

import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IAnimatableEnum;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.IVideoBuffer.TextureFilter;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.annotations.ShaderSource;
import ch.kuramo.javie.api.annotations.Effect.Categories;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.api.services.IVideoRenderSupport;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.PolarCoordinates", category=Categories.DISTORT)
public class PolarCoordinates {

	@ShaderSource
	public static final String[] RECT_TO_POLAR = {
		"uniform sampler2D texture;",
		"uniform vec2 size;",
		"uniform vec2 center;",
		"uniform float radians_per_px;",
		"uniform float interp;",
		"",
		"void main(void)",
		"{",
		"	vec2 coordRect = gl_TexCoord[0].st * size;",
		"	vec2 fromCenter = coordRect - center;",
		"",
		"	vec2 coordPolar = vec2(",
		"			atan(fromCenter.x, fromCenter.y) / radians_per_px + center.x,",
		"			length(fromCenter) * 2.0);",
		"",
		"	gl_FragColor = texture2D(texture, mix(coordRect, coordPolar, interp) / size);",
		"}"
	};

	@ShaderSource
	public static final String[] POLAR_TO_RECT = {
		"uniform sampler2D texture;",
		"uniform vec2 size;",
		"uniform vec2 center;",
		"uniform float radians_per_px;",
		"uniform float maxRadius_div_h;",
		"uniform float interp;",
		"",
		"void main(void)",
		"{",
		"	vec2 coordPolar = gl_TexCoord[0].st * size;",
		"",
		"	float theta = (coordPolar.x - center.x) * radians_per_px;",
		"	vec2 coordRect = maxRadius_div_h * coordPolar.y * vec2(cos(theta), sin(theta)) + center;",
		"",
		"	gl_FragColor = texture2D(texture, mix(coordPolar, coordRect, interp) / size);",
		"}"
	};


	public enum TypeOfConversion { RECT_TO_POLAR, POLAR_TO_RECT }


	private final IVideoEffectContext context;

	private final IVideoRenderSupport support;

	private final IShaderProgram rectToPolarProgram;

	private final IShaderProgram polarToRectProgram;

	@Property(min="0", max="100")
	private IAnimatableDouble interpolation;

	@Property
	private IAnimatableEnum<TypeOfConversion> typeOfConversion;


	@Inject
	public PolarCoordinates(IVideoEffectContext context, IVideoRenderSupport support, IShaderRegistry shaders) {
		this.context = context;
		this.support = support;
		rectToPolarProgram = shaders.getProgram(PolarCoordinates.class, "RECT_TO_POLAR");
		polarToRectProgram = shaders.getProgram(PolarCoordinates.class, "POLAR_TO_RECT");
	}

	public IVideoBuffer doVideoEffect() {
		IVideoBuffer input = context.doPreviousEffect();

		VideoBounds bounds = input.getBounds();
		if (bounds.isEmpty()) {
			return input;
		}
		
		double interp = context.value(this.interpolation);
		if (interp == 0) {
			return input;
		}

		TypeOfConversion type = context.value(this.typeOfConversion);
		IShaderProgram program = (type == TypeOfConversion.RECT_TO_POLAR)
									? rectToPolarProgram : polarToRectProgram;

		int w = bounds.width;
		int h = bounds.height;

		try {
			Set<GLUniformData> uniforms = new HashSet<GLUniformData>();
			uniforms.add(new GLUniformData("texture", 0));
			uniforms.add(new GLUniformData("size", 2, FloatBuffer.wrap(new float[] { w, h })));
			uniforms.add(new GLUniformData("center", 2, FloatBuffer.wrap(new float[] { w/2f, h/2f })));
			uniforms.add(new GLUniformData("radians_per_px", (float)Math.toRadians(360.0/w)));
			if (type == TypeOfConversion.POLAR_TO_RECT) {
				uniforms.add(new GLUniformData("maxRadius_div_h", (float)(Math.sqrt(w*w+h*h)*0.5/h)));
			}
			uniforms.add(new GLUniformData("interp", (float)(interp/100)));

			// 終わったら削除してしまうので、元に戻す必要はない。
			input.setTextureFilter(TextureFilter.LINEAR);

			return support.useShaderProgram(program, uniforms, null, input);

		} finally {
			input.dispose();
		}
	}

}
