/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.colorCorrection;

import java.util.HashSet;
import java.util.Set;

import javax.media.opengl.GLUniformData;

import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.annotations.ShaderSource;
import ch.kuramo.javie.api.annotations.Effect.Categories;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.api.services.IVideoRenderSupport;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.BrightnessAndContrast", category=Categories.COLOR_CORRECTION)
public class BrightnessAndContrast {

	@ShaderSource
	public static final String[] BRIGHTNESS_AND_CONTRAST = {
		"uniform sampler2D texture;",
		"uniform float brightness;",
		"uniform float contrastSlope;",
		"",
		"void main(void)",
		"{",
		"	vec4 color = texture2D(texture, gl_TexCoord[0].st);",
		"	if (color.a != 0.0) {",
		"		vec3 rgb = color.rgb/color.a;",
		"		rgb = (rgb+brightness-0.5)*contrastSlope+0.5;",
		"		gl_FragColor = vec4(rgb*color.a, color.a);",
		"	} else {",
		"		gl_FragColor = vec4(0.0);",
		"	}",
		"}"
	};


	@Property(min="-255", max="255")
	private IAnimatableDouble brightness;

	@Property(min="-100", max="100")
	private IAnimatableDouble contrast;


	private final IVideoEffectContext context;

	private final IVideoRenderSupport support;

	private final IShaderProgram program;

	@Inject
	public BrightnessAndContrast(IVideoEffectContext context, IVideoRenderSupport support, IShaderRegistry shaders) {
		this.context = context;
		this.support = support;
		program = shaders.getProgram(BrightnessAndContrast.class, "BRIGHTNESS_AND_CONTRAST");
	}

	public IVideoBuffer doVideoEffect() {
		IVideoBuffer input = context.doPreviousEffect();

		VideoBounds bounds = input.getBounds();
		if (bounds.isEmpty()) {
			return input;
		}

		double brightness = context.value(this.brightness);
		double contrast = context.value(this.contrast);

		if (brightness == 0 && contrast == 0) {
			return input;
		}

		contrast = Math.min(contrast, 100-(1e-10));

		try {
			Set<GLUniformData> uniforms = new HashSet<GLUniformData>();
			uniforms.add(new GLUniformData("texture", 0));
			uniforms.add(new GLUniformData("brightness", (float)(brightness/255)));
			uniforms.add(new GLUniformData("contrastSlope", (float)(contrast >= 0 ? 100/(100-contrast) : (100+contrast)/100)));

			return support.useShaderProgram(program, uniforms, null, input);

		} finally {
			input.dispose();
		}
	}

}
