/*
 * Copyright (c) 2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.expression;

import java.util.LinkedList;
import java.util.Map;

import org.mozilla.javascript.Context;
import org.mozilla.javascript.NativeArray;
import org.mozilla.javascript.Wrapper;

import ch.kuramo.javie.api.Color;
import ch.kuramo.javie.api.Vec2d;
import ch.kuramo.javie.api.Vec3d;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.expression.ColorProperty;
import ch.kuramo.javie.core.expression.Vec2dProperty;
import ch.kuramo.javie.core.expression.Vec3dProperty;

public class ExpressionUtils {

	private static final String THREAD_LOCAL_STACK = ExpressionUtils.class.getName() + ".THREAD_LOCAL_STACK";

	public static void pushThreadLocals(Context context, Object ... keys) {
		@SuppressWarnings("unchecked")
		LinkedList<Map<Object, Object>> stack = (LinkedList<Map<Object, Object>>)
											context.getThreadLocal(THREAD_LOCAL_STACK);

		if (stack == null) {
			stack = Util.newLinkedList();
			context.putThreadLocal(THREAD_LOCAL_STACK, stack);
		}

		Map<Object, Object> map = Util.newMap();
		for (Object key : keys) {
			map.put(key, context.getThreadLocal(key));
		}

		stack.add(map);
	}

	public static void popThreadLocals(Context context) {
		@SuppressWarnings("unchecked")
		LinkedList<Map<Object, Object>> stack = (LinkedList<Map<Object, Object>>)
											context.getThreadLocal(THREAD_LOCAL_STACK);

		if (stack == null) {
			throw new IllegalStateException();
		}

		Map<Object, Object> map = stack.removeLast();
		for (Map.Entry<Object, Object> e : map.entrySet()) {
			context.putThreadLocal(e.getKey(), e.getValue());
		}
	}

	public static Object toDoubleOrDoubleArray(Object jsValue) {
		if (jsValue instanceof NativeArray) {
			return Context.jsToJava(jsValue, double[].class);
		}

		if (jsValue instanceof Wrapper) {
			Object javaObj = ((Wrapper) jsValue).unwrap();

			if (javaObj instanceof Vec3dProperty) {
				javaObj = ((Vec3dProperty) javaObj).getValue();
			} else if (javaObj instanceof Vec2dProperty) {
				javaObj = ((Vec2dProperty) javaObj).getValue();
			} else if (javaObj instanceof ColorProperty) {
				javaObj = ((ColorProperty) javaObj).getValue();
			}

			if (javaObj instanceof Vec3d) {
				Vec3d v = (Vec3d) javaObj;
				return new double[] { v.x, v.y, v.z };

			} else if (javaObj instanceof Vec2d) {
				Vec2d v = (Vec2d) javaObj;
				return new double[] { v.x, v.y };

			} else if (javaObj instanceof Color) {
				Color c = (Color) javaObj;
				return new double[] { c.rawRed, c.rawGreen, c.rawBlue, c.rawAlpha };

			} else if (javaObj instanceof Object[]) {
				Object[] objArray = (Object[]) javaObj;
				double[] dblArray = new double[objArray.length];
				for (int i = 0; i < objArray.length; ++i) {
					dblArray[i] = (Double) Context.jsToJava(objArray[i], double.class);
				}
				return dblArray;
			}

			jsValue = javaObj;
		}

		if (jsValue == null) {
			return null;
		} else if (jsValue instanceof double[]) {
			return jsValue;
		} else {
			return Context.jsToJava(jsValue, double.class);
		}
	}

}
