/*
 * Copyright (c) 2010,2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.project;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.mozilla.javascript.Context;
import org.mozilla.javascript.ContextFactory;
import org.mozilla.javascript.Scriptable;
import org.mozilla.javascript.ScriptableObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ch.kuramo.javie.api.Vec3d;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.app.InjectorHolder;
import ch.kuramo.javie.app.PropertyUtil;
import ch.kuramo.javie.core.AnimatableDouble;
import ch.kuramo.javie.core.AnimatableLayerReference;
import ch.kuramo.javie.core.AnimatableString;
import ch.kuramo.javie.core.AnimatableValue;
import ch.kuramo.javie.core.AnimatableVec3d;
import ch.kuramo.javie.core.CameraLayer;
import ch.kuramo.javie.core.CastsShadows;
import ch.kuramo.javie.core.Effect;
import ch.kuramo.javie.core.EffectableLayer;
import ch.kuramo.javie.core.Keyframe;
import ch.kuramo.javie.core.Layer;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.LayerNature;
import ch.kuramo.javie.core.LightLayer;
import ch.kuramo.javie.core.LightType;
import ch.kuramo.javie.core.MediaInput;
import ch.kuramo.javie.core.MediaLayer;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.ProjectDecodeException;
import ch.kuramo.javie.core.PropertyDescriptor;
import ch.kuramo.javie.core.TextLayer;
import ch.kuramo.javie.core.TransformableLayer;
import ch.kuramo.javie.core.TextLayer.TextType;
import ch.kuramo.javie.core.services.Font;
import ch.kuramo.javie.core.services.FontList;
import ch.kuramo.javie.core.services.ProjectDecoder;
import ch.kuramo.javie.core.services.ProjectEncoder;

import com.google.inject.Inject;

public class ResetPropertiesOperation extends ProjectOperation {

	private static final Logger _logger = LoggerFactory.getLogger(ResetPropertiesOperation.class);


	private final String _compId;

	private final Object[][] _properties;

	private final Object[] _savedValues;

	@Inject
	private ProjectEncoder _encoder;

	@Inject
	private ProjectDecoder _decoder;

	@Inject
	private FontList _fontList;

	@Inject
	private ContextFactory _jsContextFactory;


	public ResetPropertiesOperation(ProjectManager pm, LayerComposition comp, Object[][] properties) {
		super(pm, "リセット");
		InjectorHolder.getInjector().injectMembers(this);

		if (properties.length == 0) {
			_compId = null;
			_properties = null;
			_savedValues = null;
			_noEffect = true;
			return;
		}

		pm.checkComposition(comp);
		_compId = comp.getId();
		_properties = new Object[properties.length][];
		_savedValues = new Object[properties.length];

		for (int i = 0; i < properties.length; ++i) {
			Layer layer = (Layer) properties[i][0];
			Integer effectIndex = (Integer) properties[i][1];
			String property = (String) properties[i][2];

			if (pm.checkLayer(layer) != comp) {
				throw new IllegalArgumentException();
			}

			if (effectIndex == -1) {
				Object obj = PropertyUtil.getProperty(layer, property);
				if (obj instanceof AnimatableValue<?>) {
					_savedValues[i] = _encoder.encodeElement(obj);
				} else {
					_savedValues[i] = obj;
				}
			} else {
				Effect effect = ((EffectableLayer) layer).getEffects().get(effectIndex);
				AnimatableValue<?> avalue = PropertyUtil.getProperty(effect, property);
				_savedValues[i] = _encoder.encodeElement(avalue);
			}

			_properties[i] = new Object[] { layer.getId(), effectIndex, property };
		}
	}

	@Override
	protected IStatus executeOrRedo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		resetOrRestore(true, project, pm);
		return Status.OK_STATUS;
	}

	@Override
	protected IStatus undo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		resetOrRestore(false, project, pm);
		return Status.OK_STATUS;
	}

	private void resetOrRestore(boolean reset, Project project, ProjectManager pm) throws ExecutionException {
		LayerComposition comp = project.getComposition(_compId);
		Object[][] data = (pm != null) ? new Object[_properties.length][] : null;

		for (int i = 0, n = _properties.length; i < n; ++i) {
			Layer layer = comp.getLayer((String) _properties[i][0]);
			Integer effectIndex = (Integer) _properties[i][1];
			String property = (String) _properties[i][2];
			PropertyDescriptor propDesc = null;

			Object obj;
			if (effectIndex == -1) {
				obj = PropertyUtil.getProperty(layer, property);
			} else {
				Effect effect = ((EffectableLayer) layer).getEffects().get(effectIndex);
				propDesc = effect.getEffectDescriptor().getPropertyDescriptor(property);
				obj = propDesc.get(effect);
			}

			if (obj instanceof AnimatableValue<?>) {
				AnimatableValue<?> avalue = (AnimatableValue<?>) obj;
				String expr1 = (pm != null) ? avalue.getExpression() : null;

				if (reset) {
					if (effectIndex == -1) {
						resetLayerProperty(comp, layer, property, avalue);
					} else {
						resetEffectProperty(comp, layer, propDesc, avalue);
					}
				} else {
					restore((String)_savedValues[i], avalue);
				}

				if (pm != null) {
					String expr2 = avalue.getExpression();
					if ((expr1 != null && !expr1.equals(expr2)) || (expr1 == null && expr2 != null)) {
						if (effectIndex == -1) {
							pm.fireLayerExpressionChange(layer, property);
						} else {
							pm.fireEffectExpressionChange((EffectableLayer)layer, effectIndex, property);
						}
					}
				}
			} else {
				if (reset) {
					resetLayerProperty(comp, layer, property, obj);
				} else {
					PropertyUtil.setProperty(layer, property, _savedValues[i]);
				}
			}

			if (pm != null) {
				data[i] = new Object[] { layer, effectIndex, property, null };
			}
		}

		if (pm != null) {
			pm.fireKeyframesChange(comp, data);
		}
	}

	private void resetLayerProperty(
			LayerComposition comp, Layer layer, String property, AnimatableValue<?> avalue) {

		double w = comp.getSize().width;
		double h = comp.getSize().height;

		if (layer instanceof TextLayer && property.equals("font")) {
			Font font = _fontList.defaultFont();
			if (font != null) {
				((AnimatableString) avalue).reset(font.psName);
				return;
			}
		} else if (layer instanceof TransformableLayer && property.equals("position")) {
			((AnimatableVec3d) avalue).reset(new Vec3d(w/2, h/2));
			return;

		} else if (layer instanceof MediaLayer) {
			MediaLayer ml = (MediaLayer) layer;
			VideoBounds bounds = LayerNature.isVideoNature(layer)
					? ml.getMediaInput().getVideoFrameBounds() : null;

			if (bounds != null && property.equals("anchorPoint")) {
				((AnimatableVec3d) avalue).reset(
						new Vec3d(bounds.width/2d + bounds.x, bounds.height/2d + bounds.y));
				return;
			}

		} else if (layer instanceof CameraLayer) {
			double fovx = 39.6;
			double zoom = w / (2 * Math.tan(Math.toRadians(fovx/2)));

			if (property.equals("pointOfInterest")) {
				((AnimatableVec3d) avalue).reset(new Vec3d(w/2, h/2));
				return;
		
			} else if (property.equals("position")) {
				((AnimatableVec3d) avalue).reset(new Vec3d(w/2, h/2, -zoom));
				return;

			} else if (property.equals("zoom")) {
				((AnimatableDouble) avalue).reset(zoom);
				return;
			}

		} else if (layer instanceof LightLayer) {
			if (property.equals("pointOfInterest")) {
				((AnimatableVec3d) avalue).reset(new Vec3d(w/2, h/2));
				return;
		
			} else if (property.equals("position")) {
				((AnimatableVec3d) avalue).reset(new Vec3d(w/2+w/24, h/2-w/24, -w/2.88));
				return;
			}
		}

		avalue.reset();
	}

	private void resetLayerProperty(
			LayerComposition comp, Layer layer, String property, Object value) {

		if (layer instanceof TextLayer && property.equals("textType")) {
			TextLayer tl = (TextLayer) layer;
			tl.setTextType(TextType.FILL_ONLY);
			return;

		} else if (layer instanceof MediaLayer) {
			MediaLayer ml = (MediaLayer) layer;

			if (property.equals("castsShadows")) {
				ml.setCastsShadows(CastsShadows.OFF);
				return;

			} else if (property.equals("acceptsShadows")) {
				ml.setAcceptsShadows(true);
				return;

			} else if (property.equals("acceptsLights")) {
				ml.setAcceptsLights(true);
				return;
			}

		} else if (layer instanceof LightLayer) {
			LightLayer ll = (LightLayer) layer;

			if (property.equals("lightType")) {
				ll.setLightType(LightType.POINT);
				return;

			} else if (property.equals("castsShadows")) {
				ll.setCastsShadows(false);
				return;
			}
		}

		throw new IllegalArgumentException();
	}

	private void resetEffectProperty(
			LayerComposition comp, Layer layer, PropertyDescriptor propDesc, AnimatableValue<?> avalue) {

		resetEffectProperty(comp, layer, propDesc, avalue, _jsContextFactory);
	}

	// TODO ThreeDSwitchOperation 等にほぼ同じメソッドがあるので整理する。
	private <A extends AnimatableValue<V>, V> void restore(String data, A dst) throws ExecutionException {
		try {
			@SuppressWarnings("unchecked")
			A src = _decoder.decodeElement(data, (Class<A>) dst.getClass());

			dst.reset(src.getStaticValue());
			dst.setExpression(src.getExpression());

			for (Keyframe<V> kf : src.getKeyframeMap().values()) {
				dst.putKeyframe(kf);
			}

		} catch (ProjectDecodeException e) {
			throw new ExecutionException("error decoding AnimatableValue data", e);
		}
	}

	public static void resetEffectProperty(
			LayerComposition comp, Layer layer,
			PropertyDescriptor propDesc, AnimatableValue<?> avalue,
			ContextFactory jsContextFactory) {

		String resetScript = propDesc.getString("reset");
		if (resetScript == null) {
			if (avalue instanceof AnimatableLayerReference) {
				((AnimatableLayerReference) avalue).reset(layer.getId());
			} else {
				avalue.reset();
			}
			return;
		}

		VideoBounds videoBounds = null;
		if (LayerNature.isCTCR(layer)) {
			videoBounds = new VideoBounds(comp.getSize());
		} else if (layer instanceof MediaLayer) {
			MediaInput input = ((MediaLayer) layer).getMediaInput();
			// FIXME プレースホルダを実装したので getMediaInput が null を返すことはもう無いんじゃなかったっけ？
			if (input != null) {
				videoBounds = input.getVideoFrameBounds();
			}
		}
		if (videoBounds == null) {
			videoBounds = new VideoBounds(0, 0);
		}

		Context jsContext = jsContextFactory.enterContext();
		try {
			Scriptable globalScope = jsContext.initStandardObjects();
			ScriptableObject.putConstProperty(globalScope, "videoBounds", Context.javaToJS(videoBounds, globalScope));
			Object jsValue = jsContext.evaluateString(globalScope, resetScript, propDesc.getName(), 1, null);

			@SuppressWarnings("unchecked")
			AnimatableValue<Object> tmp = (AnimatableValue<Object>) avalue;
			tmp.reset(tmp.jsToJava(jsValue));
			return;

		} catch (Exception e) {
			_logger.warn("error resetting AnimatableValue", e);
		} finally {
			Context.exit();
		}

		avalue.reset();
	}

}
