/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.project;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;

import ch.kuramo.javie.core.Composition;
import ch.kuramo.javie.core.CompositionItem;
import ch.kuramo.javie.core.Item;
import ch.kuramo.javie.core.ItemLayer;
import ch.kuramo.javie.core.Layer;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.Project;

public class ReplaceLayerItemOperation<T extends Item> extends ProjectOperation {

	private final String compId;

	private final String layerId;

	private final String oldItemId;

	private final String newItemId;


	public ReplaceLayerItemOperation(ProjectManager pm, ItemLayer<T> layer, T newItem) {
		super(pm, "ソースの置き換え");

		LayerComposition comp = pm.checkLayer(layer);
		pm.checkItem(newItem);

		if (newItem instanceof CompositionItem && checkCyclicComposition(comp, (CompositionItem)newItem)) {
			throw new IllegalArgumentException("cyclic composition");
		}

		compId = comp.getId();
		layerId = layer.getId();
		oldItemId = layer.getItem().getId();
		newItemId = newItem.getId();

		_noEffect = newItemId.equals(oldItemId);
	}

	@Override
	protected IStatus executeOrRedo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		setMediaItem(project, pm, newItemId);
		return Status.OK_STATUS;
	}

	@Override
	protected IStatus undo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		setMediaItem(project, pm, oldItemId);
		return Status.OK_STATUS;
	}

	private void setMediaItem(Project project, ProjectManager pm, String itemId) {
		LayerComposition comp = project.getComposition(compId);
		@SuppressWarnings("unchecked")
		ItemLayer<T> layer = (ItemLayer<T>) comp.getLayer(layerId);
		T item = project.getItem(itemId);

		layer.setItem(item);

		if (pm != null) {
			pm.fireLayerItemChange(layer);
		}
	}

	public static boolean checkCyclicComposition(LayerComposition comp, CompositionItem compItem) {
		// TODO NewLayerFromItemOperation に同じメソッドがある。

		Composition otherComp = compItem.getComposition();

		if (otherComp == comp) {
			return true;
		}

		if (otherComp instanceof LayerComposition) {
			for (Layer layer : ((LayerComposition) otherComp).getLayers()) {
				if (layer instanceof ItemLayer<?>) {
					Item item = ((ItemLayer<?>) layer).getItem();
					if (item instanceof CompositionItem
							&& checkCyclicComposition(comp, (CompositionItem) item)) {
						return true;
					}
				}
			}
		}

		// TODO LayerComposition以外のCompositionを実装したときには
		//      そのCompositionがLayerCompositionを含んでいるケースをチェックする必要がある。
		//if (otherComp instanceof TrackComposition) {
		//}

		return false;
	}

}
