/*
 * Copyright (c) 2009-2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.project;

import java.util.Collection;
import java.util.Collections;
import java.util.EventObject;
import java.util.LinkedHashSet;
import java.util.Set;

import ch.kuramo.javie.app.PropertyUtil;
import ch.kuramo.javie.core.AnimatableValue;
import ch.kuramo.javie.core.Composition;
import ch.kuramo.javie.core.Effect;
import ch.kuramo.javie.core.EffectableLayer;
import ch.kuramo.javie.core.Item;
import ch.kuramo.javie.core.ItemLayer;
import ch.kuramo.javie.core.Layer;
import ch.kuramo.javie.core.LayerComposition;

public class ProjectEvent extends EventObject {

	private static final long serialVersionUID = 4412699553814727247L;


	public enum Type {
		PROJECT_INITIALIZE,
		PROJECT_DISPOSE,

		DIRTY_CHANGE,
		FILE_CHANGE,

		ITEM_NAME_CHANGE,
		ITEM_UPDATE,
		ITEMS_ADD,
		ITEMS_REMOVE,
		ITEMS_REPARENT,

		COMPOSITION_SETTINGS_CHANGE,
		COMPOSITION_PROPERTY_CHANGE,

		LAYER_PROPERTY_CHANGE,
		LAYER_ITEM_CHANGE,
		LAYERS_ADD,
		LAYERS_REMOVE,
		LAYERS_REORDER,

		TEXT_ANIMATORS_ADD,
		TEXT_ANIMATORS_REMOVE,
		TA_SELECTORS_ADD,
		TA_SELECTORS_REMOVE,
		TA_PROPERTIES_ADD,
		TA_PROPERTIES_REMOVE,

		EFFECT_PROPERTY_CHANGE,
		EFFECTS_ADD,
		EFFECTS_REMOVE,

		LAYER_EXPRESSION_CHANGE,
		EFFECT_EXPRESSION_CHANGE,
		EXPRESSIONS_ADD,
		EXPRESSIONS_REMOVE,

		LAYER_TIMES_CHANGE,
		KEYFRAMES_CHANGE,
		LAYER_SLIP_EDIT,

		SHADOW_OPERATION_EXECUTION
	}


	public final Type type;

	public final Object data;

	public final Composition composition;

	public final Layer layer;

	public final int effectIndex;

	public final String property;


	protected ProjectEvent(
			ProjectManager pm, Type type, Object data,
			Composition comp, Layer layer, int effectIndex, String property) {

		super(pm);
		this.type = type;
		this.data = data;
		this.composition = comp;
		this.layer = layer;
		this.effectIndex = effectIndex;
		this.property = property;
	}

	public ProjectManager getProjectManager() {
		return (ProjectManager) source;
	}

	public String toString() {
		return String.format(
				"ProjectEvent {type=%s data=%s projectManager=%s composition=%s layer=%s}",
				type, data, getProjectManager(), composition, layer);
	}


	public static ProjectEvent createProjectInitializeEvent(ProjectManager pm) {
		return new ProjectEvent(pm, Type.PROJECT_INITIALIZE, null, null, null, -1, null);
	}

	public static ProjectEvent createProjectDisposeEvent(ProjectManager pm) {
		return new ProjectEvent(pm, Type.PROJECT_DISPOSE, null, null, null, -1, null);
	}

	public static ProjectEvent createDirtyChangeEvent(ProjectManager pm) {
		return new ProjectEvent(pm, Type.DIRTY_CHANGE, null, null, null, -1, null);
	}

	public static ProjectEvent createFileChangeEvent(ProjectManager pm) {
		return new ProjectEvent(pm, Type.FILE_CHANGE, null, null, null, -1, null);
	}

	public static ProjectEvent createItemNameChangeEvent(ProjectManager pm, Item item) {
		pm.checkItem(item);
		return new ProjectEvent(pm, Type.ITEM_NAME_CHANGE, item, null, null, -1, null);
	}

	public static ProjectEvent createItemUpdateEvent(ProjectManager pm, Item item) {
		pm.checkItem(item);
		return new ProjectEvent(pm, Type.ITEM_UPDATE, item, null, null, -1, null);
	}

	private static <T> Set<T> toUnmodifiableSet(Collection<? extends T> c) {
		if (c == null) {
			return Collections.<T>emptySet();
		}

		if (c instanceof Set<?>) {
			@SuppressWarnings("unchecked")
			Set<T> set = (Set<T>) c;
			return Collections.unmodifiableSet(set);
		}

		// 重複を取り除く。この時、元のコレクションでの順序は維持する。
		return Collections.unmodifiableSet(new LinkedHashSet<T>(c));
	}

	private static void checkItems(ProjectManager pm, Collection<? extends Item> items) {
		if (items != null) {
			for (Item i : items) {
				pm.checkItem(i);
			}
		}
	}

	public static ProjectEvent createItemsAddEvent(
			ProjectManager pm, Collection<? extends Item> items, Collection<? extends Item> relatedItems) {

		checkItems(pm, items);
		checkItems(pm, relatedItems);

		Object data = new Set[] { toUnmodifiableSet(items), toUnmodifiableSet(relatedItems) };
		return new ProjectEvent(pm, Type.ITEMS_ADD, data, null, null, -1, null);
	}

	public static ProjectEvent createItemsRemoveEvent(
			ProjectManager pm, Collection<? extends Item> items, Collection<? extends Item> relatedItems) {

		// itemsの方は既にプロジェクトから取り除かれているのでチェックしない
		checkItems(pm, relatedItems);

		Object data = new Set[] { toUnmodifiableSet(items), toUnmodifiableSet(relatedItems) };
		return new ProjectEvent(pm, Type.ITEMS_REMOVE, data, null, null, -1, null);
	}

	public static ProjectEvent createItemsReparentEvent(
			ProjectManager pm, Collection<? extends Item> items, Collection<? extends Item> relatedItems) {

		checkItems(pm, items);
		checkItems(pm, relatedItems);

		Object data = new Set[] { toUnmodifiableSet(items), toUnmodifiableSet(relatedItems) };
		return new ProjectEvent(pm, Type.ITEMS_REPARENT, data, null, null, -1, null);
	}

	public static ProjectEvent createCompositionSettingsChangeEvent(ProjectManager pm, Composition comp) {
		pm.checkComposition(comp);
		return new ProjectEvent(pm, Type.COMPOSITION_SETTINGS_CHANGE, null, comp, null, -1, null);
	}

	public static ProjectEvent createCompositionPropertyChangeEvent(ProjectManager pm, Composition comp, String property) {
		pm.checkComposition(comp);
		return new ProjectEvent(pm, Type.COMPOSITION_PROPERTY_CHANGE, null, comp, null, -1, property);
	}

	public static ProjectEvent createLayerPropertyChangeEvent(ProjectManager pm, Layer layer, String property) {
		return new ProjectEvent(pm, Type.LAYER_PROPERTY_CHANGE, null, pm.checkLayer(layer), layer, -1, property);
	}

	public static ProjectEvent createLayerItemChangeEvent(ProjectManager pm, ItemLayer<?> layer) {
		return new ProjectEvent(pm, Type.LAYER_ITEM_CHANGE, null, pm.checkLayer(layer), layer, -1, null);
	}

	private static void checkLayers(ProjectManager pm, LayerComposition comp, Collection<? extends Layer> layers) {
		for (Layer l : layers) {
			if (pm.checkLayer(l) != comp) {
				throw new IllegalArgumentException("a layer does not belong to the composition");
			}
		}
	}

	public static ProjectEvent createLayersAddEvent(ProjectManager pm, LayerComposition comp, Collection<? extends Layer> layers) {
		pm.checkComposition(comp);
		checkLayers(pm, comp, layers);
		return new ProjectEvent(pm, Type.LAYERS_ADD, toUnmodifiableSet(layers), comp, null, -1, null);
	}

	public static ProjectEvent createLayersRemoveEvent(ProjectManager pm, LayerComposition comp, Collection<? extends Layer> layers) {
		pm.checkComposition(comp);

		// layersのレイヤーは既にcomp内から取り除かれているのでチェックしない

		return new ProjectEvent(pm, Type.LAYERS_REMOVE, toUnmodifiableSet(layers), comp, null, -1, null);
	}

	public static ProjectEvent createLayersReorderEvent(ProjectManager pm, LayerComposition comp, Collection<? extends Layer> layers) {
		pm.checkComposition(comp);
		checkLayers(pm, comp, layers);
		return new ProjectEvent(pm, Type.LAYERS_REORDER, toUnmodifiableSet(layers), comp, null, -1, null);
	}

	public static ProjectEvent createTextAnimatorsAddEvent(ProjectManager pm, LayerComposition comp, Object[][] data) {
		return new ProjectEvent(pm, Type.TEXT_ANIMATORS_ADD, data, comp, null, -1, null);
	}

	public static ProjectEvent createTextAnimatorsRemoveEvent(ProjectManager pm, LayerComposition comp, Object[][] data) {
		return new ProjectEvent(pm, Type.TEXT_ANIMATORS_REMOVE, data, comp, null, -1, null);
	}

	public static ProjectEvent createTASelectorsAddEvent(ProjectManager pm, LayerComposition comp, Object[][] data) {
		return new ProjectEvent(pm, Type.TA_SELECTORS_ADD, data, comp, null, -1, null);
	}

	public static ProjectEvent createTASelectorsRemoveEvent(ProjectManager pm, LayerComposition comp, Object[][] data) {
		return new ProjectEvent(pm, Type.TA_SELECTORS_REMOVE, data, comp, null, -1, null);
	}

	public static ProjectEvent createTAPropertiesAddEvent(ProjectManager pm, LayerComposition comp, Object[][] data) {
		return new ProjectEvent(pm, Type.TA_PROPERTIES_ADD, data, comp, null, -1, null);
	}

	public static ProjectEvent createTAPropertiesRemoveEvent(ProjectManager pm, LayerComposition comp, Object[][] data) {
		return new ProjectEvent(pm, Type.TA_PROPERTIES_REMOVE, data, comp, null, -1, null);
	}

	public static ProjectEvent createEffectPropertyChangeEvent(ProjectManager pm, EffectableLayer layer, int effectIndex, String property) {
		return new ProjectEvent(pm, Type.EFFECT_PROPERTY_CHANGE, null, pm.checkLayer(layer), layer, effectIndex, property);
	}

	public static ProjectEvent createEffectsAddEvent(ProjectManager pm, LayerComposition comp, Object[][] data) {
		return new ProjectEvent(pm, Type.EFFECTS_ADD, data, comp, null, -1, null);
	}

	public static ProjectEvent createEffectsRemoveEvent(ProjectManager pm, LayerComposition comp, Object[][] data) {
		return new ProjectEvent(pm, Type.EFFECTS_REMOVE, data, comp, null, -1, null);
	}

	public static ProjectEvent createLayerExpressionChangeEvent(ProjectManager pm, Layer layer, String property) {
		return new ProjectEvent(pm, Type.LAYER_EXPRESSION_CHANGE, null, pm.checkLayer(layer), layer, -1, property);
	}

	public static ProjectEvent createEffectExpressionChangeEvent(ProjectManager pm, EffectableLayer layer, int effectIndex, String property) {
		return new ProjectEvent(pm, Type.EFFECT_EXPRESSION_CHANGE, null, pm.checkLayer(layer), layer, effectIndex, property);
	}

	public static ProjectEvent createExpressionsAddEvent(ProjectManager pm, LayerComposition comp, Object[][] data) {
		return new ProjectEvent(pm, Type.EXPRESSIONS_ADD, data, comp, null, -1, null);
	}

	public static ProjectEvent createExpressionsRemoveEvent(ProjectManager pm, LayerComposition comp, Object[][] data) {
		return new ProjectEvent(pm, Type.EXPRESSIONS_REMOVE, data, comp, null, -1, null);
	}

	public static ProjectEvent createLayerTimesChangeEvent(ProjectManager pm, LayerComposition comp, Collection<? extends Layer> layers) {
		return new ProjectEvent(pm, Type.LAYER_TIMES_CHANGE, toUnmodifiableSet(layers), comp, null, -1, null);
	}

	public static ProjectEvent createKeyframesChangeEvent(ProjectManager pm, LayerComposition comp, Object[][] data) {
		return new ProjectEvent(pm, Type.KEYFRAMES_CHANGE, data, comp, null, -1, null);
	}

	public static ProjectEvent createLayerSlipEditEvent(ProjectManager pm, LayerComposition comp, Object[] data) {
		return new ProjectEvent(pm, Type.LAYER_SLIP_EDIT, data, comp, null, -1, null);
	}

	public static ProjectEvent createShadowOperationExecutionEvent(ProjectManager pm) {
		return new ProjectEvent(pm, Type.SHADOW_OPERATION_EXECUTION, null, null, null, -1, null);
	}

	public static Item getItem(ProjectEvent event) {
		switch (event.type) {
			case ITEM_NAME_CHANGE:
			case ITEM_UPDATE:
				return (Item) event.data;
			default:
				throw new IllegalArgumentException("event type is neither ITEM_NAME_CHANGE nor ITEM_UPDATE: type=" + event.type);
		}
	}

	private static Set<Item> getItems(ProjectEvent event, int index) {
		switch (event.type) {
			case ITEMS_ADD:
			case ITEMS_REMOVE:
			case ITEMS_REPARENT:
				break;
			default:
				throw new IllegalArgumentException(
						"event type is not ITEMS_ADD, ITEMS_REMOVE nor ITEMS_REPARENT: type=" + event.type);
		}

		@SuppressWarnings("unchecked")
		Set<Item> set = ((Set<Item>[]) event.data)[index];
		return set;
	}

	public static Set<Item> getItems(ProjectEvent event) {
		return getItems(event, 0);
	}

	public static Set<Item> getRelatedItems(ProjectEvent event) {
		return getItems(event, 1);
	}

	public static boolean propertyIsAnimatable(ProjectEvent event) {
		switch (event.type) {
			case LAYER_PROPERTY_CHANGE:
				return (PropertyUtil.getProperty(event.layer, event.property) instanceof AnimatableValue<?>);

			case EFFECT_PROPERTY_CHANGE: {
				EffectableLayer layer = ((EffectableLayer) event.layer);
				Effect effect = layer.getEffects().get(event.effectIndex);
				return (PropertyUtil.getProperty(effect, event.property) instanceof AnimatableValue<?>);
			}

			default:
				throw new IllegalArgumentException("event type is not LAYER_PROPERTY_CHANGE nor EFFECT_PROPERTY_CHANGE: type=" + event.type);
		}
	}

	public static Set<Layer> getLayers(ProjectEvent event) {
		switch (event.type) {
			case LAYERS_ADD:
			case LAYERS_REMOVE:
			case LAYERS_REORDER:
				break;
			default:
				throw new IllegalArgumentException(
						"event type is not LAYERS_ADD, LAYERS_REMOVE nor LAYERS_REORDER: type=" + event.type);
		}

		@SuppressWarnings("unchecked")
		Set<Layer> set = (Set<Layer>) event.data;
		return set;
	}

}
