/*
 * Copyright (c) 2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.project;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;

import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.LayerNature;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.TextLayer;

public class PerCharacter3DOperation extends ProjectOperation {

	public static ProjectOperation toggle(ProjectManager pm, TextLayer layer) {
		boolean threeD = LayerNature.isThreeD(layer);
		boolean perCharacter3D = layer.isPerCharacter3D();

		if (!threeD && perCharacter3D) {
			throw new IllegalStateException();
		}

		PerCharacter3DOperation opPerChar3d = new PerCharacter3DOperation(pm, layer, !perCharacter3D);
		if (perCharacter3D || threeD) {
			return opPerChar3d;
		}

		ThreeDSwitchOperation op3dSwitch = new ThreeDSwitchOperation(pm, layer);
		op3dSwitch.setSyncShadow(true);
		opPerChar3d.setSyncShadow(true);

		ProjectOperationCollection op = new ProjectOperationCollection(pm, opPerChar3d.getLabel());
		op.add(op3dSwitch);
		op.add(opPerChar3d);
		return op;
	}

	public static ProjectOperation disable3D(ProjectManager pm, TextLayer layer) {
		if (!LayerNature.isThreeD(layer)) {
			throw new IllegalStateException();
		}

		ThreeDSwitchOperation op3dSwitch = new ThreeDSwitchOperation(pm, layer);
		if (!layer.isPerCharacter3D()) {
			return op3dSwitch;
		}

		PerCharacter3DOperation opPerChar3d = new PerCharacter3DOperation(pm, layer, false);
		opPerChar3d.setSyncShadow(true);
		op3dSwitch.setSyncShadow(true);

		ProjectOperationCollection op = new ProjectOperationCollection(pm, op3dSwitch.getLabel());
		op.add(opPerChar3d);
		op.add(op3dSwitch);
		return op;
	}


	private final String compId;

	private final String layerId;

	private final boolean perCharacter3D;


	private PerCharacter3DOperation(ProjectManager pm, TextLayer layer, boolean perCharacter3D) {
		super(pm, "文字単位の3D化");

		LayerComposition comp = pm.checkLayer(layer);
		compId = comp.getId();
		layerId = layer.getId();
		this.perCharacter3D = perCharacter3D;

		_noEffect = (layer.isPerCharacter3D() == perCharacter3D);
	}

	@Override
	protected IStatus executeOrRedo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		setPerCharacter3D(project, pm, perCharacter3D);
		return Status.OK_STATUS;
	}

	@Override
	protected IStatus undo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		setPerCharacter3D(project, pm, !perCharacter3D);
		return Status.OK_STATUS;
	}

	private void setPerCharacter3D(Project project, ProjectManager pm, boolean perCharacter3D)
			throws ExecutionException {

		LayerComposition comp = project.getComposition(compId);
		TextLayer layer = (TextLayer) comp.getLayer(layerId);

		if (perCharacter3D && !LayerNature.isThreeD(layer)) {
			throw new ExecutionException("the layer is not 3D");
		}

		layer.setPerCharacter3D(perCharacter3D);

		if (pm != null) {
			pm.fireLayerPropertyChange(layer, "perCharacter3D");
		}
	}

}
