/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.actions;

import java.util.Collections;
import java.util.List;

import org.eclipse.jface.action.Action;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;

import ch.kuramo.javie.app.project.PrecomposeOperation;
import ch.kuramo.javie.app.project.ProjectManager;
import ch.kuramo.javie.app.views.LayerCompositionView;
import ch.kuramo.javie.app.views.layercomp.LayerElement;
import ch.kuramo.javie.app.widgets.GridBuilder;
import ch.kuramo.javie.core.CompositionItem;
import ch.kuramo.javie.core.Layer;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.MediaItemLayer;
import ch.kuramo.javie.core.MediaLayer;
import ch.kuramo.javie.core.Util;

public class PrecomposeAction extends Action {

	private final LayerCompositionView view;

	private final List<Layer> layers = Util.newList();

	public PrecomposeAction(LayerCompositionView view, TreeViewer viewer) {
		super("プリコンポーズ...");
		this.view = view;

		//setId(CommandIds.PRECOMPOSE);
		//setActionDefinitionId(CommandIds.PRECOMPOSE);
		//setImageDescriptor(Activator.getImageDescriptor("/icons/precompose.png"));

		updateState(viewer.getSelection());

		viewer.addSelectionChangedListener(new ISelectionChangedListener() {
			public void selectionChanged(SelectionChangedEvent event) {
				updateState(event.getSelection());
			}
		});
	}

	private void updateState(ISelection selection) {
		List<Layer> list = Util.newList();
		boolean enabled = false;

		IStructuredSelection ss = (IStructuredSelection) selection;
		for (Object obj : ss.toList()) {
			if (obj instanceof LayerElement) {
				Layer layer = ((LayerElement)obj).layer;
				list.add(layer);
				if (layer instanceof MediaLayer) {
					enabled = true;
				}
			}
		}

		layers.clear();
		setEnabled(enabled);

		if (enabled) {
			Collections.reverse(list);
			layers.addAll(list);
		}
	}

	public void run() {
		if (!layers.isEmpty()) {
			ProjectManager pm = ProjectManager.forWorkbenchWindow(view.getSite().getWorkbenchWindow());
			CompositionItem compItem = view.getCompositionItem();

			boolean canLeaveAttrs = (layers.size() == 1 && layers.get(0) instanceof MediaItemLayer);
			String newCompName = (layers.size() == 1) ? (layers.get(0).getName() + " プリコンポ") : "プリコンポ";

			PrecomposeDialog dialog = new PrecomposeDialog(view.getSite().getShell(),
					canLeaveAttrs, compItem.getName(), pm.getUnusedItemName(newCompName));

			if (dialog.open() == IDialogConstants.OK_ID) {
				LayerComposition comp = (LayerComposition) compItem.getComposition();
				pm.postOperation(new PrecomposeOperation(pm, comp, layers,
						dialog.isLeaveAttrs(), dialog.getNewCompositionName()));
			}
		}
	}

}

class PrecomposeDialog extends Dialog {

	private final boolean canLeaveAttrs;

	private boolean leaveAttrs;

	private final String origCompName;

	private String newCompName;

	PrecomposeDialog(
			Shell parentShell, boolean canLeaveAttrs,
			String origCompName, String newCompName) {

		super(parentShell);
		this.canLeaveAttrs = canLeaveAttrs;
		this.leaveAttrs = canLeaveAttrs;
		this.origCompName = origCompName;
		this.newCompName = newCompName;
	}

	protected void configureShell(Shell newShell) {
		super.configureShell(newShell);
		newShell.setText("プリコンポーズ");
	}

	protected Control createDialogArea(Composite parent) {
		Composite composite = (Composite) super.createDialogArea(parent);

		GridBuilder gb = new GridBuilder(composite, 2, false);
		gb.getLayout().verticalSpacing = 20;

								  gb.hSpan(1).hAlign(SWT.RIGHT).label(SWT.NULL, "新規コンポジション名:");
		final Text compNameText = gb.hSpan(1).hAlign(SWT.FILL).width(300).text(SWT.BORDER, newCompName);

		Composite radioGroup = gb.hSpan(2).hAlign(SWT.FILL).composite(SWT.NULL);
		radioGroup.setLayout(new FillLayout());

		GridBuilder gb2 = new GridBuilder(radioGroup, 1, false);
		Button leaveAttrsRadio = gb2.hSpan(1).hAlign(SWT.LEFT).button(SWT.RADIO, String.format("すべての属性を “%s” に残す", origCompName));
		Button moveAttrsRadio = gb2.hSpan(1).hAlign(SWT.LEFT).button(SWT.RADIO, "すべての属性を新規コンポジションに移動する");

		leaveAttrsRadio.setEnabled(canLeaveAttrs);
		leaveAttrsRadio.setSelection(leaveAttrs);
		moveAttrsRadio.setSelection(!leaveAttrs);


		compNameText.addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				newCompName = compNameText.getText().trim();

				Button okButton = getButton(IDialogConstants.OK_ID);
				okButton.setEnabled(newCompName.length() > 0);
			}
		});

		leaveAttrsRadio.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				leaveAttrs = true;
			}
		});

		moveAttrsRadio.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				leaveAttrs = false;
			}
		});

		return composite;
	}

	public void create() {
		super.create();
		Button okButton = getButton(IDialogConstants.OK_ID);
		okButton.setEnabled(newCompName.length() > 0);
	}

	public boolean isLeaveAttrs() {
		return leaveAttrs;
	}

	public String getNewCompositionName() {
		return newCompName;
	}

}
