/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.generate;

import ca.uol.aig.fftpack.RealDoubleFFT;
import ch.kuramo.javie.api.AudioMode;
import ch.kuramo.javie.api.BlendMode;
import ch.kuramo.javie.api.IAnimatableBoolean;
import ch.kuramo.javie.api.IAnimatableColor;
import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IAnimatableEnum;
import ch.kuramo.javie.api.IAnimatableInteger;
import ch.kuramo.javie.api.IAnimatableLayerReference;
import ch.kuramo.javie.api.IAnimatableVec2d;
import ch.kuramo.javie.api.IArray;
import ch.kuramo.javie.api.IAudioBuffer;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.annotations.Effect.Categories;
import ch.kuramo.javie.api.services.IArrayPools;
import ch.kuramo.javie.api.services.IBlendSupport;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.api.services.IVideoRenderSupport;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.AudioSpectrum", category=Categories.GENERATE)
public class AudioSpectrum extends AudioDrawing {

	@Property
	private IAnimatableLayerReference audioLayer;

	@Property
	private IAnimatableVec2d startPoint;

	@Property
	private IAnimatableVec2d endPoint;

	@Property(value="20", min="1", max="48000")
	private IAnimatableDouble startFrequency;

	@Property(value="2000", min="1", max="48000")
	private IAnimatableDouble endFrequency;

	@Property(value="64", min="1", max="4096")
	private IAnimatableInteger frequencyBands;

	@Property(value="120", min="1")
	private IAnimatableDouble height;

	@Property(value="90", min="0", max="30000")
	private IAnimatableDouble audioDuration;

	@Property(value="0", min="-30000", max="30000")
	private IAnimatableDouble audioOffset;

	@Property(value="3", min="0", max="100")
	private IAnimatableDouble thickness;

	@Property("0,1,0")
	private IAnimatableColor color;

	@Property("MONO")
	private IAnimatableEnum<Channel> channel;

	@Property("BARS")
	private IAnimatableEnum<Style> style;

	@Property("true")
	private IAnimatableBoolean smoothing;

	@Property(value="100", min="0", max="100")
	private IAnimatableDouble opacity;

	@Property
	private IAnimatableEnum<BlendMode> blendMode;


	private final IArrayPools arrayPools;

	@Inject
	public AudioSpectrum(
			IVideoEffectContext context, IVideoRenderSupport support,
			IBlendSupport blendSupport, IArrayPools arrayPools) {

		super(context, support, blendSupport);
		this.arrayPools = arrayPools;
	}

	public IVideoBuffer doVideoEffect() {
		DataProvider dp = new DataProvider() {
			private int freqBands;
			private IArray<double[]> bands;
			private double[] bandsArray;

			public void init() {
				freqBands = context.value(frequencyBands);
				bands = arrayPools.getDoubleArray(freqBands);
				bandsArray = bands.getArray();

				double duration = context.value(audioDuration) / 1000;
				double offset = context.value(audioOffset) / 1000;

				AudioMode audioMode = context.getAudioMode();
				Time startOffset = new Time((long)((offset-duration/2)*audioMode.sampleRate), audioMode.sampleRate);
				Time startTime = startOffset.add(context.getTime());
				int audioFrameCount = (int)(duration*audioMode.sampleRate);

				// 使用しているfftの実装は、4の倍数のサンプル数での効率が良いらしい。
				audioFrameCount = (audioFrameCount+3)/4*4;

				context.setTime(startTime);
				context.setAudioFrameCount(audioFrameCount);

				IAudioBuffer audio = null;
				try {
					audio = context.getLayerAudioChunk(audioLayer);
					if (audio == null) {
						bands.clear();
					} else {
						double startFreq = context.value(startFrequency);
						double endFreq = Math.max(startFreq, context.value(endFrequency));
						Channel ch = context.value(channel);
						fft(audio, bands, startFreq, endFreq, ch);
					}
				} finally {
					if (audio != null) audio.dispose();
				}
			}

			public void dispose() {
				if (bands != null) {
					bands.release();
					bands = null;
				}
			}

			public int getDataCount()				{ return freqBands; }
			public float getDataLower(int index)	{ return 0; }
			public float getDataUpper(int index)	{ return (float)-bandsArray[index]; }
			public float getData(int index)			{ return (float)-bandsArray[index]; }
		};

		return draw(dp, startPoint, endPoint, height, thickness,
				color, style, smoothing, blendMode, opacity);
	}

	private void fft(IAudioBuffer audio, IArray<double[]> bands, double startFreq, double endFreq, Channel channel) {
		int n = audio.getFrameCount();

		// TODO このfftオブジェクトはサイズnをキーにしてキャッシュできるはず。
		//      あと、fftオブジェクトの内部で使っている配列をプールから取得するようにすべき。
		RealDoubleFFT fft = new RealDoubleFFT(n);

		IArray<double[]> fftData = null;
		try {
			fftData = arrayPools.getDoubleArray(n);
			double[] fftDataArray = fftData.getArray();

			switch (audio.getAudioMode().dataType) {
				case SHORT: {
					short[] audioArray = (short[]) audio.getData();
					double left = (channel == Channel.RIGHT) ? 0 : 1.0/Short.MAX_VALUE;
					double right = (channel == Channel.LEFT) ? 0 : 1.0/Short.MAX_VALUE;

					for (int i = 0; i < n; ++i) {
						fftDataArray[i] = 0.5*(1-Math.cos(2*Math.PI*i/(n-1)))
								*(left*audioArray[i*2] + right*audioArray[i*2+1]);
					}
					break;
				}

				case INT: {
					int[] audioArray = (int[]) audio.getData();
					double left = (channel == Channel.RIGHT) ? 0 : 1.0/Integer.MAX_VALUE;
					double right = (channel == Channel.LEFT) ? 0 : 1.0/Integer.MAX_VALUE;

					for (int i = 0; i < n; ++i) {
						fftDataArray[i] = 0.5*(1-Math.cos(2*Math.PI*i/(n-1)))
								*(left*audioArray[i*2] + right*audioArray[i*2+1]);
					}
					break;
				}

				case FLOAT: {
					float[] audioArray = (float[]) audio.getData();
					double left = (channel == Channel.RIGHT) ? 0 : 1;
					double right = (channel == Channel.LEFT) ? 0 : 1;

					for (int i = 0; i < n; ++i) {
						fftDataArray[i] = 0.5*(1-Math.cos(2*Math.PI*i/(n-1)))
								*(left*audioArray[i*2] + right*audioArray[i*2+1]);
					}
					break;
				}
			}

			fft.ft(fftDataArray, n);


			bands.clear();
			int numBands = bands.getLength();
			double[] bandsArray = bands.getArray();
			double bandWidth = (endFreq - startFreq) / numBands;
			double baseFreq = (double)audio.getAudioMode().sampleRate / n;

			for (int i = 1; i < n-1; i+=2) {
				double f = (i+1)/2 * baseFreq;
				int j = (int)Math.floor((f - startFreq) / bandWidth);
				if (j >= 0 && j < numBands) {
					double a = fftDataArray[i]/n*2;
					double b = fftDataArray[i+1]/n*2;
					bandsArray[j] += a*a+b*b;
				}
			}
			if ((n%2) == 0) {
				double f = n/2 * baseFreq;
				int j = (int)Math.floor((f - startFreq) / bandWidth);
				if (j >= 0 && j < numBands) {
					double a = fftDataArray[n-1]/n*2;
					bandsArray[j] += a*a;
				}
			}

			for (int i = 0; i < numBands; ++i) {
				bandsArray[i] = 10*Math.sqrt(bandsArray[i]);
			}

		} finally {
			if (fftData != null) fftData.release();
		}
	}

}
