/*
 * Copyright (c) 2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.colorCorrection;

import java.nio.FloatBuffer;
import java.util.HashSet;
import java.util.Set;

import javax.media.opengl.GLUniformData;

import ch.kuramo.javie.api.IAnimatableBoolean;
import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IAnimatableVec3d;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Vec3d;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.annotations.ShaderSource;
import ch.kuramo.javie.api.annotations.Effect.Categories;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.api.services.IVideoRenderSupport;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.ChannelMixer", category=Categories.COLOR_CORRECTION)
public class ChannelMixer {

	@Property(value="100,0,0", min="-200", max="200")
	private IAnimatableVec3d redRGB;

	@Property(value="0", min="-200", max="200")
	private IAnimatableDouble redAlpha;

	@Property(value="0", min="-200", max="200")
	private IAnimatableDouble redConst;


	@Property(value="0,100,0", min="-200", max="200")
	private IAnimatableVec3d greenRGB;

	@Property(value="0", min="-200", max="200")
	private IAnimatableDouble greenAlpha;

	@Property(value="0", min="-200", max="200")
	private IAnimatableDouble greenConst;


	@Property(value="0,0,100", min="-200", max="200")
	private IAnimatableVec3d blueRGB;

	@Property(value="0", min="-200", max="200")
	private IAnimatableDouble blueAlpha;

	@Property(value="0", min="-200", max="200")
	private IAnimatableDouble blueConst;


	@Property(value="0,0,0", min="-200", max="200")
	private IAnimatableVec3d alphaRGB;

	@Property(value="100", min="-200", max="200")
	private IAnimatableDouble alphaAlpha;

	@Property(value="0", min="-200", max="200")
	private IAnimatableDouble alphaConst;


	@Property
	private IAnimatableBoolean monochrome;


	private final IVideoEffectContext context;

	private final IVideoRenderSupport support;

	private final IShaderProgram colorProgram;

	private final IShaderProgram monoProgram;

	@Inject
	public ChannelMixer(IVideoEffectContext context,
			IVideoRenderSupport support, IShaderRegistry shaders) {

		this.context = context;
		this.support = support;
		colorProgram = shaders.getProgram(ChannelMixer.class, "COLOR");
		monoProgram = shaders.getProgram(ChannelMixer.class, "MONO");
	}

	public IVideoBuffer doVideoEffect() {
		IVideoBuffer source = context.doPreviousEffect();
		if (source.getBounds().isEmpty()) {
			return source;
		}

		Vec3d redRGB    = context.value(this.redRGB);
		double redAlpha = context.value(this.redAlpha);
		double redConst = context.value(this.redConst);

		Vec3d greenRGB    = context.value(this.greenRGB);
		double greenAlpha = context.value(this.greenAlpha);
		double greenConst = context.value(this.greenConst);

		Vec3d blueRGB    = context.value(this.blueRGB);
		double blueAlpha = context.value(this.blueAlpha);
		double blueConst = context.value(this.blueConst);

		Vec3d alphaRGB    = context.value(this.alphaRGB);
		double alphaAlpha = context.value(this.alphaAlpha);
		double alphaConst = context.value(this.alphaConst);

		double[] mat = new double[] {
				redRGB.x/100, greenRGB.x/100, blueRGB.x/100, alphaRGB.x/100,
				redRGB.y/100, greenRGB.y/100, blueRGB.y/100, alphaRGB.y/100,
				redRGB.z/100, greenRGB.z/100, blueRGB.z/100, alphaRGB.z/100,
				redAlpha/100, greenAlpha/100, blueAlpha/100, alphaAlpha/100
		};

		double[] cns = new double[] {
				redConst/100, greenConst/100, blueConst/100, alphaConst/100
		};

		try {
			IShaderProgram program = context.value(monochrome) ? monoProgram : colorProgram;

			Set<GLUniformData> uniforms = new HashSet<GLUniformData>();
			uniforms.add(new GLUniformData("texture", 0));
			uniforms.add(new GLUniformData("mat", 4, 4, toFloatBuffer(mat)));
			uniforms.add(new GLUniformData("cns", 4, toFloatBuffer(cns)));

			return support.useShaderProgram(program, uniforms, null, source);

		} finally {
			source.dispose();
		}
	}

	private FloatBuffer toFloatBuffer(double...values) {
		float[] farray = new float[values.length];
		for (int i = 0; i < values.length; ++i) {
			farray[i] = (float)values[i];
		}
		return FloatBuffer.wrap(farray);
	}


	@ShaderSource
	public static final String[] COLOR = createShaderSource(false);

	@ShaderSource
	public static final String[] MONO = createShaderSource(true);


	private static String[] createShaderSource(boolean mono) {
		return new String[] {
		 mono ? "#define MONO" : "",
				"",
				"uniform sampler2D texture;",
				"uniform mat4 mat;",
				"uniform vec4 cns;",
				"",
				"void main(void)",
				"{",
				"	vec4 color = texture2D(texture, gl_TexCoord[0].st);",
				"	if (color.a != 0.0) {",
				"		color.rgb /= color.a;",
				"	}",
				"	color = mat * color + cns;",
				"	color.a = clamp(color.a, 0.0, 1.0);",
				"",
				"#ifdef MONO",
				"	color.rgb = vec3(color.r * color.a);",
				"#else",
				"	color.rgb *= color.a;",
				"#endif",
				"",
				"	gl_FragColor = color;",
				"}"
		};
	}

}
