/*
 * Copyright (c) 2009-2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.TreeSet;

import javax.media.opengl.GL2;
import javax.media.opengl.GLUniformData;
import javax.media.opengl.glu.GLU;
import javax.vecmath.AxisAngle4d;
import javax.vecmath.Matrix3d;
import javax.vecmath.Matrix4d;
import javax.vecmath.Point2d;
import javax.vecmath.Point3d;
import javax.vecmath.Vector3d;
import javax.vecmath.Vector4d;

import net.arnx.jsonic.JSONHint;
import ch.kuramo.javie.api.Color;
import ch.kuramo.javie.api.GeometryInputType;
import ch.kuramo.javie.api.GeometryOutputType;
import ch.kuramo.javie.api.IAudioBuffer;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Quality;
import ch.kuramo.javie.api.Resolution;
import ch.kuramo.javie.api.ShaderType;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.api.Vec2d;
import ch.kuramo.javie.api.Vec3d;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.IVideoBuffer.TextureFilter;
import ch.kuramo.javie.api.annotations.GeometryShaderParameters;
import ch.kuramo.javie.api.annotations.ShaderSource;
import ch.kuramo.javie.api.services.IAntiAliasSupport;
import ch.kuramo.javie.api.services.IBlurSupport;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoRenderSupport;
import ch.kuramo.javie.api.services.IBlurSupport.BlurDimensions;
import ch.kuramo.javie.core.AbstractAnimatableEnum;
import ch.kuramo.javie.core.AnimatableColor;
import ch.kuramo.javie.core.AnimatableDouble;
import ch.kuramo.javie.core.AnimatableInteger;
import ch.kuramo.javie.core.AnimatableString;
import ch.kuramo.javie.core.AnimatableVec2d;
import ch.kuramo.javie.core.CoreContext;
import ch.kuramo.javie.core.Effect;
import ch.kuramo.javie.core.ExpressionScope;
import ch.kuramo.javie.core.Keyframe;
import ch.kuramo.javie.core.LayerNature;
import ch.kuramo.javie.core.MediaInput;
import ch.kuramo.javie.core.MediaLayer;
import ch.kuramo.javie.core.TAProperty;
import ch.kuramo.javie.core.TASelector;
import ch.kuramo.javie.core.TextAnimator;
import ch.kuramo.javie.core.TextLayer;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.VectorMediaInput;
import ch.kuramo.javie.core.VideoLayerRenderer;
import ch.kuramo.javie.core.WrappedOperation;
import ch.kuramo.javie.core.TASelector.Evaluator;
import ch.kuramo.javie.core.TextAnimator.AnchorPointGrouping;
import ch.kuramo.javie.core.TextAnimator.CharacterAlignment;
import ch.kuramo.javie.core.TextAnimator.CharacterRange;
import ch.kuramo.javie.core.VideoLayerRenderer.AbstractCRLayerRenderer;
import ch.kuramo.javie.core.VideoLayerRenderer.AbstractNormalLayerRenderer;
import ch.kuramo.javie.core.VideoLayerRenderer.CRLayerRenderer;
import ch.kuramo.javie.core.VideoLayerRenderer.NormalLayerRenderer;
import ch.kuramo.javie.core.annotations.ProjectElement;
import ch.kuramo.javie.core.internal.TextAnimatorImpl.AnimatableAnchorPointGrouping;
import ch.kuramo.javie.core.internal.TextAnimatorImpl.AnimatableCharacterAlignment;
import ch.kuramo.javie.core.services.Font;
import ch.kuramo.javie.core.services.FontList;
import ch.kuramo.javie.core.services.FontManager;
import ch.kuramo.javie.core.services.GLGlobal;
import ch.kuramo.javie.core.services.RenderContext;
import ch.kuramo.javie.core.services.VideoEffectPipeline;
import ch.kuramo.javie.core.services.VideoRenderSupport;

import com.google.inject.Inject;

import ftgl.FTGL;
import ftgl.FTGLfont;

@ProjectElement("textLayer")
public class TextLayerImpl extends AbstractMediaLayer implements TextLayer {

	private final TextInput _input = new TextInput();

	private boolean _videoEnabled = true;

	private boolean _ctcr;

	private AnimatableString _sourceText = new AnimatableString("テキスト");

	private AnimatableString _font = new AnimatableString("");

	private AnimatableInteger _fontSize = new AnimatableInteger(36, 1, Integer.MAX_VALUE);

	private TextType _textType = TextType.FILL_ONLY;

	private AnimatableColor _fillColor = new AnimatableColor(Color.WHITE);

	private AnimatableColor _strokeColor = new AnimatableColor(new Color(1, 0, 0));

	private AnimatableDouble _strokeWidth = new AnimatableDouble(1.0, 0.0, 1296.0);		// AEは0.1-1296の範囲。

	private AnimatableLineJoin _lineJoin = new AnimatableLineJoin(LineJoin.MITER);

	private AnimatableDouble _miterLimit = new AnimatableDouble(10d, 1d, Double.POSITIVE_INFINITY);

	private AnimatableOrderOfFillAndStroke _orderOfFillAndStroke = new AnimatableOrderOfFillAndStroke(OrderOfFillAndStroke.STROKE_OVER_FILL);

	private AnimatableHorizontalAlignment _horizontalAlignment = new AnimatableHorizontalAlignment(HorizontalAlignment.LEFT);

	private AnimatableDouble _tracking = new AnimatableDouble(0.0, -1000.0, 1000.0);

	private AnimatableDouble _leading = new AnimatableDouble(-1.0);

	private AnimatableAnchorPointGrouping _anchorPointGrouping = new AnimatableAnchorPointGrouping(AnchorPointGrouping.CHARACTER);

	private AnimatableVec2d _groupingAlignment = new AnimatableVec2d(Vec2d.ZERO);

	private AnimatableCharacterAlignment _characterAlignment = new AnimatableCharacterAlignment(CharacterAlignment.CENTER);

	private List<TextAnimator> _textAnimators = Util.newList();

	private boolean _perCharacter3D;


	private final RenderContext _context;

	private final IVideoRenderSupport _support;

	private final VideoRenderSupport _oldSupport;

	private final VideoEffectPipeline _vePipeline;

	private final IAntiAliasSupport _aaSupport;

	private final IBlurSupport _blurSupport;

	private final FontList _fontList;

	private final FontManager _fontManager;

	private final IShaderProgram[] _strokePrograms;

	private final boolean _geometryShaderAvailable;


	@Inject
	public TextLayerImpl(
			RenderContext context, IVideoRenderSupport support, VideoRenderSupport oldSupport,
			VideoEffectPipeline vePipeline, IAntiAliasSupport aaSupport, IBlurSupport blurSupport,
			FontList fontList, FontManager fontManager, IShaderRegistry shaders, GLGlobal glGlobal) {

		super();
		_context = context;
		_support = support;
		_oldSupport = oldSupport;
		_vePipeline = vePipeline;
		_aaSupport = aaSupport;
		_blurSupport = blurSupport;
		_fontList = fontList;
		_fontManager = fontManager;

		_strokePrograms = new IShaderProgram[] {
				shaders.getProgram(TextLayerImpl.class, "MITER"),
				shaders.getProgram(TextLayerImpl.class, "ROUND"),
				shaders.getProgram(TextLayerImpl.class, "BEVEL")
		};

		_geometryShaderAvailable = glGlobal.getExtensions().contains("GL_EXT_geometry_shader4")
								|| glGlobal.getExtensions().contains("GL_ARB_geometry_shader4");
	}

	public boolean isVideoEnabled() {
		return _videoEnabled;
	}

	public void setVideoEnabled(boolean enabled) {
		_videoEnabled = enabled;
	}

	@JSONHint(ignore=true)
	public boolean isAudioEnabled() {
		return false;
	}

	@JSONHint(ignore=true)
	public void setAudioEnabled(boolean enabled) {
		throw new UnsupportedOperationException();
	}

	@JSONHint(ignore=true)
	public MediaInput getMediaInput() {
		return _input;
	}

	public boolean isCTCR() {
		return _ctcr;
	}

	public void setCTCR(boolean ctcr) {
		_ctcr = ctcr;
	}

	public AnimatableString getSourceText() {
		return _sourceText;
	}

	public void setSourceText(AnimatableString sourceText) {
		sourceText.copyConfigurationFrom(_sourceText);
		_sourceText = sourceText;
	}

	public AnimatableString getFont() {
		return _font;
	}

	public void setFont(AnimatableString font) {
		font.copyConfigurationFrom(_font);
		_font = font;
	}

	public AnimatableInteger getFontSize() {
		return _fontSize;
	}

	public void setFontSize(AnimatableInteger fontSize) {
		fontSize.copyConfigurationFrom(_fontSize);
		_fontSize = fontSize;
	}

	public TextType getTextType() {
		return _textType;
	}

	public void setTextType(TextType textType) {
		_textType = textType;
	}

	public AnimatableColor getFillColor() {
		return _fillColor;
	}

	public void setFillColor(AnimatableColor fillColor) {
		fillColor.copyConfigurationFrom(_fillColor);
		_fillColor = fillColor;
	}

	public AnimatableColor getStrokeColor() {
		return _strokeColor;
	}

	public void setStrokeColor(AnimatableColor strokeColor) {
		strokeColor.copyConfigurationFrom(_strokeColor);
		_strokeColor = strokeColor;
	}

	public AnimatableDouble getStrokeWidth() {
		return _strokeWidth;
	}

	public void setStrokeWidth(AnimatableDouble strokeWidth) {
		strokeWidth.copyConfigurationFrom(_strokeWidth);
		_strokeWidth = strokeWidth;
	}

	public AnimatableLineJoin getLineJoin() {
		return _lineJoin;
	}

	public void setLineJoin(AnimatableLineJoin lineJoin) {
		lineJoin.copyConfigurationFrom(_lineJoin);
		_lineJoin = lineJoin;
	}

	public AnimatableDouble getMiterLimit() {
		return _miterLimit;
	}

	public void setMiterLimit(AnimatableDouble miterLimit) {
		miterLimit.copyConfigurationFrom(_miterLimit);
		_miterLimit = miterLimit;
	}

	public AnimatableOrderOfFillAndStroke getOrderOfFillAndStroke() {
		return _orderOfFillAndStroke;
	}

	public void setOrderOfFillAndStroke(AnimatableOrderOfFillAndStroke orderOfFillAndStroke) {
		orderOfFillAndStroke.copyConfigurationFrom(_orderOfFillAndStroke);
		_orderOfFillAndStroke = orderOfFillAndStroke;
	}

	public AnimatableHorizontalAlignment getHorizontalAlignment() {
		return _horizontalAlignment;
	}

	public void setHorizontalAlignment(
			AnimatableHorizontalAlignment horizontalAlignment) {
		horizontalAlignment.copyConfigurationFrom(_horizontalAlignment);
		_horizontalAlignment = horizontalAlignment;
	}

	public AnimatableDouble getTracking() {
		return _tracking;
	}

	public void setTracking(AnimatableDouble tracking) {
		tracking.copyConfigurationFrom(_tracking);
		_tracking = tracking;
	}

	public AnimatableDouble getLeading() {
		return _leading;
	}

	public void setLeading(AnimatableDouble leading) {
		leading.copyConfigurationFrom(_leading);
		_leading = leading;
	}

	public AnimatableAnchorPointGrouping getAnchorPointGrouping() {
		return _anchorPointGrouping;
	}

	public void setAnchorPointGrouping(AnimatableAnchorPointGrouping anchorPointGrouping) {
		anchorPointGrouping.copyConfigurationFrom(_anchorPointGrouping);
		_anchorPointGrouping = anchorPointGrouping;
	}

	public AnimatableVec2d getGroupingAlignment() {
		return _groupingAlignment;
	}

	public void setGroupingAlignment(AnimatableVec2d groupingAlignment) {
		groupingAlignment.copyConfigurationFrom(_groupingAlignment);
		_groupingAlignment = groupingAlignment;
	}

	public AnimatableCharacterAlignment getCharacterAlignment() {
		return _characterAlignment;
	}

	public void setCharacterAlignment(AnimatableCharacterAlignment characterAlignment) {
		characterAlignment.copyConfigurationFrom(_characterAlignment);
		_characterAlignment = characterAlignment;
	}

	public List<TextAnimator> getTextAnimators() {
		return _textAnimators;
	}

	public void setTextAnimators(List<TextAnimator> textAnimators) {
		_textAnimators = textAnimators;
	}

	public boolean isPerCharacter3D() {
		return _perCharacter3D;
	}

	public void setPerCharacter3D(boolean perCharacter3D) {
		_perCharacter3D = perCharacter3D;
	}

	@Override
	public void prepareExpression(ExpressionScope scope) {
		super.prepareExpression(scope);

		scope.assignTo(_sourceText);
		scope.assignTo(_font);
		scope.assignTo(_fontSize);
		scope.assignTo(_fillColor);
		scope.assignTo(_strokeColor);
		scope.assignTo(_strokeWidth);
		scope.assignTo(_lineJoin);
		scope.assignTo(_miterLimit);
		scope.assignTo(_orderOfFillAndStroke);
		scope.assignTo(_horizontalAlignment);
		scope.assignTo(_tracking);
		scope.assignTo(_leading);
		scope.assignTo(_anchorPointGrouping);
		scope.assignTo(_groupingAlignment);
		scope.assignTo(_characterAlignment);

		for (TextAnimator animator : _textAnimators) {
			// scopeは同じ。
			animator.prepareExpression(scope);
		}
	}

	@Override
	public Object createExpressionElement(CoreContext context) {
		return new TextLayerExpressionElement(context);
	}

	public class TextLayerExpressionElement extends MediaLayerExpressionElement {

		public TextLayerExpressionElement(CoreContext context) {
			super(context);
		}

		public Object getSourceText()			{ return elem(_sourceText); }
		public Object getFont()					{ return elem(_font); }
		public Object getFontSize()				{ return elem(_fontSize); }
		public String getTextType()				{ return _textType.name(); }
		public Object getFillColor()			{ return elem(_fillColor); }
		public Object getStrokeColor()			{ return elem(_strokeColor); }
		public Object getStrokeWidth()			{ return elem(_strokeWidth); }
		public Object getLineJoin()				{ return elem(_lineJoin); }
		public Object getMiterLimit()			{ return elem(_miterLimit); }
		public Object getOrderOfFillAndStroke()	{ return elem(_orderOfFillAndStroke); }
		public Object getHorizontalAlignment()	{ return elem(_horizontalAlignment); }
		public Object getTracking()				{ return elem(_tracking); }
		public Object getLeading()				{ return elem(_leading); }
		public Object getAnchorPointGrouping()	{ return elem(_anchorPointGrouping); }
		public Object getGroupingAlignment()	{ return elem(_groupingAlignment); }
		public Object getCharacterAlignment()	{ return elem(_characterAlignment); }
		public boolean isPerCharacter3D()		{ return _perCharacter3D; }

		public Object getTextAnimators() {
			Object[] array = new Object[_textAnimators.size()];
			for (int i = 0; i < array.length; ++i) {
				array[i] = elem(_textAnimators.get(i));
			}
			return array;
		}

		public Object textAnimator(int index) {
			return elem(_textAnimators.get(index - 1));
		}

		public Object textAnimator(String name) {
			for (TextAnimator animator : _textAnimators) {
				if (animator.getName().equals(name)) {
					return elem(animator);
				}
			}
			return null;
		}
	}

	public List<VideoLayerRenderer> createPerCharacter3DRenderers() {
		boolean perCharacter3D = LayerNature.isThreeD(this) && isPerCharacter3D();
		if (!perCharacter3D) {
			throw new IllegalStateException();
		}

		boolean fillOnly = (getTextType() == TextType.FILL_ONLY);

		String psName = _font.value(_context);
		int fontSize = _fontSize.value(_context);
		FTGLfont polygonFont = getFillFont(psName, fontSize, true);
		FTGLfont strokeFont = fillOnly ? null : getStrokeFont(psName, fontSize);

		if (polygonFont == null /*|| (!fillOnly && strokeFont == null)*/) {	// strokeFontが取得できない場合でもとりあえず続行する。
			// ここで返すリストは、Collections.emptyList() を
			// 使ってはいけないはず？（あとで変更される可能性がある？）以下同様。
			return Util.newList();
		}

		String sourceText = _sourceText.value(_context);

		int[] totals = new int[4];
		List<TACharacter> chars = createTACharacters(totals, sourceText, polygonFont);

		if (chars.isEmpty()) {
			return Util.newList();
		}

		FTGLfont bufferFont = null;
		if (fillOnly) {
			bufferFont = getFillFont(psName, fontSize, false);
			if (bufferFont == null) {
				return Util.newList();
			}
		}

		List<VideoLayerRenderer> renderers = Util.newList();

		boolean ctcr = LayerNature.isCTCR(this);
		Map<Time, List<TACharacter>> charsCache = Util.newMap();

		for (int i = 0; i < chars.size(); ++i) {
			TACharacter c = chars.get(i);
			// 空白は文字コード/文字オフセットアニメータの対象外なので除外する。
			if (c.charIndex2 < 0) continue;
			// 次のふたつは文字コード/文字オフセットアニメータで表示可能な文字に
			// 変化する可能性があるので、ここで除外してはいけない。
			//if (!Character.isDefined(c.codePoint)) continue;
			//if (Character.isISOControl(c.codePoint)) continue;

			if (ctcr) {
				renderers.add(createPerCharacter3DRendererCR(
						totals, chars, i, charsCache, bufferFont, polygonFont, strokeFont));
			} else {
				renderers.add(createPerCharacter3DRendererNormal(
						totals, chars, i, charsCache, bufferFont, polygonFont, strokeFont));
			}
		}

		return renderers;
	}

	private NormalLayerRenderer createPerCharacter3DRendererNormal(
			final int[] totals, final List<TACharacter> chars, final int charIndex,
			final Map<Time, List<TACharacter>> charsCache,
			final FTGLfont bufferFont, final FTGLfont polygonFont, final FTGLfont strokeFont) {

		final Resolution resolution = _context.getVideoResolution();

		return new AbstractNormalLayerRenderer() {

			public MediaLayer getLayer() {
				return TextLayerImpl.this;
			}

			public double getOpacity() {
				return TextLayerImpl.this.getOpacity().value(_context) / 100;
			}

			public void multModelViewMatrix(double[] mvMatrix) {
				TACharacter c = pc3d_getChar(totals, chars, charIndex, charsCache, polygonFont, resolution);
				pc3d_multCharMatrix(mvMatrix, c);
			}

			public VideoBounds calcBounds(boolean withEffects) {
				return _vePipeline.getVideoBounds(
						TextLayerImpl.this, getEffects(withEffects), inputBoundsOperation);
			}

			public IVideoBuffer render(boolean withEffects, boolean frameBlendEnabled) {
				return _vePipeline.doVideoEffects(
						TextLayerImpl.this, getEffects(withEffects), inputBoundsOperation,
						createInputBufferOperation(frameBlendEnabled));
			}

			private List<Effect> getEffects(boolean withEffects) {
				return (withEffects && isEffectsEnabled()) ? TextLayerImpl.this.getEffects() : Collections.<Effect>emptyList();
			}

			private final WrappedOperation<VideoBounds> inputBoundsOperation = new WrappedOperation<VideoBounds>() {
				public VideoBounds execute() {
					TACharacter c = pc3d_getChar(totals, chars, charIndex, charsCache, polygonFont, resolution);
					return resolution.scale(bbox2ToVideoBounds(c));
				}
			};

			private WrappedOperation<IVideoBuffer> createInputBufferOperation(boolean frameBlendEnabled) {
				return new WrappedOperation<IVideoBuffer>() {
					public IVideoBuffer execute() {
						TACharacter c = pc3d_getChar(totals, chars, charIndex, charsCache, polygonFont, resolution);
						return pc3d_renderCharNormal(c, bufferFont, polygonFont, strokeFont, resolution);
					}
				};
			}
		};
	}

	private CRLayerRenderer createPerCharacter3DRendererCR(
			final int[] totals, final List<TACharacter> chars, final int charIndex,
			final Map<Time, List<TACharacter>> charsCache,
			final FTGLfont bufferFont, final FTGLfont polygonFont, final FTGLfont strokeFont) {

		final Resolution resolution = _context.getVideoResolution();

		return new AbstractCRLayerRenderer() {

			public MediaLayer getLayer() {
				return TextLayerImpl.this;
			}

			public double getOpacity() {
				return TextLayerImpl.this.getOpacity().value(_context) / 100;
			}

			public void multModelViewMatrix(double[] mvMatrix) {
				TACharacter c = pc3d_getChar(totals, chars, charIndex, charsCache, polygonFont, resolution);
				pc3d_multCharMatrix(mvMatrix, c);
			}

			public VideoBounds calcBounds(boolean withEffects, VideoBounds viewport) {
				return _vePipeline.getVideoBounds(
						TextLayerImpl.this, getEffects(withEffects),
						createInputBoundsOperation(viewport));
			}

			public IVideoBuffer render(boolean withEffects, VideoBounds viewport, double[] prjMatrix, double[] mvMatrix) {
				return _vePipeline.doVideoEffects(
						TextLayerImpl.this, getEffects(withEffects),
						createInputBoundsOperation(viewport),
						createInputBufferOperation(viewport, prjMatrix, mvMatrix));
			}

			public IVideoBuffer render() {
				return _context.saveAndExecute(new WrappedOperation<IVideoBuffer>() {
					public IVideoBuffer execute() {
						TACharacter c = pc3d_getChar(totals, chars, charIndex, charsCache, polygonFont, resolution);
						return pc3d_renderCharNormal(c, bufferFont, polygonFont, strokeFont, resolution);
					}
				});
			}

			private List<Effect> getEffects(boolean withEffects) {
				return (withEffects && isEffectsEnabled()) ? TextLayerImpl.this.getEffects() : Collections.<Effect>emptyList();
			}

			private WrappedOperation<VideoBounds> createInputBoundsOperation(final VideoBounds viewport) {
				return new WrappedOperation<VideoBounds>() {
					public VideoBounds execute() {
						return viewport;
					}
				};
			}

			private WrappedOperation<IVideoBuffer> createInputBufferOperation(
					final VideoBounds bounds, final double[] prjMatrix, final double[] mvMatrix) {

				return new WrappedOperation<IVideoBuffer>() {
					public IVideoBuffer execute() {
						IVideoBuffer buffer = null;
						try {
							buffer = _support.createVideoBuffer(bounds);
							buffer.clear();

							TACharacter c = pc3d_getChar(totals, chars, charIndex, charsCache, polygonFont, resolution);

							TAContext ctx = new TAContext();
							ctx.viewport = bounds;
							ctx.prjMatrix = prjMatrix;
							ctx.mvMatrix = mvMatrix;
							ctx.resolution = resolution;
							ctx.chars = Collections.singletonList(c);
							ctx.perCharacter3D = true;
							ctx.polygonFont = true;
							ctx.fillFont = polygonFont;
							ctx.strokeFont = strokeFont;

							TextLayerImpl.this.render(ctx, buffer);

							IVideoBuffer result = buffer;
							buffer = null;
							return result;

						} finally {
							if (buffer != null) buffer.dispose();
						}
					}
				};
			}
		};
	}

	private TACharacter pc3d_getChar(
			int[] totals, List<TACharacter> chars, int charIndex,
			Map<Time, List<TACharacter>> charsCache,
			FTGLfont fillFont, Resolution resolution) {

		Time mediaTime = calcVideoMediaTime(getMediaInput());

		List<TACharacter> chars2 = charsCache.get(mediaTime);
		if (chars2 == null) {
			chars2 = new ArrayList<TACharacter>(chars.size());
			for (TACharacter c : chars) {
				chars2.add(c.clone());
			}

			initTACharacters(totals, chars2, true, fillFont, mediaTime);

			for (TACharacter c : chars2) {
				calcCharMatrix(c, true, resolution);
			}

			charsCache.put(mediaTime, chars2);
		}
		return chars2.get(charIndex);
	}

	private void pc3d_multCharMatrix(double[] mvMatrix, TACharacter c) {
		_oldSupport.setMatrix(null, mvMatrix);
		LayerMatrixUtil.multModelViewMatrix(TextLayerImpl.this, _context, _oldSupport);
		_oldSupport.getMatrix(null, mvMatrix);

		Matrix4d m = new Matrix4d(mvMatrix);
		m.transpose();
		m.mul(c.matrix);

		for (int i = 0; i < 16; ++i) {
			mvMatrix[i] = m.getElement(i%4, i/4);
		}
	}

	private IVideoBuffer pc3d_renderCharNormal(
			TACharacter c, FTGLfont bufferFont, FTGLfont polygonFont,
			FTGLfont strokeFont, Resolution resolution) {

		IVideoBuffer buffer = null;
		try {
			VideoBounds bounds = resolution.scale(bbox2ToVideoBounds(c));

			buffer = _support.createVideoBuffer(bounds);
			buffer.clear();

			TAContext ctx = new TAContext();
			ctx.viewport = bounds;
			ctx.resolution = resolution;
			ctx.chars = Collections.singletonList(c);
			ctx.perCharacter3D = true;
			ctx.polygonFont = (bufferFont == null);
			ctx.fillFont = (bufferFont != null) ? bufferFont : polygonFont;
			ctx.strokeFont = strokeFont;

			TextLayerImpl.this.render(ctx, buffer);

			IVideoBuffer result = buffer;
			buffer = null;
			return result;

		} finally {
			if (buffer != null) buffer.dispose();
		}
	}


	@ShaderSource(type=ShaderType.VERTEX_SHADER, program=false)
	public static final String[] line_join_vs = {
		"void main(void)",
		"{",
		"	gl_Position = gl_Vertex;",
		"	gl_FrontColor = gl_Color;",
		"}"
	};

	@ShaderSource(type=ShaderType.FRAGMENT_SHADER, program=false)
	public static final String[] line_join_fs = {
		"void main(void)",
		"{",
		"	gl_FragColor = gl_Color;",
		"	gl_FragDepth = 0.0;",
		"}"
	};

	@ShaderSource(type=ShaderType.GEOMETRY_SHADER, attach={"line_join_vs", "line_join_fs"})
	@GeometryShaderParameters(inputType=GeometryInputType.LINES_ADJACENCY,
			outputType=GeometryOutputType.TRIANGLE_STRIP, verticesOut=8)
	public static final String[] MITER = createLineJoinGeometryShader(LineJoin.MITER);

	@ShaderSource(type=ShaderType.GEOMETRY_SHADER, attach={"line_join_vs", "line_join_fs"})
	@GeometryShaderParameters(inputType=GeometryInputType.LINES_ADJACENCY,
			outputType=GeometryOutputType.TRIANGLE_STRIP, verticesOut=58)
	public static final String[] ROUND = createLineJoinGeometryShader(LineJoin.ROUND);

	@ShaderSource(type=ShaderType.GEOMETRY_SHADER, attach={"line_join_vs", "line_join_fs"})
	@GeometryShaderParameters(inputType=GeometryInputType.LINES_ADJACENCY,
			outputType=GeometryOutputType.TRIANGLE_STRIP, verticesOut=7)
	public static final String[] BEVEL = createLineJoinGeometryShader(LineJoin.BEVEL);

	private static String[] createLineJoinGeometryShader(LineJoin lineJoin) {
		return new String[] {
				"#version 120",
				"#extension GL_EXT_geometry_shader4 : enable",
				"",
				"#define " + lineJoin.name(),
				"",
				"uniform float halfWidth;",
				"#ifdef MITER",
				"uniform float miterLimit;",
				"#endif",
				"",
				"void vertex(vec3 v)",
				"{",
				"	gl_Position = gl_ModelViewProjectionMatrix * vec4(v, 1.0);",
				"	gl_FrontColor = gl_FrontColorIn[1];",	// [2]から生成された頂点がもあるが、
				"	EmitVertex();",							// 入力は全部同じ色なので、全部[1]にしてしまう。
				"}",
				"",
				"void main(void)",
				"{",
				"	vec3 p0 = gl_PositionIn[0].xyz;",
				"	vec3 p1 = gl_PositionIn[1].xyz;",
				"	vec3 p2 = gl_PositionIn[2].xyz;",
				"	vec3 p3 = gl_PositionIn[3].xyz;",
				"",
				"	vec3 v1 = p1 - p0;",
				"	vec3 v2 = p2 - p1;",
				"",
				"	vec3 c = cross(v1, v2);",
				"	vec3 v4;",
				"	if (length(c) > 0.0) {",
				"		vec3 v3 = normalize(cross(v1, c));",
				"		v4 = normalize(cross(v2, c));",
				"",
				"#ifdef ROUND",
				"		vec3 v5 = v3;",
				"		float r = acos(dot(v3, v4));",
				"		float s = -sign(c.z);",
				"		for (float t = 0.175; t < r; t += 0.175) {",
				"			vertex(p1+halfWidth*v5);",
				"			vertex(p1);",
				"",
				"			float cos = cos(t);",
				"			float sin = sin(s*t);",
				"			vec3 v6 = mat3(cos,-sin,0,sin,cos,0,0,0,1)*v3;",
				"			vertex(p1+halfWidth*v6);",
				"",
				"			EndPrimitive();",
				"			v5 = v6;",
				"		}",
				"		vertex(p1+halfWidth*v5);",
				"		vertex(p1);",
				"		vertex(p1+halfWidth*v4);",
				"		EndPrimitive();",
				"",
				"#else",
				"		vertex(p1+halfWidth*v3);",
				"#ifdef MITER",
				"		if (dot(normalize(-v1), normalize(v2)) <= miterLimit) {",
				"			vec3 v5 = normalize(v3+v4);",
				"			vertex(p1+halfWidth*v5/dot(v3, v5));",
				"		}",
				"#endif",
				"		vertex(p1);",
				"		vertex(p1+halfWidth*v4);",
				"#endif",
				"		EndPrimitive();",
				"",
				"	} else {",
				"		c = vec3(0.0, 0.0, 1.0);",
				"		v4 = normalize(cross(v2, c));",
				"	}",
				"",
				"	vertex(p1+halfWidth*v4);",
				"	vertex(p1-halfWidth*v4);",
				"	vertex(p2+halfWidth*v4);",
				"	vertex(p2-halfWidth*v4);",
				"	EndPrimitive();",
				"}"
		};
	}

	private void render(final TAContext ctx, IVideoBuffer buffer) {
		final TextType textType = getTextType();
		ctx.lineJoin = (textType != TextType.FILL_ONLY)
				? _lineJoin.value(_context) : null;
		ctx.miterLimit = (ctx.lineJoin == LineJoin.MITER)
				? Math.cos(2*Math.asin(1/_miterLimit.value(_context))) : 0;
		final OrderOfFillAndStroke fasOrder = (textType == TextType.FILL_AND_STROKE)
				? _orderOfFillAndStroke.value(_context) : null;

		final boolean depthTest;
		if (textType != TextType.FILL_ONLY && ctx.strokeFont != null) {
			boolean dt = false;
			for (TACharacter c : ctx.chars) {
				double opacity = c.strokeOpacity * c.opacity;
				if (c.strokeColor != null && c.strokeWidth > 0
						&& opacity != 0 && opacity != 1) {		// opacityが0の場合、描画する必要がない
					dt = true;									// opacityが1の場合、デプステストは不要。
					break;
				}
			}
			depthTest = dt;
		} else {
			depthTest = false;
		}

		try {
			switch (textType) {
				case FILL_ONLY:
					createBlurredImages(ctx, true, false);
					break;

				case STROKE_ONLY:
					createBlurredImages(ctx, false, true);
					break;

				case FILL_AND_STROKE:
					createBlurredImages(ctx, true, true);
					break;
			}

			Runnable operation = new Runnable() {
				public void run() {
					switch (textType) {
						case FILL_ONLY:
							renderCharacters(ctx, true, false, false);
							return;

						case STROKE_ONLY:
							renderCharacters(ctx, false, true, false);
							return;

						case FILL_AND_STROKE:
							switch (fasOrder) {
								case STROKE_OVER_FILL:
									renderCharacters(ctx, true, true, false);
									break;

								case FILL_OVER_STROKE:
									renderCharacters(ctx, false, true, true);
									break;

								case ALL_STROKES_OVER_ALL_FILLS:
									renderCharacters(ctx, true, false, false);
									renderCharacters(ctx, false, true, false);
									break;

								case ALL_FILLS_OVER_ALL_STROKES:
									renderCharacters(ctx, false, true, false);
									renderCharacters(ctx, true, false, false);
									break;
							}
							break;
					}
				}
			};

			if (ctx.polygonFont) {
				final VideoBounds bounds = buffer.getBounds();
				final Runnable innerOp = operation;
				operation = new Runnable() {
					public void run() {
						_aaSupport.antiAlias(bounds.width, bounds.height,
								depthTest, Color.COLORLESS_TRANSPARENT, innerOp);
					}
				};
			}

			int pushAttribs = GL2.GL_ENABLE_BIT | GL2.GL_COLOR_BUFFER_BIT | GL2.GL_CURRENT_BIT
							| GL2.GL_DEPTH_BUFFER_BIT | GL2.GL_TEXTURE_BIT;
			_support.useFramebuffer(operation, pushAttribs, buffer);

		} finally {
			for (TACharacter c : ctx.chars) {
				if (c.blurredFill != null) {
					c.blurredFill.dispose();
					c.blurredFill = null;
				}
				if (c.blurredStroke != null) {
					c.blurredStroke.dispose();
					c.blurredStroke = null;
				}
			}
		}
	}

	private void createBlurredImages(TAContext ctx, boolean fill, boolean stroke) {
		for (TACharacter c : ctx.chars) {
			if (c.blurX > 0 || c.blurY > 0) {
				if (ctx.prjMatrix == null) {
					if (fill && c.fillColor != null
							&& c.fillOpacity > 0 && c.opacity > 0) {

						createBlurredImageNormal(ctx, c, true);
					}

					if (_geometryShaderAvailable
							&& stroke && ctx.strokeFont != null
							&& c.strokeColor != null && c.strokeWidth > 0
							&& c.strokeOpacity > 0 && c.opacity > 0) {

						createBlurredImageNormal(ctx, c, false);
					}

				} else {
					clipCharBBox2(ctx, c);
					if (c.bbox2cp != null) {
						if (fill && c.fillColor != null
								&& c.fillOpacity > 0 && c.opacity > 0) {

							createBlurredImageCR(ctx, c, true);
						}

						if (_geometryShaderAvailable
								&& stroke && ctx.strokeFont != null
								&& c.strokeColor != null && c.strokeWidth > 0
								&& c.strokeOpacity > 0 && c.opacity > 0) {

							createBlurredImageCR(ctx, c, false);
						}
					}
				}
			}
		}
	}

	private void createBlurredImageNormal(final TAContext ctx, final TACharacter c, boolean fill) {
		if (fill && c.blurredFill != null) {
			throw new IllegalStateException();
		}
		if (!fill && c.blurredStroke != null) {
			throw new IllegalStateException();
		}

		final VideoBounds bounds;
		if (ctx.perCharacter3D) {
			bounds = ctx.viewport;
		} else {
			bounds = bbox2ToVideoBounds(c);
		}

		IVideoBuffer buffer = null;
		try {
			buffer = _support.createVideoBuffer(bounds);
			// _aaSupport.antiAlias(...) でクリアされるので、ここでのクリアは不要。
			//buffer.clear();

			Runnable operation;

			if (fill) {
				operation = new Runnable() {
					public void run() {
						setupMatrix(bounds, null, null, null, ctx.resolution);
						fillCharacter(c, ctx.fillFont);
					}
				};
				if (ctx.polygonFont) {
					final Runnable innerOp = operation;
					operation = new Runnable() {
						public void run() {
							_aaSupport.antiAlias(bounds.width, bounds.height, innerOp);
						}
					};
				} else {
					buffer.clear();
				}
			} else {
				operation = new Runnable() {
					public void run() {
						_aaSupport.antiAlias(bounds.width, bounds.height,
								c.strokeOpacity*c.opacity < 1, Color.COLORLESS_TRANSPARENT, new Runnable() {
							public void run() {
								setupMatrix(bounds, null, null, null, ctx.resolution);
								strokeCharacter(c, ctx.strokeFont, ctx.lineJoin, ctx.miterLimit);
							}
						});
					}
				};
			}

			int pushAttribs = GL2.GL_ENABLE_BIT | GL2.GL_COLOR_BUFFER_BIT | GL2.GL_CURRENT_BIT
							| (fill ? 0 : GL2.GL_DEPTH_BUFFER_BIT);
			_support.useFramebuffer(operation, pushAttribs, buffer);

			if (fill) {
				c.blurredFill = blur(buffer, ctx.resolution.scale(c.blurX), ctx.resolution.scale(c.blurY));
			} else {
				c.blurredStroke = blur(buffer, ctx.resolution.scale(c.blurX), ctx.resolution.scale(c.blurY));
			}

		} finally {
			if (buffer != null) buffer.dispose();
		}
	}

	private IVideoBuffer blur(IVideoBuffer buffer, double blurX, double blurY) {
		if (blurX <= 0 && blurY <= 0) {
			throw new IllegalArgumentException();
		}

		if (blurX == blurY) {
			return _blurSupport.gaussianBlur(buffer, blurX, BlurDimensions.BOTH, true, true);

		} else {
			IVideoBuffer tmp = null;
			try {
				if (blurX > 0) {
					tmp = _blurSupport.gaussianBlur(buffer, blurX, BlurDimensions.HORIZONTAL, true, true);
				} else {
					tmp = buffer;
				}

				if (blurY > 0) {
					return _blurSupport.gaussianBlur(tmp, blurY, BlurDimensions.VERTICAL, true, true);
				} else {
					IVideoBuffer result = tmp;
					tmp = null;
					return result;
				}
			} finally {
				if (tmp != null && tmp != buffer) tmp.dispose();
			}
		}
	}

	private void createBlurredImageCR(final TAContext ctx, final TACharacter c, boolean fill) {
		if (fill && c.blurredFill != null) {
			throw new IllegalStateException();
		}
		if (!fill && c.blurredStroke != null) {
			throw new IllegalStateException();
		}

		double tmpScale = 0;
		for (int i = 0, n = c.bbox2c.length; i < n; ++i) {
			for (int j = i+1; j < n; ++j) {
				tmpScale = Math.max(tmpScale, c.bbox2cp[i].distance(c.bbox2cp[j])
											/ c.bbox2c[i].distance(c.bbox2c[j]));
			}
		}
		tmpScale = Math.min(tmpScale, (LayerNature.getQuality(this) == Quality.BEST) ? 2 : 1
									* Math.max(ctx.viewport.width, ctx.viewport.height)
									/ Math.max(c.bbox2[2]-c.bbox2[0], c.bbox2[1]-c.bbox2[3]));

		final double scale = tmpScale;

		int x = (int)Math.floor(c.bbox2[0]*scale);
		int y = (int)Math.floor(c.bbox2[3]*scale);
		int width = (int)Math.ceil(c.bbox2[2]*scale - x);
		int height = (int)Math.ceil(c.bbox2[1]*scale - y);

		final VideoBounds scaledBounds = new VideoBounds(x, y, width, height);

		IVideoBuffer buf1 = null;
		IVideoBuffer buf2 = null;
		IVideoBuffer buf3 = null;
		IVideoBuffer buf4 = null;
		try {
			buf1 = _support.createVideoBuffer(scaledBounds);
			// _aaSupport.antiAlias(...) でクリアされるので、ここでのクリアは不要。
			//buf1.clear();

			Runnable operation;

			if (fill) {
				operation = new Runnable() {
					public void run() {
						_aaSupport.antiAlias(scaledBounds.width, scaledBounds.height, new Runnable() {
							public void run() {
								setupMatrix(scaledBounds, null, null, null, scale);
								fillCharacter(c, ctx.fillFont);
							}
						});
					}
				};
			} else {
				operation = new Runnable() {
					public void run() {
						_aaSupport.antiAlias(scaledBounds.width, scaledBounds.height,
								c.strokeOpacity*c.opacity < 1, Color.COLORLESS_TRANSPARENT, new Runnable() {
							public void run() {
								setupMatrix(scaledBounds, null, null, null, scale);
								strokeCharacter(c, ctx.strokeFont, ctx.lineJoin, ctx.miterLimit);
							}
						});
					}
				};
			}

			int pushAttribs = GL2.GL_ENABLE_BIT | GL2.GL_COLOR_BUFFER_BIT | GL2.GL_CURRENT_BIT
							| (fill ? 0 : GL2.GL_DEPTH_BUFFER_BIT);
			_support.useFramebuffer(operation, pushAttribs, buf1);


			// scale倍したブラーの量が上限6250(BlurSupportImplが処理可能な最大値)を
			// 超える場合は6250に制限する。しかし、実際にはその前にテクスチャサイズの上限に達するはず。
			double blurX = Math.min(6250, scale*c.blurX);
			double blurY = Math.min(6250, scale*c.blurY);
			buf2 = blur(buf1, blurX, blurY);

			if (LayerNature.getQuality(this) == Quality.DRAFT) {
				buf2.setTextureFilter(TextureFilter.NEAREST);
			} else {
				buf2.setTextureFilter(TextureFilter.MIPMAP);
			}


			operation = new Runnable() {
				public void run() {
					GL2 gl = _context.getGL().getGL2();
					gl.glViewport(0, 0, ctx.viewport.width, ctx.viewport.height);
					gl.glMatrixMode(GL2.GL_PROJECTION);
					gl.glLoadMatrixd(ctx.prjMatrix, 0);
					gl.glMatrixMode(GL2.GL_MODELVIEW);
					gl.glLoadMatrixd(ctx.mvMatrix, 0);

					if (!ctx.perCharacter3D) {
						double[] matrix = new double[] {
								c.matrix.m00, c.matrix.m10, c.matrix.m20, c.matrix.m30,
								c.matrix.m01, c.matrix.m11, c.matrix.m21, c.matrix.m31,
								c.matrix.m02, c.matrix.m12, c.matrix.m22, c.matrix.m32,
								c.matrix.m03, c.matrix.m13, c.matrix.m23, c.matrix.m33
						};
						gl.glMultMatrixd(matrix, 0);
					}

					float rslScalef = (float)ctx.resolution.scale;
					gl.glScalef(rslScalef, rslScalef, rslScalef);

					gl.glClearColor(0, 0, 0, 0);
					gl.glClear(GL2.GL_COLOR_BUFFER_BIT);

					gl.glBegin(GL2.GL_POLYGON);
					for (Point3d pt : c.bbox2c) {
						double tx = (pt.x*scale-scaledBounds.x)/scaledBounds.width;
						double ty = (pt.y*scale-scaledBounds.y)/scaledBounds.height;
						gl.glTexCoord2f((float)tx, (float)ty);
						gl.glVertex2f((float)pt.x, (float)pt.y);
					}
					gl.glEnd();
				}
			};

			buf3 = _support.createVideoBuffer(ctx.viewport);

			pushAttribs = GL2.GL_COLOR_BUFFER_BIT;
			_support.useFramebuffer(operation, pushAttribs, buf3, buf2);


			buf4 = _support.createVideoBuffer(c.bbox4);
			_support.copy(buf3, buf4);


			if (fill) {
				c.blurredFill = buf4;
			} else {
				c.blurredStroke = buf4;
			}
			buf4 = null;

		} finally {
			if (buf1 != null) buf1.dispose();
			if (buf2 != null) buf2.dispose();
			if (buf3 != null) buf3.dispose();
			if (buf4 != null) buf4.dispose();
		}
	}

	private void renderCharacters(TAContext ctx, boolean fill, boolean stroke, boolean fill2) {
		for (TACharacter c : ctx.chars) {
			// perCharacter3Dの場合、c.matrix は既に mvMatrix に組み込まれている。
			Matrix4d charMatrix = ctx.perCharacter3D ? null : c.matrix;

			if (fill && c.fillColor != null
					&& c.fillOpacity > 0 && c.opacity > 0) {

				if (c.blurredFill == null) {
					setupMatrix(ctx.viewport, ctx.prjMatrix, ctx.mvMatrix, charMatrix, ctx.resolution);
					fillCharacter(c, ctx.fillFont);
				} else {
					setupMatrix(ctx.viewport, null, null, (ctx.prjMatrix == null) ? charMatrix : null, Resolution.FULL);
					composeBlurredImage(ctx, c.blurredFill);
				}
			}

			if (_geometryShaderAvailable
					&& stroke && ctx.strokeFont != null
					&& c.strokeColor != null && c.strokeWidth > 0
					&& c.strokeOpacity > 0 && c.opacity > 0) {

				if (c.blurredStroke == null) {
					setupMatrix(ctx.viewport, ctx.prjMatrix, ctx.mvMatrix, charMatrix, ctx.resolution);
					strokeCharacter(c, ctx.strokeFont, ctx.lineJoin, ctx.miterLimit);
				} else {
					setupMatrix(ctx.viewport, null, null, (ctx.prjMatrix == null) ? charMatrix : null, Resolution.FULL);
					composeBlurredImage(ctx, c.blurredStroke);
				}
			}

			if (fill2 && c.fillColor != null
					&& c.fillOpacity > 0 && c.opacity > 0) {

				if (c.blurredFill == null) {
					setupMatrix(ctx.viewport, ctx.prjMatrix, ctx.mvMatrix, charMatrix, ctx.resolution);
					fillCharacter(c, ctx.fillFont);
				} else {
					setupMatrix(ctx.viewport, null, null, (ctx.prjMatrix == null) ? charMatrix : null, Resolution.FULL);
					composeBlurredImage(ctx, c.blurredFill);
				}
			}
		}
	}

	private void setupMatrix(
			VideoBounds viewport, double[] prjMatrix, double[] mvMatrix,
			Matrix4d charMatrix, Resolution resolution) {

		setupMatrix(viewport, prjMatrix, mvMatrix, charMatrix, resolution.scale);
	}

	private void setupMatrix(
			VideoBounds viewport, double[] prjMatrix, double[] mvMatrix,
			Matrix4d charMatrix, double extraScaling) {

		GL2 gl = _context.getGL().getGL2();

		if (prjMatrix == null) {
			_support.ortho2D(viewport);
			gl.glMatrixMode(GL2.GL_MODELVIEW);
		} else {
			gl.glViewport(0, 0, viewport.width, viewport.height);
			gl.glMatrixMode(GL2.GL_PROJECTION);
			gl.glLoadMatrixd(prjMatrix, 0);
			gl.glMatrixMode(GL2.GL_MODELVIEW);
			gl.glLoadMatrixd(mvMatrix, 0);
		}

		if (charMatrix != null) {
			double[] matrix = new double[] {
					charMatrix.m00, charMatrix.m10, charMatrix.m20, charMatrix.m30,
					charMatrix.m01, charMatrix.m11, charMatrix.m21, charMatrix.m31,
					charMatrix.m02, charMatrix.m12, charMatrix.m22, charMatrix.m32,
					charMatrix.m03, charMatrix.m13, charMatrix.m23, charMatrix.m33
			};
			gl.glMultMatrixd(matrix, 0);
		}

		float scale = (float) extraScaling;
		gl.glScalef(scale, scale, scale);
	}

	private void fillCharacter(TACharacter c, FTGLfont fillFont) {
		GL2 gl = _context.getGL().getGL2();

		float a = (float)(c.fillOpacity * c.opacity);
		float r = (float) c.fillColor.r;
		float g = (float) c.fillColor.g;
		float b = (float) c.fillColor.b;
		gl.glColor4f(r, g, b, a);

		// オリジナルのFTGLは内部でアルファブレンドの設定を行っているが、
		// Javieはそれを無効にした修正版FTGLを使用している。
		// JavieにはFTGLの設定とは異なるアルファブレンドが必要なため。
		gl.glEnable(GL2.GL_BLEND);
		gl.glBlendFuncSeparate(
				GL2.GL_SRC_ALPHA, GL2.GL_ONE_MINUS_SRC_ALPHA,
				GL2.GL_ONE, GL2.GL_ONE_MINUS_SRC_ALPHA);

		FTGL.ftglRenderFont(fillFont, c.string, FTGL.FTGL_RENDER_ALL);

		gl.glDisable(GL2.GL_BLEND);
	}

	private void strokeCharacter(
			final TACharacter c, final FTGLfont strokeFont, 
			final LineJoin lineJoin, final double miterLimit) {

		final IShaderProgram program = _strokePrograms[lineJoin.ordinal()];
		if (program == null) return;

		final GL2 gl = _context.getGL().getGL2();

		float a = (float)(c.strokeOpacity * c.opacity);
		float r = (float) c.strokeColor.r;
		float g = (float) c.strokeColor.g;
		float b = (float) c.strokeColor.b;
		gl.glColor4f(r, g, b, a);

		if (a < 1) {
			// 線が重なる部分が一度だけ描画されるようにするためにデプスバッファを使用する。
			gl.glEnable(GL2.GL_DEPTH_TEST);
			gl.glClearDepthf(1);
			gl.glClear(GL2.GL_DEPTH_BUFFER_BIT);
		}

		gl.glEnable(GL2.GL_BLEND);
		gl.glBlendFuncSeparate(
				GL2.GL_SRC_ALPHA, GL2.GL_ONE_MINUS_SRC_ALPHA,
				GL2.GL_ONE, GL2.GL_ONE_MINUS_SRC_ALPHA);

		program.useProgram(new Runnable() {
			public void run() {
				GLUniformData data = new GLUniformData("halfWidth", (float)(c.strokeWidth*0.5));
				data.setLocation(program.getUniformLocation(data.getName()));
				gl.glUniform(data);

				if (lineJoin == LineJoin.MITER) {
					data = new GLUniformData("miterLimit", (float)miterLimit);
					data.setLocation(program.getUniformLocation(data.getName()));
					gl.glUniform(data);
				}

				FTGL.ftglRenderFont(strokeFont, c.string, FTGL.FTGL_RENDER_ALL);
			}
		});

		gl.glDisable(GL2.GL_BLEND);
		gl.glDisable(GL2.GL_DEPTH_TEST);
	}

	private void composeBlurredImage(TAContext ctx, IVideoBuffer buffer) {
		// ブラー用のバッファは使い終わったら不要になるので、
		// フィルタの設定を元に戻す必要はない。
		Quality quality = (ctx.resolution.scale < 1) ? Quality.DRAFT : LayerNature.getQuality(this);
		switch (quality) {
			case BEST:
				if (ctx.prjMatrix == null && !ctx.perCharacter3D) {
					buffer.setTextureFilter(TextureFilter.MIPMAP);
				} else {
					buffer.setTextureFilter(TextureFilter.LINEAR);
				}
				break;
			case NORMAL:
				buffer.setTextureFilter(TextureFilter.LINEAR);
				break;
			case DRAFT:
				buffer.setTextureFilter(TextureFilter.NEAREST);
				break;
		}

		GL2 gl = _context.getGL().getGL2();

		gl.glColor4f(1, 1, 1, 1);

		gl.glEnable(GL2.GL_BLEND);
		gl.glBlendFuncSeparate(
				GL2.GL_ONE, GL2.GL_ONE_MINUS_SRC_ALPHA,
				GL2.GL_ONE, GL2.GL_ONE_MINUS_SRC_ALPHA);

		gl.glEnable(GL2.GL_TEXTURE_2D);
		gl.glBindTexture(GL2.GL_TEXTURE_2D, buffer.getTexture());

		VideoBounds b = buffer.getBounds();
		_support.quad2D(b, new double[][] {{0,0},{1,0},{1,1},{0,1}});

		gl.glBindTexture(GL2.GL_TEXTURE_2D, 0);
		gl.glDisable(GL2.GL_TEXTURE_2D);
		gl.glDisable(GL2.GL_BLEND);
	}

	private FTGLfont getFillFont(String psName, int fontSize, boolean polygonFont) {
		Font font = _fontList.get(psName);
		if (font == null && (font = _fontList.defaultFont()) == null) {
			return null;
		}

		String fontPath = font.fontFile.getAbsolutePath();
		int faceIndex = font.faceIndex;

		if (polygonFont) {
			return _fontManager.getPolygonFont(fontPath, faceIndex, fontSize);
		} else {
			return _fontManager.getBufferFont(fontPath, faceIndex, fontSize);
		}
	}

	private FTGLfont getStrokeFont(String psName, int fontSize) {
		Font font = _fontList.get(psName);
		if (font == null && (font = _fontList.defaultFont()) == null) {
			return null;
		}

		String fontPath = font.fontFile.getAbsolutePath();
		int faceIndex = font.faceIndex;

		return _fontManager.getOutlineFont(fontPath, faceIndex, fontSize);
	}

	private VideoBounds calcBounds(List<TACharacter> chars, Resolution resolution) {
		boolean empty = true;
		double left, top, right, bottom;
		left = top = Double.POSITIVE_INFINITY;
		right = bottom = Double.NEGATIVE_INFINITY;

		for (TACharacter c : chars) {
			calcCharMatrix(c, false, resolution);
			calcCharBBox3(c, resolution);

			left   = Math.min(c.bbox3[0], left);
			top    = Math.min(c.bbox3[1], top);
			right  = Math.max(c.bbox3[2], right);
			bottom = Math.max(c.bbox3[3], bottom);

			empty = false;
		}

		if (empty) {
			return new VideoBounds(0, 0);
		}

		// leftとtopが整数値でない場合、レイヤーの画質スイッチに応じた補間が
		// レイヤー合成の段階で行われてしまうので、ぼけた感じになってしまう。
		// そのため、leftとtopを整数値に切り捨てている。
		left = Math.floor(left);
		top = Math.floor(top);

		int width = (int)Math.ceil(right-left);
		int height = (int)Math.ceil(bottom-top);
		return new VideoBounds(left, top, width, height);
	}

	private void calcCharMatrix(TACharacter c, boolean perCharacter3D, Resolution resolution) {
		if (c.matrix != null) {
			return;
		}

		Matrix4d m = new Matrix4d();
		m.setIdentity();

		Matrix4d t = new Matrix4d();

		// 位置
		t.setIdentity();
		t.setTranslation(new Vector3d(
				resolution.scale(c.positionX),
				resolution.scale(c.positionY),
				resolution.scale(perCharacter3D ? c.positionZ : 0)));
		m.mul(t);

		// 回転
		if (perCharacter3D) {
			t.setIdentity();
			t.setRotation(new AxisAngle4d(1, 0, 0, Math.toRadians(c.rotationX)));
			m.mul(t);

			t.setIdentity();
			t.setRotation(new AxisAngle4d(0, 1, 0, Math.toRadians(c.rotationY)));
			m.mul(t);
		}
		t.setIdentity();
		t.setRotation(new AxisAngle4d(0, 0, 1, Math.toRadians(c.rotationZ)));
		m.mul(t);

		// スケール
		t.setIdentity();
		t.setRotationScale(new Matrix3d(c.scaleX, 0, 0, 0, c.scaleY, 0, 0, 0, perCharacter3D ? c.scaleZ : 1));
		m.mul(t);

		// 歪曲軸をもとに戻す
		double skewAxis = Math.toRadians(c.skewAxis);
		t.setIdentity();
		t.setRotation(new AxisAngle4d(0, 0, 1, -skewAxis));
		m.mul(t);

		// 歪曲
		t.setIdentity();
		t.m01 = Math.tan(Math.toRadians(-c.skew));
		m.mul(t);

		// 歪曲軸を傾ける
		t.setIdentity();
		t.setRotation(new AxisAngle4d(0, 0, 1, skewAxis));
		m.mul(t);

		// アンカーポイント
		t.setIdentity();
		t.setTranslation(new Vector3d(
				resolution.scale(-c.anchorPointX),
				resolution.scale(-c.anchorPointY),
				resolution.scale(perCharacter3D ? -c.anchorPointZ : 0)));
		m.mul(t);

		// Y反転
		t.setIdentity();
		t.setRotationScale(new Matrix3d(1, 0, 0, 0, -1, 0, 0, 0, 1));
		m.mul(t);

		c.matrix = m;
	}

	private void calcCharBBox2(TACharacter c) {
		if (c.bbox2 != null) {
			return;
		}

		// 線幅とブラーの分だけ広げる。
		double outerX = c.strokeWidth*0.5 + c.blurX;
		double outerY = c.strokeWidth*0.5 + c.blurY;

		// LineJoin がマイターまたはラウンドの場合はさらに広げないと端が切れることがある。
		// しかし、デフォルトの miterLimit だと strokeWidth の10倍広げることになるが
		// それはちょっと広げ過ぎ。ラウンドの場合は最大でも strokeWidth*0.5 だけ広げればよい。
		// というわけで strokeWidth*0.5 だけ広げておく。
		outerX += c.strokeWidth*0.5;
		outerY += c.strokeWidth*0.5;

		// getVideoFrameBounds() からの呼び出し時は BufferFont の場合でも PolygonFont としているので、
		// 計算結果が微妙にずれる可能性がある。そのため、その誤差分として +2 広げる。(+2が妥当かどうかは不明)
		// (アンチエイリアスのためbboxよりも僅かに外側に広がっているので、その分としても広げることは必要)
		outerX += 2;
		outerY += 2;

		c.bbox2 = new double[] {
				c.bbox[0]-outerX,	// left
				c.bbox[4]+outerY,	// top
				c.bbox[3]+outerX,	// right
				c.bbox[1]-outerY	// bottom
		};
	}

	private void calcCharBBox3(TACharacter c, Resolution resolution) {
		if (c.bbox3 != null) {
			return;
		}

		calcCharBBox2(c);

		Point3d[] bbox = new Point3d[] {
				new Point3d(c.bbox2[0], c.bbox2[1], 0),
				new Point3d(c.bbox2[2], c.bbox2[1], 0),
				new Point3d(c.bbox2[2], c.bbox2[3], 0),
				new Point3d(c.bbox2[0], c.bbox2[3], 0)
		};

		double[] bbox3 = new double[] {
				Double.POSITIVE_INFINITY,
				Double.POSITIVE_INFINITY,
				Double.NEGATIVE_INFINITY,
				Double.NEGATIVE_INFINITY
		};

		Matrix4d m = new Matrix4d(c.matrix);
		m.mul(new Matrix4d(
				resolution.scale, 0, 0, 0,
				0, resolution.scale, 0, 0,
				0, 0, resolution.scale, 0,
				0, 0, 0, 1));

		for (Point3d pt : bbox) {
			m.transform(pt);
			bbox3[0] = Math.min(bbox3[0], pt.x);
			bbox3[1] = Math.min(bbox3[1], pt.y);
			bbox3[2] = Math.max(bbox3[2], pt.x);
			bbox3[3] = Math.max(bbox3[3], pt.y);
		}

		c.bbox3 = bbox3;
	}

	private static int[][] frustumPlaneTable = new int[][] {
		// 最初の3つがクリップ面を構成する頂点のインデックス。
		// 残りひとつがそのクリップ面と向き合う面の頂点のひとつ。
		{0,2,6,1}, {1,3,7,0}, {0,1,2,6}, {6,7,4,0}, {0,1,6,2}, {2,3,4,0}
	};

	private void clipCharBBox2(TAContext ctx, TACharacter c) {
		if (c.bbox2c != null) {
			return;
		}

		calcCharBBox2(c);

		VideoBounds viewport = ctx.viewport;
		double[] prjMatrix = ctx.prjMatrix;
		double[] mvMatrix = ctx.mvMatrix;
		Resolution resolution = ctx.resolution;

		Matrix4d m = new Matrix4d(mvMatrix);
		m.transpose();

		if (!ctx.perCharacter3D) {
			m.mul(c.matrix);
		}

		m.mul(new Matrix4d(
				resolution.scale, 0, 0, 0,
				0, resolution.scale, 0, 0,
				0, 0, resolution.scale, 0,
				0, 0, 0, 1));

		mvMatrix = new double[] {
				m.m00, m.m10, m.m20, m.m30,
				m.m01, m.m11, m.m21, m.m31,
				m.m02, m.m12, m.m22, m.m32,
				m.m03, m.m13, m.m23, m.m33
		};

		// 以下は VideoLayerComposer.intersectsWithFrustum とほぼ同じ。

		int w = viewport.width;
		int h = viewport.height;
		int[] vp = new int[] { 0, 0, w, h };
		double[][] unprj = new double[8][3];

		GLU glu = _context.getGLU();
		glu.gluUnProject(0, 0, 0, mvMatrix, 0, prjMatrix, 0, vp, 0, unprj[0], 0);
		glu.gluUnProject(0, 0, 1, mvMatrix, 0, prjMatrix, 0, vp, 0, unprj[1], 0);
		glu.gluUnProject(w, 0, 0, mvMatrix, 0, prjMatrix, 0, vp, 0, unprj[2], 0);
		glu.gluUnProject(w, 0, 1, mvMatrix, 0, prjMatrix, 0, vp, 0, unprj[3], 0);
		glu.gluUnProject(w, h, 0, mvMatrix, 0, prjMatrix, 0, vp, 0, unprj[4], 0);
		glu.gluUnProject(w, h, 1, mvMatrix, 0, prjMatrix, 0, vp, 0, unprj[5], 0);
		glu.gluUnProject(0, h, 0, mvMatrix, 0, prjMatrix, 0, vp, 0, unprj[6], 0);
		glu.gluUnProject(0, h, 1, mvMatrix, 0, prjMatrix, 0, vp, 0, unprj[7], 0);

		Vector4d[] frustumPlanes = new Vector4d[6];
		for (int i = 0; i < 6; ++i) {
			Point3d p0 = new Point3d(unprj[frustumPlaneTable[i][0]]);
			Point3d p1 = new Point3d(unprj[frustumPlaneTable[i][1]]);
			Point3d p2 = new Point3d(unprj[frustumPlaneTable[i][2]]);

			Vector3d v1 = new Vector3d();
			v1.sub(p1, p0);
			Vector3d v2 = new Vector3d();
			v2.sub(p2, p0);

			Vector3d n = new Vector3d();
			n.cross(v1, v2);
			n.normalize();

			double d = -n.dot(new Vector3d(p0));
			frustumPlanes[i] = new Vector4d(n.x, n.y, n.z, d);
		}

		List<Point3d> vertices = Util.newList();
		vertices.add(new Point3d(c.bbox2[0], c.bbox2[1], 0));
		vertices.add(new Point3d(c.bbox2[2], c.bbox2[1], 0));
		vertices.add(new Point3d(c.bbox2[2], c.bbox2[3], 0));
		vertices.add(new Point3d(c.bbox2[0], c.bbox2[3], 0));

		for (int i = 0; i < 6; ++i) {
			Vector4d plane = frustumPlanes[i];

			double[] opposite = unprj[frustumPlaneTable[i][3]];
			double signum = Math.signum(plane.dot(
					new Vector4d(opposite[0], opposite[1], opposite[2], 1)));

			List<Point3d>[] partitioned = partitionPolygon(plane, signum, vertices);
			if (partitioned == null) {
				// クリップ面のひとつに(ほぼ)張り付いている場合。
				continue;
			}
			if (partitioned[1] == null) {
				continue;
			} else if (partitioned[0] == null) {
				// 全ての頂点がクリップ面の外側にある場合。
				vertices.clear();
				break;
			} else {
				vertices = partitioned[0];
			}
		}

		c.bbox2c = vertices.toArray(new Point3d[vertices.size()]);


		if (!vertices.isEmpty()) {
			List<Point2d> bbox2cp = Util.newList();

			double[] bbox4 = new double[] {
					Double.POSITIVE_INFINITY,
					Double.POSITIVE_INFINITY,
					Double.NEGATIVE_INFINITY,
					Double.NEGATIVE_INFINITY
			};

			double[] winPos = new double[3];
			for (Point3d pt : vertices) {
				glu.gluProject(pt.x, pt.y, pt.z, mvMatrix, 0, ctx.prjMatrix, 0, vp, 0, winPos, 0);
				bbox2cp.add(new Point2d(winPos));
				bbox4[0] = Math.min(bbox4[0], winPos[0]);
				bbox4[1] = Math.min(bbox4[1], winPos[1]);
				bbox4[2] = Math.max(bbox4[2], winPos[0]);
				bbox4[3] = Math.max(bbox4[3], winPos[1]);
			}

			c.bbox2cp = bbox2cp.toArray(new Point2d[bbox2cp.size()]);

			int x = (int)Math.floor(bbox4[0]);
			int y = (int)Math.floor(bbox4[1]);
			int width = (int)Math.ceil(bbox4[2] - x);
			int height = (int)Math.ceil(bbox4[3] - y);

			c.bbox4 = new VideoBounds(x, y, width, height);
		}
	}

	// VideoLayerComposer に同じメソッドがある。
	private List<Point3d>[] partitionPolygon(Vector4d plane, double signum, List<Point3d> vertices) {
		double d = 0;
		for (Point3d pt : vertices) {
			double dd = plane.dot(new Vector4d(pt.x, pt.y, pt.z, 1.0));
			if (Math.abs(dd) > Math.abs(d)) {
				d = dd;
			}
		}
		if (Math.abs(d) < 1e-3) {	// TODO 1e-3が妥当な値かどうかわからない。1e-4ではダメなケースがあることは確認した。
			return null;
		}
//		if (Math.abs(d) < 0.01) {
//			System.out.printf("%f: %f%n", context.getTime().toSecond(), d);
//		}

		vertices = Util.newList(vertices);
		int vertIndex1 = -1;
		int vertIndex2 = -1;

		for (int i1 = 0; i1 < vertices.size(); ++i1) {
			int i2 = (i1+1) % vertices.size();
			Point3d vert1 = vertices.get(i1);
			Point3d vert2 = vertices.get(i2);

			double d1 = plane.dot(new Vector4d(vert1.x, vert1.y, vert1.z, 1.0));
			double d2 = plane.dot(new Vector4d(vert2.x, vert2.y, vert2.z, 1.0));

			if (Math.signum(d1) != Math.signum(d2)) {
				if (/*d1 == 0*/ Math.abs(d1) < 1e-3) {
					if (vertIndex1 == -1) {
						vertIndex1 = i1;
					} else {
						vertIndex2 = i1;
						break;
					}
				} else if (/*d2 == 0*/ Math.abs(d2) < 1e-3) {
					if (vertIndex1 == -1) {
						vertIndex1 = ++i1;
					} else {
						vertIndex2 = i1+1;
						break;
					}
				} else {
					Vector3d v = new Vector3d();
					v.sub(vert2, vert1);

					double t = -d1 / plane.dot(new Vector4d(v));
					v.scale(t);

					Point3d vertNew = new Point3d();
					vertNew.add(vert1, v);
					vertices.add(i1+1, vertNew);

					if (vertIndex1 == -1) {
						vertIndex1 = ++i1;
					} else {
						vertIndex2 = i1+1;
						break;
					}
				}
			}
		}

		if (vertIndex2 == -1 || vertIndex2-vertIndex1 <= 1 || vertIndex2-vertIndex1 >= vertices.size()-1) {
			d = 0;
			for (Point3d pt : vertices) {
				double dd = plane.dot(new Vector4d(pt.x, pt.y, pt.z, 1.0));
				if (Math.abs(dd) > Math.abs(d)) {
					d = dd;
				}
			}

			@SuppressWarnings("unchecked")
			List<Point3d>[] result = (Math.signum(d) == signum)
								   ? new List[] { vertices, null }
								   : new List[] { null, vertices };
			return result;
		}

		vertices.add(vertices.get(0));
		List<Point3d> vertices1 = Util.newList(vertices.subList(0, vertIndex1+1));
		List<Point3d> vertices2 = Util.newList(vertices.subList(vertIndex1, vertIndex2+1));
		vertices1.addAll(vertices.subList(vertIndex2, vertices.size()-1));

		d = 0;
		for (Point3d pt : vertices1) {
			double dd = plane.dot(new Vector4d(pt.x, pt.y, pt.z, 1.0));
			if (Math.abs(dd) > Math.abs(d)) {
				d = dd;
			}
		}

		@SuppressWarnings("unchecked")
		List<Point3d>[] result = (Math.signum(d) == signum)
							   ? new List[] { vertices1, vertices2 }
							   : new List[] { vertices2, vertices1 };
		return result;
	}

	private VideoBounds bbox2ToVideoBounds(TACharacter c) {
		calcCharBBox2(c);

		int x = (int)Math.floor(c.bbox2[0]);
		int y = (int)Math.floor(c.bbox2[3]);
		int width = (int)Math.ceil(c.bbox2[2] - x);
		int height = (int)Math.ceil(c.bbox2[1] - y);

		return new VideoBounds(x, y, width, height);
	}

	private List<TACharacter> createTACharacters(int[] totals, String sourceText, FTGLfont font) {
		if (!sourceText.endsWith("\n")) {
			sourceText += "\n";
		}

		List<TACharacter> chars = Util.newList();

		int charTotal = 0;
		int charTotal2 = 0;
		int wordTotal = 0;
		int lineTotal = 0;
		boolean word = false;

		for (int i = 0; i < sourceText.length(); ++i) {
			int codePoint = sourceText.codePointAt(i);
			if (Character.isSupplementaryCodePoint(codePoint)) {
				++i;
			}
			if (codePoint < CharacterGroup.MIN_CODE_POINT || codePoint > CharacterGroup.MAX_CODE_POINT) {
				codePoint = '#';
			}

			if (codePoint == '\n') {
				if (word) {
					++wordTotal;
					word = false;
				}
				++lineTotal;
				continue;
			}

			String string = new String(new int[] { codePoint }, 0, 1);
			double advance = FTGL.ftglGetFontAdvance(font, string);
			float[] bbox = new float[6];
			FTGL.ftglGetFontBBox(font, string, -1, bbox);

			int type = Character.getType(codePoint);
			if (type == Character.SPACE_SEPARATOR) {
				chars.add(new TACharacter(codePoint, string, advance, bbox,
									charTotal, -1, -1, lineTotal));

				if (word) {
					++wordTotal;
					word = false;
				}
			} else {
				chars.add(new TACharacter(codePoint, string, advance, bbox,
									charTotal, charTotal2, wordTotal, lineTotal));

				++charTotal2;
				word = true;
			}
			++charTotal;
		}

		totals[0] = charTotal;
		totals[1] = charTotal2;
		totals[2] = wordTotal;
		totals[3] = lineTotal;
		return chars;
	}

	private void initTACharacters(
			int[] totals, List<TACharacter> chars, boolean perCharacter3D, FTGLfont font, Time mediaTime) {

		Color fillColor = null;
		Color strokeColor = null;
		double strokeWidth = 0;
		if (_textType == TextType.FILL_ONLY || _textType == TextType.FILL_AND_STROKE) {
			fillColor = _fillColor.value(_context);
		}
		if (_textType == TextType.STROKE_ONLY || _textType == TextType.FILL_AND_STROKE) {
			strokeColor = _strokeColor.value(_context);
			strokeWidth = _strokeWidth.value(_context);
		}

		for (TACharacter c : chars) {
			c.fillColor = fillColor;
			c.strokeColor = strokeColor;
			c.strokeWidth = strokeWidth;
		}

		double tracking = _tracking.value(_context);
		double leading = _leading.value(_context);
		HorizontalAlignment hAlign = _horizontalAlignment.value(_context);

		int fontSize = FTGL.ftglGetFontFaceSize(font);
		tracking *= 0.001 * fontSize;
		if (leading < 0) {
			leading = fontSize * 1.2;	// AEは「自動」のときにはフォントサイズの1.2倍になってるっぽい。	
		}

		if (_textAnimators.isEmpty()) {
			initPosition(chars, tracking, leading);
			horizontalAlignment(chars, hAlign);

		} else {
			Map<TextAnimator, List<Evaluator>> evaluators = selectorEvaluators(totals, mediaTime);
			Map<TextAnimator, Map<TAProperty, Object>> properties = animatorProperties(perCharacter3D);

			CharacterAlignment charAlign = _characterAlignment.value(_context);
			if (charAlign == CharacterAlignment.ADJUST_KERNING) {
				characterValueAndOffset(chars, font, charAlign, evaluators, properties);
			}

			initPosition(chars, tracking, leading);
			horizontalAlignment(chars, hAlign);

			tracking(chars, hAlign, evaluators, properties);
			lineSpacing(chars, evaluators, properties);

			AnchorPointGrouping ancGrouping = _anchorPointGrouping.value(_context);
			Vec2d grAlign = _groupingAlignment.value(_context);
			double ascender = FTGL.ftglGetFontAscender(font);
			anchorPointGrouping(chars, ancGrouping, grAlign.x*0.01, grAlign.y*0.01, ascender);

			if (charAlign != CharacterAlignment.ADJUST_KERNING) {
				characterValueAndOffset(chars, font, charAlign, evaluators, properties);
			}

			otherProperties(chars, perCharacter3D, evaluators, properties);
		}

		for (Iterator<TACharacter> it = chars.iterator(); it.hasNext(); ) {
			TACharacter c = it.next();
			if (c.charIndex2 < 0
					|| !Character.isDefined(c.codePoint)
					|| Character.isISOControl(c.codePoint)) {
				it.remove();
			}
		}
	}

	private void characterValueAndOffset(
			List<TACharacter> chars, FTGLfont font, CharacterAlignment charAlign,
			Map<TextAnimator, List<Evaluator>> evaluators,
			Map<TextAnimator, Map<TAProperty, Object>> properties) {

		List<TextAnimator> animators = Util.newList();
		for (TextAnimator animator : _textAnimators) {
			if (animator.isEnabled() && (animator.getProperties().contains(TAProperty.characterValue)
										|| animator.getProperties().contains(TAProperty.characterOffset))) {
				animators.add(animator);
			}
		}
		if (animators.isEmpty()) return;

		for (TextAnimator animator : animators) {
			Map<TAProperty, Object> props = properties.get(animator);
			Object charValueObj = props.get(TAProperty.characterValue);
			Object charOffsetObj = props.get(TAProperty.characterOffset);
			Object charRange = props.get(TAProperty.characterRange);
			if (charRange == null) {
				charRange = CharacterRange.GROUP;
			}

			for (TACharacter c : chars) {
				if (c.charIndex2 < 0) continue;

				double t;
				List<Evaluator> list = evaluators.get(animator);
				if (list.isEmpty()) {
					t = 1.0;
				} else {
					double[] tvec = null;
					for (Evaluator evaluator : list) {
						tvec = evaluator.evaluate(tvec, c.indices);
					}
					if (tvec == null) continue;
					t = tvec[0];
				}

				int codePoint = c.codePoint;

				if (charValueObj != null) {
					int charValue = (Integer) charValueObj;
					if (charValue > 0) {
						if (charRange == CharacterRange.GROUP) {
							codePoint = CharacterGroup.map(codePoint);
							charValue = CharacterGroup.map(charValue);
							int[] r0 = CharacterGroup.rangeOf(codePoint);
							int[] r1 = CharacterGroup.rangeOf(charValue);
							if (r0[0] < r1[0]) {
								int gap = r1[0] - r0[1];
								codePoint = (int)Math.round((1-t)*codePoint + t*(charValue-gap));
								if (codePoint >= r0[1]) {
									codePoint += gap;
								}
							} else if (r1[0] < r0[0]) {
								int gap = r0[0] - r1[1];
								codePoint = (int)Math.round((1-t)*(codePoint-gap) + t*charValue);
								if (codePoint >= r1[1]) {
									codePoint += gap;
								}
							} else {
								codePoint = (int)Math.round((1-t)*codePoint + t*charValue);
							}
							codePoint = CharacterGroup.unmap(codePoint);
						} else {
							codePoint = (int)Math.round((1-t)*codePoint + t*charValue);
						}
					}
				}

				if (charOffsetObj != null) {
					int charOffset = (Integer) charOffsetObj;
					if (charOffset != 0) {
						int[] range;
						if (charRange == CharacterRange.GROUP) {
							codePoint = CharacterGroup.map(codePoint);
							range = CharacterGroup.rangeOf(codePoint);
						} else {
							range = CharacterGroup.fullRange();
						}

						int mod = range[1] - range[0];
						codePoint = (int)Math.round(codePoint + t*charOffset - range[0]) % mod;
						if (codePoint < 0) {
							codePoint += mod;
						}
						codePoint += range[0];

						if (charRange == CharacterRange.GROUP) {
							codePoint = CharacterGroup.unmap(codePoint);
						}
					}
				}

				double oldAdvance = c.advance;

				c.codePoint = codePoint;
				c.string = new String(new int[] { codePoint }, 0, 1);
				c.advance = FTGL.ftglGetFontAdvance(font, c.string);
				c.bbox = new float[6];
				FTGL.ftglGetFontBBox(font, c.string, -1, c.bbox);

				switch (charAlign) {
					case LEFT_OR_TOP:
						// nothing to do
						break;

					case CENTER:
						c.anchorPointX += (c.advance - oldAdvance) * 0.5;
						break;

					case RIGHT_OR_BOTTOM:
						c.anchorPointX += c.advance - oldAdvance;
						break;

					case ADJUST_KERNING:
						// initPositionで設定されるので、ここでは何もしない。
						break;
				}
			}
		}
	}

	private void initPosition(List<TACharacter> chars, double tracking, double leading) {
		double y = 0;
		int lineIndex = 0;
		List<TACharacter> lineChars = Util.newList();
		for (Iterator<TACharacter> it = chars.iterator();;) {
			TACharacter c = null;
			if (it.hasNext()) {
				c = it.next();
				if (c.lineIndex == lineIndex) {
					lineChars.add(c);
					continue;
				}
			}

			if (!lineChars.isEmpty()) {
				double x = 0;
				for (TACharacter lc : lineChars) {
					lc.positionX = x;
					lc.positionY = y;
					x += Math.max(0, lc.advance + tracking);
				}
				lineChars.clear();
			}

			if (c != null) {
				y += leading * (c.lineIndex - lineIndex);
				lineIndex = c.lineIndex;
				lineChars.add(c);
			} else {
				break;
			}
		}
	}

	private void horizontalAlignment(List<TACharacter> chars, HorizontalAlignment hAlign) {
		int lineIndex = 0;
		List<TACharacter> lineChars = Util.newList();
		for (Iterator<TACharacter> it = chars.iterator();;) {
			TACharacter c = null;
			if (it.hasNext()) {
				c = it.next();
				if (c.lineIndex == lineIndex) {
					lineChars.add(c);
					continue;
				}
			}

			if (!lineChars.isEmpty()) {
				double dx;
				switch (hAlign) {
					case LEFT:
						dx = Double.POSITIVE_INFINITY;
						for (TACharacter lc : lineChars) {
							dx = Math.min(dx, lc.positionX);
						}
						break;

					case CENTER:
						double left = Double.POSITIVE_INFINITY;
						double right = Double.NEGATIVE_INFINITY;
						for (TACharacter lc : lineChars) {
							left = Math.min(left, lc.positionX);
							right = Math.max(right, lc.positionX + lc.advance);
						}
						dx = (left + right) * 0.5;
						break;

					default: //case RIGHT:
						dx = Double.NEGATIVE_INFINITY;
						for (TACharacter lc : lineChars) {
							dx = Math.max(dx, lc.positionX + lc.advance);
						}
						break;
				}
				for (TACharacter lc : lineChars) {
					lc.positionX -= dx;
				}
				lineChars.clear();
			}

			if (c != null) {
				lineIndex = c.lineIndex;
				lineChars.add(c);
			} else {
				break;
			}
		}
	}

	private void anchorPointGrouping(List<TACharacter> chars,
			AnchorPointGrouping ancGrouping, double grAlignX, double grAlignY, double ascender) {

		switch (ancGrouping) {
			case CHARACTER:
				for (TACharacter c : chars) {
					c.anchorPointX = c.advance * 0.5 * (1 + grAlignX);
					c.anchorPointY = ascender * grAlignY;
					c.positionX += c.anchorPointX;
					c.positionY += c.anchorPointY;
				}
				break;

			case WORD: {
				int wordIndex = 0;
				List<TACharacter> wordChars = Util.newList();
				for (Iterator<TACharacter> it = chars.iterator();;) {
					TACharacter c = null;
					if (it.hasNext()) {
						c = it.next();
						if (c.wordIndex < 0) {
							continue;
						} else if (c.wordIndex == wordIndex) {
							wordChars.add(c);
							continue;
						}
					}

					if (!wordChars.isEmpty()) {
						double left = Double.POSITIVE_INFINITY;
						double right = Double.NEGATIVE_INFINITY;
						for (TACharacter wc : wordChars) {
							left = Math.min(left, wc.positionX);
							right = Math.max(right, wc.positionX + wc.advance);
						}
						double posX = (left + right) * 0.5 + (right - left) * grAlignX * 0.5;
						double ancY = ascender * grAlignY;
						for (TACharacter wc : wordChars) {
							wc.anchorPointX = posX - wc.positionX;
							wc.anchorPointY = ancY;
							wc.positionX = posX;
							wc.positionY += ancY;
						}
						wordChars.clear();
					}

					if (c != null) {
						wordIndex = c.wordIndex;
						wordChars.add(c);
					} else {
						break;
					}
				}
				break;
			}

			case LINE: {
				int lineIndex = 0;
				List<TACharacter> lineChars = Util.newList();
				for (Iterator<TACharacter> it = chars.iterator();;) {
					TACharacter c = null;
					if (it.hasNext()) {
						c = it.next();
						if (c.lineIndex == lineIndex) {
							lineChars.add(c);
							continue;
						}
					}

					if (!lineChars.isEmpty()) {
						double left = Double.POSITIVE_INFINITY;
						double right = Double.NEGATIVE_INFINITY;
						for (TACharacter lc : lineChars) {
							left = Math.min(left, lc.positionX);
							right = Math.max(right, lc.positionX + lc.advance);
						}
						double posX = (left + right) * 0.5 + (right - left) * grAlignX * 0.5;
						double ancY = ascender * grAlignY;
						for (TACharacter lc : lineChars) {
							lc.anchorPointX = posX - lc.positionX;
							lc.anchorPointY = ancY;
							lc.positionX = posX;
							lc.positionY += ancY;
						}
						lineChars.clear();
					}

					if (c != null) {
						lineIndex = c.lineIndex;
						lineChars.add(c);
					} else {
						break;
					}
				}
				break;
			}

			case ALL: {
				double left = Double.POSITIVE_INFINITY;
				double top = Double.POSITIVE_INFINITY;
				double right = Double.NEGATIVE_INFINITY;
				double bottom = Double.NEGATIVE_INFINITY;
				for (TACharacter c : chars) {
					left = Math.min(left, c.positionX);
					top = Math.min(top, c.positionY - ascender);
					right = Math.max(right, c.positionX + c.advance);
					bottom = Math.max(bottom, c.positionY);
				}
				double posX = (left + right) * 0.5 + (right - left) * grAlignX * 0.5;
				double posY = (top + bottom) * 0.5 + (bottom - top) * grAlignY * 0.5;
				for (TACharacter c : chars) {
					c.anchorPointX = posX - c.positionX;
					c.anchorPointY = posY - c.positionY;
					c.positionX = posX;
					c.positionY = posY;
				}
				break;
			}
		}
	}

	private void tracking(
			List<TACharacter> chars, HorizontalAlignment hAlign,
			Map<TextAnimator, List<Evaluator>> evaluators,
			Map<TextAnimator, Map<TAProperty, Object>> properties) {

		Map<TextAnimator, Double> trackingValues = Util.newMap();
		for (TextAnimator animator : _textAnimators) {
			Double tracking = (Double) properties.get(animator).get(TAProperty.tracking);
			if (tracking != null) {
				trackingValues.put(animator, tracking);
			}
		}
		if (trackingValues.isEmpty()) return;

		int lineIndex = 0;
		List<TACharacter> lineChars = Util.newList();
		for (Iterator<TACharacter> it = chars.iterator();;) {
			TACharacter c = null;
			if (it.hasNext()) {
				c = it.next();
				if (c.lineIndex == lineIndex) {
					lineChars.add(c);
					continue;
				}
			}

			if (!lineChars.isEmpty()) {
				double dx = 0;
				for (Iterator<TACharacter> it2 = lineChars.iterator();;) {
					TACharacter lc = it2.next();
					lc.positionX += dx;

					if (!it2.hasNext()) break;

					for (TextAnimator animator : _textAnimators) {
						Double tracking = trackingValues.get(animator);
						if (tracking == null) continue;

						List<Evaluator> list = evaluators.get(animator);
						if (list.isEmpty()) {
							dx += tracking;
						} else {
							double[] tvec = null;
							for (Evaluator evaluator : list) {
								tvec = evaluator.evaluate(tvec, lc.indices);
							}
							if (tvec != null) {
								dx += tvec[0] * tracking;
							}
						}
					}
				}

				switch (hAlign) {
					case CENTER:
						dx *= 0.5;
						// fall through
					case RIGHT:
						for (TACharacter lc : lineChars) {
							lc.positionX -= dx;
						}
						break;
				}

				lineChars.clear();
			}

			if (c != null) {
				lineIndex = c.lineIndex;
				lineChars.add(c);
			} else {
				break;
			}
		}
	}

	private void lineSpacing(
			List<TACharacter> chars,
			Map<TextAnimator, List<Evaluator>> evaluators,
			Map<TextAnimator, Map<TAProperty, Object>> properties) {

		Map<TextAnimator, Vec2d> lineSpacingValues = Util.newMap();
		for (TextAnimator animator : _textAnimators) {
			Vec2d lineSpacing = (Vec2d) properties.get(animator).get(TAProperty.lineSpacing);
			if (lineSpacing != null) {
				lineSpacingValues.put(animator, lineSpacing);
			}
		}
		if (lineSpacingValues.isEmpty()) return;

		double dx = 0, dy = 0;
		int lineIndex = 0;
		List<TACharacter> lineChars = Util.newList();
		for (Iterator<TACharacter> it = chars.iterator();;) {
			TACharacter c = null;
			if (it.hasNext()) {
				c = it.next();
				if (c.lineIndex == 0) {
					continue;
				} else if (c.lineIndex == lineIndex) {
					lineChars.add(c);
					continue;
				}
			}

			if (!lineChars.isEmpty()) {
				int[][] indices = new int[lineChars.size()][];
				for (int i = 0; i < indices.length; ++i) {
					indices[i] = lineChars.get(i).indices;
				}

				for (TextAnimator animator : _textAnimators) {
					Vec2d lineSpacing = lineSpacingValues.get(animator);
					if (lineSpacing == null) continue;

					List<Evaluator> list = evaluators.get(animator);
					if (list.isEmpty()) {
						dx += lineSpacing.x;
						dy += lineSpacing.y;
					} else {
						double[] tvec = null;
						for (Evaluator evaluator : list) {
							tvec = evaluator.evaluate(tvec, indices);
						}
						if (tvec != null) {
							dx += tvec[0] * lineSpacing.x;
							dy += tvec[1] * lineSpacing.y;
						}
					}
				}

				for (TACharacter lc : lineChars) {
					lc.positionX += dx;
					lc.positionY += dy;
				}

				lineChars.clear();
			}

			if (c != null) {
				lineIndex = c.lineIndex;
				lineChars.add(c);
			} else {
				break;
			}
		}
	}

	private void otherProperties(
			List<TACharacter> chars, boolean perCharacter3D,
			Map<TextAnimator, List<Evaluator>> evaluators,
			Map<TextAnimator, Map<TAProperty, Object>> properties) {

		for (TextAnimator animator : _textAnimators) {
			if (!animator.isEnabled()) continue;

			CHAR_LOOP:
			for (TACharacter c : chars) {
				double[] tvec = null;
				double[] fillHSL = null;
				double[] strokeHSL = null;

				for (Map.Entry<TAProperty, Object> e : properties.get(animator).entrySet()) {
					TAProperty p = e.getKey();
					switch (p) {
						case tracking:
						case lineSpacing:
						case characterRange:
						case characterValue:
						case characterOffset:
							// これらはすでに処理済み。
							continue;
					}

					// tvecの取得はこのループの外側で先にやるのが自然だが、
					// TRACKING等の処理済みのプロパティのみの場合、
					// 先にやっておくと無駄が生じるのでここで行っている。
					// またその影響で、continue で抜ける場合は抜ける先が CHAR_LOOP となっている。
					if (tvec == null) {
						List<Evaluator> list = evaluators.get(animator);
						if (list.isEmpty()) {
							tvec = new double[] { 1.0, 1.0, 1.0 };
						} else {
							for (Evaluator evaluator : list) {
								tvec = evaluator.evaluate(tvec, c.indices);
							}
							if (tvec == null) continue CHAR_LOOP;
						}
					}

					Object v = e.getValue();
					switch (p) {
						case anchorPoint:
							c.anchorPointX += tvec[0] * ((Vec3d)v).x;
							c.anchorPointY += tvec[1] * ((Vec3d)v).y;
							if (perCharacter3D) {
								c.anchorPointZ += tvec[2] * ((Vec3d)v).z;
							}
							break;

						case position:
							c.positionX += tvec[0] * ((Vec3d)v).x;
							c.positionY += tvec[1] * ((Vec3d)v).y;
							if (perCharacter3D) {
								c.positionZ += tvec[2] * ((Vec3d)v).z;
							}
							break;

						case scale:
							c.scaleX *= (1-tvec[0]) + tvec[0] * ((Vec3d)v).x * 0.01;
							c.scaleY *= (1-tvec[1]) + tvec[1] * ((Vec3d)v).y * 0.01;
							if (perCharacter3D) {
								c.scaleZ *= (1-tvec[2]) + tvec[2] * ((Vec3d)v).z * 0.01;
							}
							break;

						case skew:
							c.skew += tvec[0] * (Double)v;
							break;

						case skewAxis:
							c.skewAxis += tvec[0] * (Double)v;
							break;

						case rotationX:
							if (perCharacter3D) {
								c.rotationX += tvec[0] * (Double)v;
							}
							break;

						case rotationY:
							if (perCharacter3D) {
								c.rotationY += tvec[1] * (Double)v;
							}
							break;

						case rotationZ:
							c.rotationZ += tvec[2] * (Double)v;
							break;

						case opacity:
							c.opacity *= (1-tvec[0]) + tvec[0] * (Double)v * 0.01;
							break;

						case fillRGB:
							if (c.fillColor != null) {
								Color color = (Color)v;
								c.fillColor = new Color(
										(1-tvec[0]) * c.fillColor.r + tvec[0] * color.r,
										(1-tvec[1]) * c.fillColor.g + tvec[1] * color.g,
										(1-tvec[2]) * c.fillColor.b + tvec[2] * color.b);
							}
							break;

						case fillHue:
						case fillSaturation:
						case fillLuminosity:
							if (c.fillColor != null) {
								if (fillHSL == null) {
									fillHSL = new double[3];
								}
								fillHSL[p.ordinal()-TAProperty.fillHue.ordinal()] = (Double)v;
							}
							break;

						case fillOpacity:
							if (c.fillColor != null) {
								c.fillOpacity *= (1-tvec[0]) + tvec[0] * (Double)v * 0.01;
							}
							break;

						case strokeRGB:
							if (c.strokeColor != null) {
								Color color = (Color)v;
								c.strokeColor = new Color(
										(1-tvec[0]) * c.strokeColor.r + tvec[0] * color.r,
										(1-tvec[1]) * c.strokeColor.g + tvec[1] * color.g,
										(1-tvec[2]) * c.strokeColor.b + tvec[2] * color.b);
							}
							break;

						case strokeHue:
						case strokeSaturation:
						case strokeLuminosity:
							if (c.strokeColor != null) {
								if (strokeHSL == null) {
									strokeHSL = new double[3];
								}
								strokeHSL[p.ordinal()-TAProperty.strokeHue.ordinal()] = (Double)v;
							}
							break;

						case strokeOpacity:
							if (c.strokeColor != null) {
								c.strokeOpacity *= (1-tvec[0]) + tvec[0] * (Double)v * 0.01;
							}
							break;

						case strokeWidth:
							if (getTextType() != TextType.FILL_ONLY) {
								c.strokeWidth = Math.max(0, c.strokeWidth + tvec[0] * (Double)v);
							}
							break;

						case blur:
							c.blurX = Math.max(0, c.blurX + tvec[0] * ((Vec2d)v).x);
							c.blurY = Math.max(0, c.blurY + tvec[1] * ((Vec2d)v).y);
							break;

						default:
							throw new RuntimeException("unknown TAProperty: " + p);
					}
				}

				if (fillHSL != null) {
					c.fillColor = HSLUtil.addHSL(c.fillColor,
							tvec[0]*fillHSL[0]/360, tvec[1]*fillHSL[1]*0.01, tvec[2]*fillHSL[2]*0.01);
				}
				if (strokeHSL != null) {
					c.strokeColor = HSLUtil.addHSL(c.strokeColor,
							tvec[0]*strokeHSL[0]/360, tvec[1]*strokeHSL[1]*0.01, tvec[2]*strokeHSL[2]*0.01);
				}
			}
		}
	}

	private Map<TextAnimator, List<Evaluator>> selectorEvaluators(int[] totals, Time mediaTime) {
		Map<TextAnimator, List<Evaluator>> map = Util.newMap();

		for (TextAnimator animator : _textAnimators) {
			List<Evaluator> list = Util.newList();
			map.put(animator, list);

			if (!animator.isEnabled()) continue;

			for (TASelector selector : animator.getSelectors()) {
				if (selector.isEnabled()) {
					list.add(selector.createEvaluator(totals, mediaTime));
				}
			}
		}

		return map;
	}

	private Map<TextAnimator, Map<TAProperty, Object>> animatorProperties(boolean perCharacter3D) {
		Map<TextAnimator, Map<TAProperty, Object>> map = Util.newMap();

		for (TextAnimator animator : _textAnimators) {
			Map<TAProperty, Object> map2 = Util.newMap();
			map.put(animator, map2);

			if (!animator.isEnabled()) continue;

			for (TAProperty p : animator.getProperties()) {
				if (!perCharacter3D) {
					switch (p) {
						case rotationX:
						case rotationY:
							continue;
					}
				}

				Object value;
				switch (p) {
					case anchorPoint: value = animator.getAnchorPoint().value(_context); break;
					case position:    value = animator.getPosition().value(_context); break;
					case scale:       value = animator.getScale().value(_context); break;
					case skew:        value = animator.getSkew().value(_context); break;
					case skewAxis:    value = animator.getSkewAxis().value(_context); break;
					case rotationX:   value = animator.getRotationX().value(_context); break;
					case rotationY:   value = animator.getRotationY().value(_context); break;
					case rotationZ:   value = animator.getRotationZ().value(_context); break;
					case opacity:     value = animator.getOpacity().value(_context); break;

					case fillRGB:        value = animator.getFillRGB().value(_context); break;
					case fillHue:        value = animator.getFillHue().value(_context); break;
					case fillSaturation: value = animator.getFillSaturation().value(_context); break;
					case fillLuminosity: value = animator.getFillLuminosity().value(_context); break;
					case fillOpacity:    value = animator.getFillOpacity().value(_context); break;

					case strokeRGB:        value = animator.getStrokeRGB().value(_context); break;
					case strokeHue:        value = animator.getStrokeHue().value(_context); break;
					case strokeSaturation: value = animator.getStrokeSaturation().value(_context); break;
					case strokeLuminosity: value = animator.getStrokeLuminosity().value(_context); break;
					case strokeOpacity:    value = animator.getStrokeOpacity().value(_context); break;
					case strokeWidth:      value = animator.getStrokeWidth().value(_context); break;

					case tracking:    value = animator.getTracking().value(_context); break;
					case lineSpacing: value = animator.getLineSpacing().value(_context); break;

					case characterRange:  value = animator.getCharacterRange().value(_context); break;
					case characterValue:  value = animator.getCharacterValue().value(_context); break;
					case characterOffset: value = animator.getCharacterOffset().value(_context); break;

					case blur: value = animator.getBlur().value(_context); break;

					default: throw new RuntimeException("unknown TAProperty: " + p);
				}
				map2.put(p, value);
			}
		}

		return map;
	}

	private static class TAContext {
		VideoBounds viewport;
		double[] prjMatrix;
		double[] mvMatrix;
		Resolution resolution;
		List<TACharacter> chars;
		boolean perCharacter3D;
		boolean polygonFont;
		FTGLfont fillFont;
		FTGLfont strokeFont;
		LineJoin lineJoin;
		double miterLimit;
	}

	private static class TACharacter implements Cloneable {
		int codePoint;
		String string;
		double advance;
		float[] bbox;		// left, lower, near, right, upper, far

		//final int charIndex;
		final int charIndex2;
		final int wordIndex;
		final int lineIndex;
		final int[] indices;

		double anchorPointX, anchorPointY, anchorPointZ;
		double positionX, positionY, positionZ;
		double scaleX = 1.0, scaleY = 1.0, scaleZ = 1.0;
		double skew, skewAxis;
		double rotationX, rotationY, rotationZ;
		double opacity = 1.0;
		Color fillColor;
		double fillOpacity = 1.0;
		Color strokeColor;
		double strokeOpacity = 1.0;
		double strokeWidth;
		double blurX, blurY;

		Matrix4d matrix;

		double[] bbox2;		// bboxに線の幅やブラーの量などを加えた矩形 (left, top, right, bottomの順)
		double[] bbox3;		// bbox2をmatrix*resolutionでトランスフォームしてできる領域を囲む矩形 (順序はbbox2と同じ)

		Point3d[] bbox2c;	// bbox2を視体積でクリッピングしてできたポリゴンの頂点
		Point2d[] bbox2cp;	// bbox2cの各頂点をスクリーン空間に投影したもの
		VideoBounds bbox4;	// bbox2cをスクリーン空間に投影してできる領域を囲む矩形

		IVideoBuffer blurredFill;
		IVideoBuffer blurredStroke;

		TACharacter(int codePoint, String string, double advance, float[] bbox,
				int charIndex, int charIndex2, int wordIndex, int lineIndex) {

			this.codePoint = codePoint;
			this.string = string;
			this.advance = advance;
			this.bbox = bbox;

			//this.charIndex = charIndex;
			this.charIndex2 = charIndex2;
			this.wordIndex = wordIndex;
			this.lineIndex = lineIndex;
			this.indices = new int[] { charIndex, charIndex2, wordIndex, lineIndex };
		}

		protected TACharacter clone() {
			try {
				return (TACharacter) super.clone();
			} catch (CloneNotSupportedException e) {
				throw new RuntimeException(e);
			}
		}
	}


	private class TextInput implements VectorMediaInput {

		public boolean isVideoAvailable() {
			return true;
		}
		
		public boolean isAudioAvailable() {
			return false;
		}

		public Time getDuration() {
			return null;
		}

		public Time getVideoFrameDuration() {
			return null;
		}

		public VideoBounds getVideoFrameBounds() {
			if (!_context.isActive()) {
				// テキストレイヤーの VideoBounds を計算するにはアクティブなコンテキストが必要。
				// コンテキストがアクティブでない場合はダミー値を返すしかないが、(0, 0) にすると
				// エフェクトのプロパティの初期値などで不都合がある場合があるので (100, 100) にしている。
				return new VideoBounds(100, 100);
			}

			// 線のみの場合も PolygonFont で計算する。

			String psName = _font.value(_context);
			int fontSize = _fontSize.value(_context);
			FTGLfont fillFont = getFillFont(psName, fontSize, true);

			if (fillFont == null) {
				return new VideoBounds(0, 0);
			}

			String sourceText = _sourceText.value(_context);

			int[] totals = new int[4];
			List<TACharacter> chars = createTACharacters(totals, sourceText, fillFont);

			if (chars.isEmpty()) {
				return new VideoBounds(0, 0);
			}

			initTACharacters(totals, chars, false, fillFont, calcVideoMediaTime(this));

			return calcBounds(chars, Resolution.FULL);
		}

		public IVideoBuffer getVideoFrame(Time mediaTime) {
			boolean fillOnly = (getTextType() == TextType.FILL_ONLY);

			String psName = _font.value(_context);
			int fontSize = _fontSize.value(_context);
			FTGLfont fillFont = getFillFont(psName, fontSize, true);
			FTGLfont strokeFont = fillOnly ? null : getStrokeFont(psName, fontSize);

			if (fillFont == null /*|| (!fillOnly && strokeFont == null)*/) {	// strokeFontが取得できない場合でもとりあえず続行する。
				return _support.createVideoBuffer(new VideoBounds(0, 0));
			}

			String sourceText = _sourceText.value(_context);

			int[] totals = new int[4];
			List<TACharacter> chars = createTACharacters(totals, sourceText, fillFont);

			if (chars.isEmpty()) {
				return _support.createVideoBuffer(new VideoBounds(0, 0));
			}

			initTACharacters(totals, chars, false, fillFont, mediaTime);

			if (fillOnly) {
				fillFont = getFillFont(psName, fontSize, false);
				if (fillFont == null) {
					return _support.createVideoBuffer(new VideoBounds(0, 0));
				}
			}

			Resolution resolution = _context.getVideoResolution();
			VideoBounds bounds = calcBounds(chars, resolution);

			IVideoBuffer buffer = null;
			try {
				buffer = _support.createVideoBuffer(bounds);
				buffer.clear();

				TAContext ctx = new TAContext();
				ctx.viewport = bounds;
				ctx.resolution = resolution;
				ctx.chars = chars;
				ctx.perCharacter3D = false;
				ctx.polygonFont = !fillOnly;
				ctx.fillFont = fillFont;
				ctx.strokeFont = strokeFont;

				render(ctx, buffer);

				IVideoBuffer result = buffer;
				buffer = null;
				return result;

			} finally {
				if (buffer != null) buffer.dispose();
			}
		}

		public void rasterize(IVideoBuffer buffer, double[] prjMatrix, double[] mvMatrix, Time mediaTime) {
			boolean fillOnly = (getTextType() == TextType.FILL_ONLY);

			String psName = _font.value(_context);
			int fontSize = _fontSize.value(_context);
			FTGLfont fillFont = getFillFont(psName, fontSize, true);
			FTGLfont strokeFont = fillOnly ? null : getStrokeFont(psName, fontSize);

			if (fillFont == null /*|| (!fillOnly && strokeFont == null)*/) {	// strokeFontが取得できない場合でもとりあえず続行する。
				return;
			}

			String sourceText = _sourceText.value(_context);

			int[] totals = new int[4];
			List<TACharacter> chars = createTACharacters(totals, sourceText, fillFont);

			if (chars.isEmpty()) {
				return;
			}

			initTACharacters(totals, chars, false, fillFont, mediaTime);

			TAContext ctx = new TAContext();
			ctx.viewport = buffer.getBounds();
			ctx.prjMatrix = prjMatrix;
			ctx.mvMatrix = mvMatrix;
			ctx.resolution = _context.getVideoResolution();
			ctx.chars = chars;
			ctx.perCharacter3D = false;
			ctx.polygonFont = true;
			ctx.fillFont = fillFont;
			ctx.strokeFont = strokeFont;

			for (Iterator<TACharacter> it = chars.iterator(); it.hasNext(); ) {
				TACharacter c = it.next();
				calcCharMatrix(c, false, ctx.resolution);
				clipCharBBox2(ctx, c);

				if (c.bbox2c.length == 0) {
					it.remove();
				}
			}

			render(ctx, buffer);
		}

		public IAudioBuffer getAudioChunk(Time mediaTime) {
			throw new UnsupportedOperationException("audio is not available");
		}

	}


	public static class AnimatableLineJoin
			extends AbstractAnimatableEnum<LineJoin> {

		public AnimatableLineJoin(
				LineJoin staticValue,
				Collection<Keyframe<LineJoin>> keyframes,
				String expression) {

			super(LineJoin.class, staticValue, keyframes, expression);
		}

		public AnimatableLineJoin(LineJoin defaultValue) {
			super(LineJoin.class, defaultValue);
		}

	}

	public static class AnimatableOrderOfFillAndStroke
			extends AbstractAnimatableEnum<OrderOfFillAndStroke> {

		public AnimatableOrderOfFillAndStroke(
				OrderOfFillAndStroke staticValue,
				Collection<Keyframe<OrderOfFillAndStroke>> keyframes,
				String expression) {

			super(OrderOfFillAndStroke.class, staticValue, keyframes, expression);
		}

		public AnimatableOrderOfFillAndStroke(OrderOfFillAndStroke defaultValue) {
			super(OrderOfFillAndStroke.class, defaultValue);
		}

	}

	public static class AnimatableHorizontalAlignment
			extends AbstractAnimatableEnum<HorizontalAlignment> {

		public AnimatableHorizontalAlignment(
				HorizontalAlignment staticValue,
				Collection<Keyframe<HorizontalAlignment>> keyframes,
				String expression) {

			super(HorizontalAlignment.class, staticValue, keyframes, expression);
		}

		public AnimatableHorizontalAlignment(HorizontalAlignment defaultValue) {
			super(HorizontalAlignment.class, defaultValue);
		}

	}

}

class CharacterGroup {

	static final int MIN_CODE_POINT = Character.MIN_CODE_POINT;
	static final int MAX_CODE_POINT = Character./*MAX_CODE_POINT*/MAX_VALUE;	// 0x10000以上はFTGLがクラッシュするので。

	static final int MIN_MAPPED_SYMBOL = 0x110000;
	static final int MAX_MAPPED_SYMBOL = MIN_MAPPED_SYMBOL + ('0'-'!') + ('A'-':') + ('a'-'[') + (0x7F-'{') - 1;
	static final int MIN_MAPPED_HIRAGANA = MAX_MAPPED_SYMBOL + 1;
	static final int MAX_MAPPED_HIRAGANA = MIN_MAPPED_HIRAGANA + (0x3097-0x3041) + (0x30A0-0x3099) - 1;

	static int map(int codePoint) {
		switch (BLOCKS.headSet(codePoint+1).last()) {
			case '!': return MIN_MAPPED_SYMBOL + (codePoint-'!');
			case ':': return MIN_MAPPED_SYMBOL + ('0'-'!') + (codePoint-':');
			case '[': return MIN_MAPPED_SYMBOL + ('0'-'!') + ('A'-':') + (codePoint-'[');
			case '{': return MIN_MAPPED_SYMBOL + ('0'-'!') + ('A'-':') + ('a'-'[') + (codePoint-'{');

			case 0x3041: return MIN_MAPPED_HIRAGANA + (codePoint-0x3041);
			case 0x3099: return MIN_MAPPED_HIRAGANA + (0x3097-0x3041) + (codePoint-0x3099);
		}

		return codePoint;
	}

	static int unmap(int codePoint) {
		if (codePoint >= MIN_MAPPED_SYMBOL && codePoint <= MAX_MAPPED_SYMBOL) {
			codePoint -= MIN_MAPPED_SYMBOL;
			codePoint += '!';
			if (codePoint >= '0') codePoint += (':'-'0');
			if (codePoint >= 'A') codePoint += ('['-'A');
			if (codePoint >= 'a') codePoint += ('{'-'a');

		} else if (codePoint >= MIN_MAPPED_HIRAGANA && codePoint <= MAX_MAPPED_HIRAGANA) {
			codePoint -= MIN_MAPPED_HIRAGANA;
			codePoint += 0x3041;
			if (codePoint >= 0x3097) codePoint += (0x3099-0x3097);
		}

		return codePoint;
	}

	static int[] rangeOf(int codePoint) {
		int start, next;

		if (codePoint >= MIN_CODE_POINT && codePoint <= MAX_CODE_POINT) {
			start = BLOCKS.headSet(codePoint+1).last();
			next = BLOCKS.tailSet(codePoint+1).first();

		} else if (codePoint >= MIN_MAPPED_SYMBOL && codePoint <= MAX_MAPPED_SYMBOL) {
			start = MIN_MAPPED_SYMBOL;
			next = MAX_MAPPED_SYMBOL + 1;

		} else if (codePoint >= MIN_MAPPED_HIRAGANA && codePoint <= MAX_MAPPED_HIRAGANA) {
			start = MIN_MAPPED_HIRAGANA;
			next = MAX_MAPPED_HIRAGANA + 1;
			
		} else {
			throw new IllegalArgumentException();
		}

		return new int[] { start, next };
	}

	static int[] fullRange() {
		return new int[] { MIN_CODE_POINT, MAX_CODE_POINT+1 };
	}

	/**
	 * @see http://www.unicode.org/Public/4.0-Update/Blocks-4.0.0.txt
	 */
	private static final TreeSet<Integer> BLOCKS = new TreeSet<Integer>(Arrays.asList(new Integer[] {
			0x0000, // 0000..007F; Basic Latin

			'!',
			'0',
			':',
			'A',
			'[',
			'a',
			'{',
			0x007F,

			0x0080, // 0080..00FF; Latin-1 Supplement
			0x0100, // 0100..017F; Latin Extended-A
			0x0180, // 0180..024F; Latin Extended-B
			0x0250, // 0250..02AF; IPA Extensions
			0x02B0, // 02B0..02FF; Spacing Modifier Letters
			0x0300, // 0300..036F; Combining Diacritical Marks
			0x0370, // 0370..03FF; Greek and Coptic
			0x0400, // 0400..04FF; Cyrillic
			0x0500, // 0500..052F; Cyrillic Supplementary
			0x0530, // 0530..058F; Armenian
			0x0590, // 0590..05FF; Hebrew
			0x0600, // 0600..06FF; Arabic
			0x0700, // 0700..074F; Syriac
			0x0750,
			0x0780, // 0780..07BF; Thaana
			0x07C0,
			0x0900, // 0900..097F; Devanagari
			0x0980, // 0980..09FF; Bengali
			0x0A00, // 0A00..0A7F; Gurmukhi
			0x0A80, // 0A80..0AFF; Gujarati
			0x0B00, // 0B00..0B7F; Oriya
			0x0B80, // 0B80..0BFF; Tamil
			0x0C00, // 0C00..0C7F; Telugu
			0x0C80, // 0C80..0CFF; Kannada
			0x0D00, // 0D00..0D7F; Malayalam
			0x0D80, // 0D80..0DFF; Sinhala
			0x0E00, // 0E00..0E7F; Thai
			0x0E80, // 0E80..0EFF; Lao
			0x0F00, // 0F00..0FFF; Tibetan
			0x1000, // 1000..109F; Myanmar
			0x10A0, // 10A0..10FF; Georgian
			0x1100, // 1100..11FF; Hangul Jamo
			0x1200, // 1200..137F; Ethiopic
			0x1380,
			0x13A0, // 13A0..13FF; Cherokee
			0x1400, // 1400..167F; Unified Canadian Aboriginal Syllabics
			0x1680, // 1680..169F; Ogham
			0x16A0, // 16A0..16FF; Runic
			0x1700, // 1700..171F; Tagalog
			0x1720, // 1720..173F; Hanunoo
			0x1740, // 1740..175F; Buhid
			0x1760, // 1760..177F; Tagbanwa
			0x1780, // 1780..17FF; Khmer
			0x1800, // 1800..18AF; Mongolian
			0x18B0,
			0x1900, // 1900..194F; Limbu
			0x1950, // 1950..197F; Tai Le
			0x1980,
			0x19E0, // 19E0..19FF; Khmer Symbols
			0x1A00,
			0x1D00, // 1D00..1D7F; Phonetic Extensions
			0x1D80,
			0x1E00, // 1E00..1EFF; Latin Extended Additional
			0x1F00, // 1F00..1FFF; Greek Extended
			0x2000, // 2000..206F; General Punctuation
			0x2070, // 2070..209F; Superscripts and Subscripts
			0x20A0, // 20A0..20CF; Currency Symbols
			0x20D0, // 20D0..20FF; Combining Diacritical Marks for Symbols
			0x2100, // 2100..214F; Letterlike Symbols
			0x2150, // 2150..218F; Number Forms
			0x2190, // 2190..21FF; Arrows
			0x2200, // 2200..22FF; Mathematical Operators
			0x2300, // 2300..23FF; Miscellaneous Technical
			0x2400, // 2400..243F; Control Pictures
			0x2440, // 2440..245F; Optical Character Recognition
			0x2460, // 2460..24FF; Enclosed Alphanumerics
			0x2500, // 2500..257F; Box Drawing
			0x2580, // 2580..259F; Block Elements
			0x25A0, // 25A0..25FF; Geometric Shapes
			0x2600, // 2600..26FF; Miscellaneous Symbols
			0x2700, // 2700..27BF; Dingbats
			0x27C0, // 27C0..27EF; Miscellaneous Mathematical Symbols-A
			0x27F0, // 27F0..27FF; Supplemental Arrows-A
			0x2800, // 2800..28FF; Braille Patterns
			0x2900, // 2900..297F; Supplemental Arrows-B
			0x2980, // 2980..29FF; Miscellaneous Mathematical Symbols-B
			0x2A00, // 2A00..2AFF; Supplemental Mathematical Operators
			0x2B00, // 2B00..2BFF; Miscellaneous Symbols and Arrows
			0x2C00,
			0x2E80, // 2E80..2EFF; CJK Radicals Supplement
			0x2F00, // 2F00..2FDF; Kangxi Radicals
			0x2FE0,
			0x2FF0, // 2FF0..2FFF; Ideographic Description Characters
			0x3000, // 3000..303F; CJK Symbols and Punctuation
			0x3040, // 3040..309F; Hiragana

			0x3041,
			0x3097,
			0x3099,

			0x30A0, // 30A0..30FF; Katakana
			0x3100, // 3100..312F; Bopomofo
			0x3130, // 3130..318F; Hangul Compatibility Jamo
			0x3190, // 3190..319F; Kanbun
			0x31A0, // 31A0..31BF; Bopomofo Extended
			0x31C0,
			0x31F0, // 31F0..31FF; Katakana Phonetic Extensions
			0x3200, // 3200..32FF; Enclosed CJK Letters and Months
			0x3300, // 3300..33FF; CJK Compatibility
			0x3400, // 3400..4DBF; CJK Unified Ideographs Extension A
			0x4DC0, // 4DC0..4DFF; Yijing Hexagram Symbols
			0x4E00, // 4E00..9FFF; CJK Unified Ideographs
			0xA000, // A000..A48F; Yi Syllables
			0xA490, // A490..A4CF; Yi Radicals
			0xA4D0,
			0xAC00, // AC00..D7AF; Hangul Syllables
			0xD7B0,
			0xD800, // D800..DB7F; High Surrogates
			0xDB80, // DB80..DBFF; High Private Use Surrogates
			0xDC00, // DC00..DFFF; Low Surrogates
			0xE000, // E000..F8FF; Private Use Area
			0xF900, // F900..FAFF; CJK Compatibility Ideographs
			0xFB00, // FB00..FB4F; Alphabetic Presentation Forms
			0xFB50, // FB50..FDFF; Arabic Presentation Forms-A
			0xFE00, // FE00..FE0F; Variation Selectors
			0xFE10,
			0xFE20, // FE20..FE2F; Combining Half Marks
			0xFE30, // FE30..FE4F; CJK Compatibility Forms
			0xFE50, // FE50..FE6F; Small Form Variants
			0xFE70, // FE70..FEFF; Arabic Presentation Forms-B
			0xFF00, // FF00..FFEF; Halfwidth and Fullwidth Forms
			0xFFF0, // FFF0..FFFF; Specials

			MAX_CODE_POINT+1 // 終端を表すダミーデータ

//			0x10000, // 10000..1007F; Linear B Syllabary
//			0x10080, // 10080..100FF; Linear B Ideograms
//			0x10100, // 10100..1013F; Aegean Numbers
//			0x10140,
//			0x10300, // 10300..1032F; Old Italic
//			0x10330, // 10330..1034F; Gothic
//			0x10350,
//			0x10380, // 10380..1039F; Ugaritic
//			0x103A0,
//			0x10400, // 10400..1044F; Deseret
//			0x10450, // 10450..1047F; Shavian
//			0x10480, // 10480..104AF; Osmanya
//			0x104B0,
//			0x10800, // 10800..1083F; Cypriot Syllabary
//			0x10840,
//			0x1D000, // 1D000..1D0FF; Byzantine Musical Symbols
//			0x1D100, // 1D100..1D1FF; Musical Symbols
//			0x1D200,
//			0x1D300, // 1D300..1D35F; Tai Xuan Jing Symbols
//			0x1D360,
//			0x1D400, // 1D400..1D7FF; Mathematical Alphanumeric Symbols
//			0x1D800,
//			0x20000, // 20000..2A6DF; CJK Unified Ideographs Extension B
//			0x2A6E0,
//			0x2F800, // 2F800..2FA1F; CJK Compatibility Ideographs Supplement
//			0x2FA20,
//			0xE0000, // E0000..E007F; Tags
//			0xE0080,
//			0xE0100, // E0100..E01EF; Variation Selectors Supplement
//			0xE01F0,
//			0xF0000, // F0000..FFFFF; Supplementary Private Use Area-A
//			0x100000 // 100000..10FFFF; Supplementary Private Use Area-B
	}));

}

class HSLUtil {

	static Color addHSL(Color color, double hue, double sat, double lum) {
		hue = (hue(color) + hue) % 1;
		if (hue < 0) hue += 1;
		sat = clamp(sat(color) + sat, 0, 1);
		lum = clamp(lum(color) + lum, 0, 1);

		Color tmp = hsl2rgb(new double[] { hue, 1.0, 0.5 });
		return set_lum(set_sat(tmp, sat), lum);
	}

	private static double clamp(double value, double a, double b) {
		return Math.min(Math.max(value, Math.min(a, b)), Math.max(a, b));
	}

	private static double min(double a, double b, double c) {
		return Math.min(Math.min(a, b), c);
	}

	private static double max(double a, double b, double c) {
		return Math.max(Math.max(a, b), c);
	}

	private static double sat(Color color) {
		return max(color.r, color.g, color.b) - min(color.r, color.g, color.b);
	}

	private static double lum(Color color) {
		return 0.299*color.r + 0.587*color.g + 0.114*color.b;
	}

	private static Color set_sat(Color color, double s) {
		double r = color.r;
		double g = color.g;
		double b = color.b;
		double n = min(r, g, b);
		double x = max(r, g, b);
		if (x > n) {
			if (x == r) {
				if (n == g) {
					b = (b-n) * s / (x-n);
					g = 0.0;
				} else {
					g = (g-n) * s / (x-n);
					b = 0.0;
				}
				r = s;
			} else if (x == g) {
				if (n == b) {
					r = (r-n) * s / (x-n);
					b = 0.0;
				} else {
					b = (b-n) * s / (x-n);
					r = 0.0;
				}
				g = s;
			} else {
				if (n == r) {
					g = (g-n) * s / (x-n);
					r = 0.0;
				} else {
					r = (r-n) * s / (x-n);
					g = 0.0;
				}
				b = s;
			}
			return new Color(r, g, b);
		} else {
			return Color.BLACK;
		}
	}

	private static Color set_lum(Color color, double l) {
		double d = l - lum(color);
		return clipColor(color.r + d, color.g + d, color.b + d);
	}

	private static Color clipColor(double r, double g, double b) {
		double l = clamp(0.299*r + 0.587*g + 0.114*b, 0, 1);
		double n = min(r, g, b);
		if (n < 0.0) {
			r = l + (r-l) * l / (l-n);
			g = l + (g-l) * l / (l-n);
			b = l + (b-l) * l / (l-n);
		}
		double x = max(r, g, b);
		if (x > 1.0) {
			r = l + (r-l) * (1.0-l) / (x-l);
			g = l + (g-l) * (1.0-l) / (x-l);
			b = l + (b-l) * (1.0-l) / (x-l);
		}
		return new Color(r, g, b);
	}

	// 以下、HSLKeyなどにもほぼ同じものがある。

	private static double hue(Color rgb) {
		double min = min(rgb.r, rgb.g, rgb.b);
		double max = max(rgb.r, rgb.g, rgb.b);
		double dmax = max - min;

		if (dmax == 0) {
			return 0;
		}

		double dr = ((max-rgb.r)/6 + dmax/2)/dmax;
		double dg = ((max-rgb.g)/6 + dmax/2)/dmax;
		double db = ((max-rgb.b)/6 + dmax/2)/dmax;

		double hue = (rgb.r == max) ? db-dg
				   : (rgb.g == max) ? 1./3 + dr-db
				   :				  2./3 + dg-dr;

		hue %= 1;
		if (hue < 0) hue += 1;

		return hue;
	}

	private static Color hsl2rgb(double[] hsl) {
		double hue = hsl[0];
		double sat = hsl[1];
		double lum = hsl[2];

		if (sat == 0) {
			return new Color(lum, lum, lum);
		} else {
			double t2 = (lum < 0.5) ? lum*(1+sat) : lum+sat-lum*sat;
			double t1 = lum*2 - t2;

			return new Color(
					hue2rgb(t1, t2, hue+1./3),
					hue2rgb(t1, t2, hue),
					hue2rgb(t1, t2, hue-1./3));
		}
	}

	private static double hue2rgb(double t1, double t2, double hue) {
		hue %= 1;
		if (hue < 0) hue += 1;

		return (hue*6 < 1) ? t1 + (t2-t1)*6*hue
			 : (hue*2 < 1) ? t2
			 : (hue*3 < 2) ? t1 + (t2-t1)*(2./3-hue)*6
			 :				 t1;
	}

}
