/*
 * Copyright (c) 2009 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.views;

import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;

import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.TreePath;
import org.eclipse.jface.viewers.TreeSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;

import ch.kuramo.javie.app.project.ProjectEvent;
import ch.kuramo.javie.app.project.ProjectListener;
import ch.kuramo.javie.app.project.ProjectManager;
import ch.kuramo.javie.core.Folder;
import ch.kuramo.javie.core.Item;
import ch.kuramo.javie.core.Util;

public class ItemTreeContentProvider implements ITreeContentProvider, ProjectListener {

	private TreeViewer _viewer;

	private ProjectManager _projectManager;


	public void dispose() {
		_viewer = null;
		_projectManager = null;
	}

	public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
		if (_projectManager != null) {
			_projectManager.removeProjectListener(this);
		}

		_viewer = (TreeViewer) viewer;
		_projectManager = (ProjectManager) newInput;

		if (_projectManager != null) {
			_projectManager.addProjectListener(this);
		}
	}

	public boolean hasChildren(Object element) {
		return (element instanceof Folder)
				? _projectManager.hasChildren((Folder) element) : false;
	}

	public Object[] getChildren(Object parentElement) {
		return (parentElement instanceof Folder)
				? _projectManager.listChildren((Folder) parentElement).toArray() : null;
	}

	public Object[] getElements(Object inputElement) {
		return _projectManager.listChildren(null).toArray();
	}

	public Object getParent(Object element) {
		// elementが最上位のアイテムの場合、返す値はnullでいいのか_projectManagerを返すべきなのか？
		// ITreeContentProvider#getParentの説明によると、
		//   null indicating that the parent can't be computed.
		//   In this case the tree-structured viewer can't expand
		//   a given node correctly if requested.
		// とのことなのでnullを返すのは間違いな気がするが、
		// 最上位のアイテムではこのメソッドが呼ばれないようでもある。
		//
		// 以下の処理では最上位のアイテムの時にnullを返すが、
		// 最上位のアイテムでこのメソッドが呼ばれないのならこれでよい。

		return ((Item) element).getParent();
	}

	public void handleProjectEvent(ProjectEvent event) {
		if (event.getProjectManager() != _projectManager) {
			return;
		}

		switch (event.type) {
			case ITEM_NAME_CHANGE:
			case ITEM_UPDATE:
				update(ProjectEvent.getItem(event));
				break;

			case ITEMS_ADD:
			case ITEMS_REPARENT:
				refresh(ProjectEvent.getItems(event), ProjectEvent.getRelatedItems(event));
				break;

			case ITEMS_REMOVE:
				refresh(Collections.<Item>emptySet(), ProjectEvent.getRelatedItems(event));
				break;
		}
	}

	private void update(Item item) {
		_viewer.update(item, null);
		select(Collections.singleton(item));
	}

	private void refresh(Set<Item> items, Set<Item> relatedItems) {
		if (relatedItems.isEmpty()) {
			_viewer.refresh();
		} else {
			for (Item i : relatedItems) {
				_viewer.refresh(i);
			}
		}
		select(items);
	}

	private void select(Set<Item> items) {
		List<TreePath> list = Util.newList();
		for (Item i : items) {
			LinkedList<Item> segmentList = new LinkedList<Item>();
			do {
				segmentList.addFirst(i);
			} while ((i = i.getParent()) != null);
			list.add(new TreePath(segmentList.toArray()));
		}
		_viewer.setSelection(new TreeSelection(list.toArray(new TreePath[list.size()])), true);
	}

}
