/*
 * Copyright (c) 2009,2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.project;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;

import ch.kuramo.javie.app.PropertyUtil;
import ch.kuramo.javie.core.AnimatableValue;
import ch.kuramo.javie.core.Effect;
import ch.kuramo.javie.core.EffectableLayer;
import ch.kuramo.javie.core.Keyframe;
import ch.kuramo.javie.core.Layer;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.Project;

public class RemoveKeyframesOperation extends ProjectOperation {

	private final String _compId;

	private final Object[][] _data;


	public RemoveKeyframesOperation(ProjectManager projectManager, LayerComposition comp, Object[][] data) {
		super(projectManager, "キーフレームの削除");
		projectManager.checkComposition(comp);
		_compId = comp.getId();
		_data = data;
	}

	@Override
	protected IStatus executeOrRedo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		removeOrRestore(true, project, pm);
		return Status.OK_STATUS;
	}

	@Override
	protected IStatus undo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		removeOrRestore(false, project, pm);
		return Status.OK_STATUS;
	}

	private void removeOrRestore(boolean remove, Project project, ProjectManager pm) {
		LayerComposition comp = project.getComposition(_compId);
		Object[][] data = (pm != null) ? new Object[_data.length][] : null;

		for (int i = 0, n = _data.length; i < n; ++i) {
			Layer layer = comp.getLayer((String) _data[i][0]);
			Integer effectIndex = (Integer) _data[i][1];
			String property = (String) _data[i][2];
			@SuppressWarnings("unchecked")
			Keyframe<Object> kf = (Keyframe<Object>) _data[i][3];

			AnimatableValue<Object> avalue;
			if (effectIndex == -1) {
				avalue = PropertyUtil.getProperty(layer, property);
			} else {
				Effect effect = ((EffectableLayer) layer).getEffects().get(effectIndex);
				avalue = PropertyUtil.getProperty(effect, property);
			}

			if (remove) {
				avalue.removeKeyframe(kf.time);
				if (data != null) {
					data[i] = new Object[] { layer, effectIndex, property, null };
				}
			} else {
				avalue.putKeyframe(kf);
				if (data != null) {
					data[i] = new Object[] { layer, effectIndex, property, kf.time };
				}
			}
		}

		if (pm != null) {
			pm.fireKeyframesChange(comp, data);
		}
	}

}
