/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.project;

import java.util.Collection;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.ListIterator;
import java.util.Set;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;

import ch.kuramo.javie.app.InjectorHolder;
import ch.kuramo.javie.core.Composition;
import ch.kuramo.javie.core.CompositionItem;
import ch.kuramo.javie.core.Folder;
import ch.kuramo.javie.core.Item;
import ch.kuramo.javie.core.ItemLayer;
import ch.kuramo.javie.core.Layer;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.ProjectDecodeException;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.services.ProjectDecoder;
import ch.kuramo.javie.core.services.ProjectEncoder;

import com.google.inject.Inject;

public class RemoveItemsOperation extends ProjectOperationCollection {

	public RemoveItemsOperation(final ProjectManager pm, Collection<Item> items) {
		super(pm, "アイテムの削除");

		final LinkedHashSet<Item> items2 = Util.newLinkedHashSet();
		for (Item item : items) {
			pm.checkItem(item);

			if (item instanceof Folder) {
				new Object() {
					void addFolder(Folder folder) {
						for (Item item : pm.listChildren(folder)) {
							if (item instanceof Folder) {
								addFolder((Folder) item);
							} else {
								items2.add(item);
							}
						}
						items2.add(folder);
					}
				}.addFolder((Folder) item);
			} else {
				items2.add(item);
			}
		}

		for (Composition comp : pm.getProject().getCompositions()) {
			if (comp instanceof LayerComposition) {
				LayerComposition layerComp = (LayerComposition) comp;
				List<Layer> layers = Util.newList();

				for (Layer l : layerComp.getLayers()) {
					if (l instanceof ItemLayer<?> && items2.contains(((ItemLayer<?>) l).getItem())) {
						layers.add(l);
					}
				}

				if (!layers.isEmpty()) {
					add(new RemoveLayersOperation(pm, layers));
				}
			}
		}

		add(new RemoveItemsOperation2(pm, items2));
	}

}

class RemoveItemsOperation2 extends ProjectOperation {

	private final List<Object[]> itemIndexAndData = Util.newList();

	private List<Object[]> compIndexAndData;

	@Inject
	private ProjectEncoder encoder;

	@Inject
	private ProjectDecoder decoder;


	/**
	 * @param pm
	 * @param items このコレクションの順で削除する。したがって、フォルダは内包するどのアイテムよりも後方に挿入されている必要がある。
	 */
	RemoveItemsOperation2(ProjectManager pm, Collection<Item> items) {
		super(pm, "アイテムの削除");
		InjectorHolder.getInjector().injectMembers(this);

		List<Item> tmp = Util.newList(pm.getProject().getItems());
		for (Item item : items) {
			int i = tmp.indexOf(item);
			if (i == -1) {
				throw new IllegalArgumentException("item not found");
			}
			itemIndexAndData.add(new Object[] { i, encoder.encodeElement(item) });
			tmp.remove(i);
		}
	}

	@Override
	protected IStatus executeOrRedo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		boolean encodeComp = false;
		if (compIndexAndData == null) {
			compIndexAndData = Util.newList();
			encodeComp = true;
		}

		List<Item> items = Util.newList();
		for (Object[] o : itemIndexAndData) {
			Item item = project.getItems().get((Integer) o[0]);

			if (item instanceof CompositionItem && encodeComp) {
				Composition comp = ((CompositionItem) item).getComposition();
				int compIndex = project.getCompositions().indexOf(comp);
				compIndexAndData.add(new Object[] { compIndex, encoder.encodeElement(comp) });
			}

			project.getItems().remove(item);
			items.add(item);
		}

		for (Object[] o : compIndexAndData) {
			int compIndex = (Integer) o[0];
			project.getCompositions().remove(compIndex);
		}

		if (pm != null) {
			fireItemsRemove(items, project, pm);
		}

		disposeAll(items);
		return Status.OK_STATUS;
	}

	@Override
	protected IStatus undo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		List<Composition> comps = Util.newList();
		for (ListIterator<Object[]> it = compIndexAndData.listIterator(compIndexAndData.size()); it.hasPrevious(); ) {
			Object[] o = it.previous();
			Composition comp;
			try {
				comp = decoder.decodeElement((String) o[1], Composition.class);
				comps.add(comp);
				comp.afterDecode(project);
			} catch (ProjectDecodeException e) {
				project.getCompositions().removeAll(comps);
				throw new ExecutionException("error decoding Composition data", e);
			}
			project.getCompositions().add((Integer) o[0], comp);
		}

		List<Item> items = Util.newList();
		for (ListIterator<Object[]> it = itemIndexAndData.listIterator(itemIndexAndData.size()); it.hasPrevious(); ) {
			Object[] o = it.previous();
			Item item;
			try {
				item = decoder.decodeElement((String) o[1], Item.class);
				items.add(item);
				item.afterDecode(project);
			} catch (ProjectDecodeException e) {
				project.getItems().removeAll(items);
				project.getCompositions().removeAll(comps);
				disposeAll(items);
				throw new ExecutionException("error decoding Item data", e);
			}
			project.getItems().add((Integer) o[0], item);
		}

		if (pm != null) {
			fireItemsAdd(items, project, pm);
		}

		return Status.OK_STATUS;
	}

	private void fireItemsAdd(Collection<Item> items, Project project, ProjectManager pm) {
		Set<Item> parents = Util.newSet();
		for (Item item : items) {
			Item parent = item.getParent();
			if (parent != null) {
				parents.add(parent);
			} else {
				parents.clear();
				break;
			}
		}

		if (parents.isEmpty()) {
			pm.fireItemsAdd(items, null);
		} else {
			pm.fireItemsAdd(items, parents);
		}
	}

	private void fireItemsRemove(Collection<Item> items, Project project, ProjectManager pm) {
		Set<Item> parents = Util.newSet();
		for (Item item : items) {
			Item parent = item.getParent();
			if (parent != null) {
				parents.add(parent);
			} else {
				parents.clear();
				break;
			}
		}
		parents.retainAll(project.getItems());

		if (parents.isEmpty()) {
			pm.fireItemsRemove(items, null);
		} else {
			pm.fireItemsRemove(items, parents);
		}
	}

	private void disposeAll(Collection<Item> items) {
		for (Item item : items) {
			item.dispose();
		}
	}

}
