/*
 * Copyright (c) 2009 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.project;

import java.util.Collections;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;

import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.app.InjectorHolder;
import ch.kuramo.javie.core.Layer;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.ProjectDecodeException;
import ch.kuramo.javie.core.services.ProjectDecoder;
import ch.kuramo.javie.core.services.ProjectElementFactory;
import ch.kuramo.javie.core.services.ProjectEncoder;

import com.google.inject.Inject;

public abstract class AbstractNewLayerOperation<T extends Layer> extends ProjectOperation {

	private final String _compId;

	private final String _layerId;

	private final String _data;

	@Inject
	private ProjectElementFactory _elementFactory;

	@Inject
	private ProjectEncoder _encoder;

	@Inject
	private ProjectDecoder _decoder;


	public AbstractNewLayerOperation(
			ProjectManager projectManager, LayerComposition comp, String label) {

		super(projectManager, label);
		InjectorHolder.getInjector().injectMembers(this);

		projectManager.checkComposition(comp);
		_compId = comp.getId();

		T layer = createNewLayer(projectManager, comp, _elementFactory);

		Time time0 = Time.fromFrameNumber(0, comp.getFrameDuration());
		layer.setStartTime(time0);
		layer.setInPoint(time0);
		layer.setOutPoint(comp.getDuration());

		_layerId = layer.getId();
		_data = _encoder.encodeElement(layer);
	}

	protected abstract T createNewLayer(
			ProjectManager projectManager, LayerComposition comp, ProjectElementFactory factory);

	@Override
	protected IStatus executeOrRedo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		LayerComposition comp = project.getComposition(_compId);

		Layer layer;
		try {
			layer = _decoder.decodeElement(_data, Layer.class);
			layer.afterDecode(project, comp);
		} catch (ProjectDecodeException e) {
			throw new ExecutionException("error decoding layer data", e);
		}

		comp.getLayers().add(layer);

		if (pm != null) {
			pm.fireLayersAdd(comp, Collections.singleton(layer));
		}

		return Status.OK_STATUS;
	}

	@Override
	protected IStatus undo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		LayerComposition comp = project.getComposition(_compId);

		Layer layer = comp.getLayer(_layerId);
		comp.getLayers().remove(layer);

		if (pm != null) {
			pm.fireLayersRemove(comp, Collections.singleton(layer));
		}

		return Status.OK_STATUS;
	}

}
