/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.actions;

import java.io.File;
import java.lang.reflect.InvocationTargetException;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.dialogs.ProgressMonitorDialog;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.jface.wizard.WizardDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.ui.IWorkbenchWindow;

import ch.kuramo.javie.app.InjectorHolder;
import ch.kuramo.javie.app.project.ProjectManager;
import ch.kuramo.javie.app.views.LayerCompositionView;
import ch.kuramo.javie.core.Composition;
import ch.kuramo.javie.core.CompositionItem;
import ch.kuramo.javie.core.Item;
import ch.kuramo.javie.core.JavieRuntimeException;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.ProjectDecodeException;
import ch.kuramo.javie.core.output.Output;
import ch.kuramo.javie.core.output.ProgressMonitor;
import ch.kuramo.javie.core.services.ProjectDecoder;
import ch.kuramo.javie.core.services.ProjectEncoder;

import com.google.inject.Inject;
import com.google.inject.Injector;

public abstract class AbstractOutputAction<T extends Output> extends Action {

	private static File defaultFolder;


	private final LayerCompositionView view;

	@Inject
	private Injector injector;

	@Inject
	private ProjectEncoder encoder;

	@Inject
	private ProjectDecoder decoder;


	public AbstractOutputAction(String text, LayerCompositionView view) {
		super(text);
		InjectorHolder.getInjector().injectMembers(this);
		this.view = view;
	}

	protected IWorkbenchWindow getWindow() {
		return view.getSite().getWorkbenchWindow();
	}

	protected Composition getComposition() {
		return view.getCompositionItem().getComposition();
	}

	public void run() {
		ProjectManager pm = ProjectManager.forWorkbenchWindow(getWindow());
		if (pm == null) {
			return;
		}

		T output = createOutput(injector);

		Wizard wizard = createWizard(output);
		if (wizard != null) {
			WizardDialog dialog = new WizardDialog(getWindow().getShell(), wizard);
			if (dialog.open() != WizardDialog.OK) {
				return;
			}
		}

		if (defaultFolder == null) {
			File file = pm.getFile();
			if (file != null) {
				defaultFolder = file.getParentFile();
			}
		}

		CompositionItem compItem = view.getCompositionItem();

		File file = showSaveDialog(defaultFolder, compItem.getName());
		if (file != null) {
			output.setFile(file);
			defaultFolder = file.getParentFile();
			doOutput(pm.getProject(), compItem.getId(), output);
		}
	}

	private File showSaveDialog(File folder, String compositionName) {
		String fileName = compositionName;
		String defaultExtension = getDefaultExtension();
		if (defaultExtension != null && defaultExtension.length() > 0) {
			if (defaultExtension.startsWith(".")) {
				fileName += defaultExtension;
			} else {
				fileName += "." + defaultExtension;
			}
		}

		String[] filters = getSaveDialogFilters();
		String[] filterNames = new String[filters.length/2];
		String[] filterExtensions = new String[filters.length/2];
		String platform = SWT.getPlatform();
		boolean windows = platform.equals("win32") || platform.equals("wpf");
		for (int i = 0; i < filterNames.length; ++i) {
			filterNames[i] = filters[i*2];
			filterExtensions[i] = filters[i*2+1];
			if (windows && filterExtensions[i].equals("*")) {
				filterExtensions[i] = "*.*";
			}
		}

		FileDialog dialog = new FileDialog(getWindow().getShell(), SWT.SAVE /*| SWT.SHEET*/);
		dialog.setFilterNames(filterNames);
		dialog.setFilterExtensions(filterExtensions);
		dialog.setFilterPath(folder != null ? folder.getAbsolutePath() : null);
		dialog.setFileName(fileName);
		dialog.setOverwrite(isPromptedForFileOverwrite());

		String path = dialog.open();
		return (path != null) ? new File(path) : null;
	}

	private void doOutput(Project project, String compItemId, final T output) {
		Project copy = null;
		try {
			copy = decoder.decodeElement(encoder.encodeElement(project), Project.class);
			copy.afterDecode();

			CompositionItem compItem = (CompositionItem) copy.getItem(compItemId);
			output.setComposition(compItem.getComposition());

			ProgressMonitorDialog dialog = new ProgressMonitorDialog(getWindow().getShell());
			dialog.create();
			dialog.getShell().setText("書き出し");
			dialog.run(true, true, new IRunnableWithProgress() {
				public void run(IProgressMonitor monitor) throws InvocationTargetException, InterruptedException {
					output.doOutput(createProgressMonitor(monitor));
					if (monitor.isCanceled()) throw new InterruptedException();
				}
			});

		} catch (ProjectDecodeException e) {
			throw new JavieRuntimeException(e);
		} catch (InvocationTargetException e) {
			throw new JavieRuntimeException(e);
		} catch (InterruptedException e) {
			// ユーザーがキャンセルした場合
		} finally {
			if (copy != null) {
				// TODO Project#dispose メソッドを作る。
				for (Item i : copy.getItems()) {
					i.dispose();
				}
			}
		}
	}

	private ProgressMonitor createProgressMonitor(final IProgressMonitor monitor) {
		return new ProgressMonitor() {
			private int totalWork;
			private int currentWork;

			public void beginTask(int totalWork) {
				monitor.beginTask(String.format("書き出し: %s", view.getCompositionItem().getName()), totalWork);
				this.totalWork = totalWork;
			}

			public void currentWork(int currentWork) {
				if (currentWork > this.currentWork) {
					monitor.worked(currentWork - this.currentWork);
					monitor.subTask(getSubTaskText(currentWork, totalWork));
					this.currentWork = currentWork;
				}
			}

			public void done() {
				monitor.done();
			}

			public boolean isCanceled() {
				return monitor.isCanceled();
			}
		};
	}

	protected abstract T createOutput(Injector injector);

	protected Wizard createWizard(T output) {
		return null;
	}

	protected abstract String getDefaultExtension();

	protected String[] getSaveDialogFilters() {
		return new String[] { "すべてのファイル", "*" };
	}

	protected boolean isPromptedForFileOverwrite() {
		return true;
	}

	protected String getSubTaskText(int currentWork, int totalWork) {
		return String.format("%.1f%%", 100.0 * currentWork / totalWork);
	}

}
