/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.api;

/**
 * <p><code>Time</code> クラスは時間を表現します。
 * ここでの「時間」とは、特定の時刻や時間の長さなどの複数の意味を含んでおり、
 * <code>Time</code> クラスが使われる文脈によってどの意味を指すかが異なります。</p>
 * 
 * <p><code>Time</code> オブジェクトは、<code>timeValue</code> と <code>timeScale</code>
 * の2つの値を保持します。<code>timeValue</code> は時間の値です。
 * <code>timeScale</code> は <code>timeValue</code> が表す値の単位で、
 * 1秒あたりの <code>timeValue</code> の数です。したがって、<code>timeValue</code> を
 * <code>timeScale</code> で割った値は秒単位の時間を表します。
 * <code>timeValue</code> は long 値、<code>timeScale</code> は int 値です。</p>
 * 
 * <p><code>Time</code> オブジェクトは不変です。
 * <code>Time</code> オブジェクトは作成したあとに変更できないため、共用することができます。</p>
 */
public final class Time implements Comparable<Time> {

	/**
	 * よく使用されるフレームレートで共通のタイムスケールです。
	 * 例えば 25fps, 29.97fps, 30fps, 59.94fps, 60fps などは、
	 * このタイムスケールを使用して正確に時刻を表すことができます。
	 */
	public static final int COMMON_TIME_SCALE = 120000;

	/**
	 * 時間の値です。
	 */
	public final long timeValue;

	/**
	 * <code>timeValue</code> が表す値の単位です。
	 */
	public final int timeScale;

	/**
	 * 秒単位で表した時間です。
	 */
	private final double second;

	/**
	 * この <code>Time</code> オブジェクトのハッシュコードです。
	 */
	private final int hashCode;


	/**
	 * <code>timeValue</code> と <code>timeScale</code> の値を指定して
	 * <code>Time</code> オブジェクトを生成します。
	 * <code>timeScale</code> を <code>120000</code> に変換可能な場合は変換します。
	 * 
	 * @param timeValue 時間の値
	 * @param timeScale <code>timeValue</code> が表す値の単位
	 */
	public Time(long timeValue, int timeScale) {
		super();

		// タイムスケールを120000に変換可能な場合は変換する。
		int q = COMMON_TIME_SCALE/timeScale;
		if (q*timeScale == COMMON_TIME_SCALE) {
			timeValue *= q;
			timeScale = COMMON_TIME_SCALE;
		}

		this.timeValue = timeValue;
		this.timeScale = timeScale;

		second = (double) timeValue / timeScale;
		hashCode = (int) (timeValue * 1000 / timeScale);
	}

	/**
	 * フレーム番号 <code>frameNumber</code> のフレームが開始する時刻を返します。
	 * これは、フレーム番号 <code>0</code> のフレームが開始する時刻を起点とし、
	 * フレーム継続時間を <code>frameDuration</code> とした場合の値です。 
	 * <code>frameNumber</code> に負の値を指定することもできます。
	 * 
	 * @param frameNumber	フレーム番号
	 * @param frameDuration	フレーム継続時間
	 * @return フレーム番号 <code>frameNumber</code> のフレームが開始する時刻
	 */
	public static Time fromFrameNumber(long frameNumber, Time frameDuration) {
		return new Time(frameNumber * frameDuration.timeValue, frameDuration.timeScale);
	}

	/**
	 * この <code>Time</code> オブジェクトが表す時刻におけるフレーム番号を返します。
	 * これは、フレーム番号 <code>0</code> のフレームが開始する時刻を起点とし、
	 * フレーム継続時間を <code>frameDuration</code> とした場合の値です。 
	 * 
	 * @param frameDuration フレーム継続時間
	 * @return この <code>Time</code> オブジェクトが表す時刻におけるフレーム番号
	 */
	public long toFrameNumber(Time frameDuration) {
		if (this.timeScale == frameDuration.timeScale) {
			return (long) Math.floor((double) this.timeValue / frameDuration.timeValue);
		} else {
			return (long) Math.floor(this.second / frameDuration.second);
		}
	}

	/**
	 * この <code>Time</code> オブジェクトが表す時間を、秒単位の値で返します。
	 * 
	 * @return この <code>Time</code> オブジェクトが表す時間の秒単位の値
	 */
	public double toSecond() {
		return second;
	}

	/**
	 * この <code>Time</code> オブジェクトが引数 <code>o</code>
	 * よりも前（過去の時刻、時間が短いなど）かどうかを判定します。
	 * 
	 * @param o 比較対象の <code>Time</code> オブジェクト
	 * @return	この <code>Time</code> オブジェクトが引数 <code>o</code>
	 * 			よりも前の場合は true、そうでない場合は false
	 */
	public boolean before(Time o) {
		return (compareTo(o) < 0);
	}

	/**
	 * この <code>Time</code> オブジェクトが引数 <code>o</code>
	 * よりも後（未来の時刻、時間が長いなど）かどうかを判定します。
	 * 
	 * @param o 比較対象の <code>Time</code> オブジェクト
	 * @return	この <code>Time</code> オブジェクトが引数 <code>o</code>
	 * 			よりも後の場合は true、そうでない場合は false
	 */
	public boolean after(Time o) {
		return (compareTo(o) > 0);
	}

	/**
	 * 順序付けのために、この <code>Time</code> オブジェクトと引数 <code>o</code> を比較します。
	 * 
	 * @param o 比較対象の <code>Time</code> オブジェクト
	 * @return	この <code>Time</code> オブジェクトと引数 <code>o</code> が等しい場合は <code>0</code>、
	 * 			この <code>Time</code> オブジェクトが引数 <code>o</code> より前の場合は <code>0</code> より小さい値、
	 * 			この <code>Time</code> オブジェクトが引数 <code>o</code> より後の場合は <code>0</code> より大きい値
	 */
	public int compareTo(Time o) {
		if (this.timeScale == o.timeScale) {
			long diff = this.timeValue - o.timeValue;
			return (diff > 0) ? 1 : (diff < 0) ? -1 : 0;
		} else {
			double diff = this.second - o.second;
			return (diff > 0) ? 1 : (diff < 0) ? -1 : 0;
		}
	}

	/**
	 * この <code>Time</code> オブジェクトに引数 <code>addend</code> を加えた値を返します。
	 * 返される値の <code>timeScale</code> はこの <code>Time</code> オブジェクトの <code>timeScale</code> と同じになります。
	 * 
	 * @param addend この <code>Time</code> オブジェクトに加える値
	 * @return この <code>Time</code> オブジェクトに引数 <code>addend</code> を加えた値
	 */
	public Time add(Time addend) {
		if (addend.timeScale == timeScale) {
			return new Time(timeValue + addend.timeValue, timeScale);
		} else {
			return new Time((long) Math.floor((this.second + addend.second)*timeScale), timeScale);
		}
	}

	/**
	 * この <code>Time</code> オブジェクトから引数 <code>subtrahend</code> を引いた値を返します。
	 * 返される値の <code>timeScale</code> はこの <code>Time</code> オブジェクトの <code>timeScale</code> と同じになります。
	 * 
	 * @param subtrahend この <code>Time</code> オブジェクトから引く値
	 * @return この <code>Time</code> オブジェクトから引数 <code>subtrahend</code> を引いた値
	 */
	public Time subtract(Time subtrahend) {
		if (subtrahend.timeScale == timeScale) {
			return new Time(timeValue - subtrahend.timeValue, timeScale);
		} else {
			return new Time((long) Math.floor((this.second - subtrahend.second)*timeScale), timeScale);
		}
	}

	/*
	 * (non-Javadoc)
	 * @see java.lang.Object#hashCode()
	 */
	@Override
	public int hashCode() {
		return hashCode;
	}

	/*
	 * (non-Javadoc)
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;

		return (compareTo((Time) obj) == 0);
	}

	/*
	 * (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		return String.valueOf(second);
	}

}
