/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal.services;

import java.util.Set;
import java.util.concurrent.locks.ReentrantLock;

import javax.media.opengl.GL;
import javax.media.opengl.GL2;
import javax.media.opengl.GLContext;
import javax.media.opengl.glu.GLU;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ch.kuramo.javie.api.AudioMode;
import ch.kuramo.javie.api.ColorMode;
import ch.kuramo.javie.api.Resolution;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.core.Camera;
import ch.kuramo.javie.core.Light;
import ch.kuramo.javie.core.WrappedOperation;
import ch.kuramo.javie.core.services.GLGlobal;
import ch.kuramo.javie.core.services.RenderContext;

import com.google.inject.Inject;

public class RenderContextImpl extends AbstractCoreContext implements RenderContext {

	private static final Logger _logger = LoggerFactory.getLogger(RenderContextImpl.class);

	@Inject
	private GLGlobal _glGlobal;

	@Inject
	private AntiAliasSupportProxy.Lifecycle _aaSupportLifecycle;

	@Inject
	private FontManagerProxy.Lifecycle _fontManagerLifecycle;


	private GLContext _glContext;

	private GLU _glu;

	private int _fboId;

	private Resolution _videoResolution;

	private ColorMode _colorMode;

	private Time _videoFrameDuration;

	private Camera _camera;

	private Set<Light> _lights;

	private AudioMode _audioMode;

	private int _audioAnimationRate;

	private int _audioFrameCount;


	protected void finalize() throws Throwable {
		if (_glContext != null) {
			// TODO 解放忘れの場合どうする？
			//		finalize は別のスレッドから呼ばれるので、ここで deactivate するわけにはいかない。
			_logger.warn("finalizing a VideoRenderContextImpl object, but the object is not deactivated.");
		}
		super.finalize();
	}

	private void initGL() {
		if (_glContext != null) {
			return;
		}

		ReentrantLock lock = _glGlobal.getGlobalLock();
		lock.lock();
		try {
			_glContext = _glGlobal.createContext();
			_glContext.makeCurrent();
		} finally {
			lock.unlock();
		}

		//_glContext.setGL(new DebugGL2(_glContext.getGL().getGL2()));
		GL2 gl = _glContext.getGL().getGL2();

		_glu = GLU.createGLU(gl);

		int[] fboId = new int[1];
		gl.glGenFramebuffers(1, fboId, 0);
		_fboId = fboId[0];

		gl.glBindFramebuffer(GL2.GL_FRAMEBUFFER, _fboId);

		_aaSupportLifecycle.createImpl();
		_fontManagerLifecycle.createImpl();
	}

	public void deactivate() {
		if (_glContext != null) {
			_fontManagerLifecycle.disposeImpl();
			_aaSupportLifecycle.disposeImpl();

			GL2 gl = _glContext.getGL().getGL2();

			gl.glBindFramebuffer(GL2.GL_FRAMEBUFFER, 0);
			gl.glDeleteBuffers(1, new int[] { _fboId }, 0);
			_fboId = 0;

			ReentrantLock lock = _glGlobal.getGlobalLock();
			lock.lock();
			try {
				_glContext.release();
				_glGlobal.destroyContext(_glContext);
			} finally {
				lock.unlock();
			}

			_glContext = null;
			_glu = null;
		}

		_videoResolution = null;
		_colorMode = null;
		_videoFrameDuration = null;
		_camera = null;
		_lights = null;

		_audioMode = null;
		_audioAnimationRate = 0;
		_audioFrameCount = 0;

		super.deactivate();
	}

	public void reset() {
		super.reset();

		_videoResolution = null;
		_colorMode = null;
		_videoFrameDuration = null;
		_camera = null;
		_lights = null;

		_audioMode = null;
		_audioAnimationRate = 0;
		_audioFrameCount = 0;
	}

	public <T> T saveAndExecute(WrappedOperation<T> operation) {
		ColorMode colorMode = _colorMode;
		Time videoFrameDuration = _videoFrameDuration;
		Camera camera = _camera;
		Set<Light> lights = _lights;
		int audioFrameCount = _audioFrameCount;
		try {
			return super.saveAndExecute(operation);
		} finally {
			_colorMode = colorMode;
			_videoFrameDuration = videoFrameDuration;
			_camera = camera;
			_lights = lights;
			_audioFrameCount = audioFrameCount;
		}
	}

	public GL getGL() {
		initGL();
		return _glContext.getGL();
	}

	public GLU getGLU() {
		initGL();
		return _glu;
	}

	public Resolution getVideoResolution() {
		if (_videoResolution == null) {
			throw new IllegalStateException("videoResolution is not set.");
		}
		return _videoResolution;
	}

	public void setVideoResolution(Resolution resolution) {
		if (_videoResolution != null) {
			throw new IllegalStateException("videoResolution has already been set.");
		}
		_videoResolution = resolution;
	}

	public ColorMode getColorMode() {
		return _colorMode;
	}

	public void setColorMode(ColorMode colorMode) {
		_colorMode = colorMode;
	}

	public Time getVideoFrameDuration() {
		return _videoFrameDuration;
	}

	public void setVideoFrameDuration(Time frameDuration) {
		_videoFrameDuration = frameDuration;
	}

	public Camera getCamera() {
		return _camera;
	}

	public void setCamera(Camera camera) {
		_camera = camera;
	}

	public Set<Light> getLights() {
		return _lights;
	}

	public void setLights(Set<Light> lights) {
		_lights = lights;
	}

	public AudioMode getAudioMode() {
		if (_audioMode == null) {
			throw new IllegalStateException("audioMode is not set.");
		}
		return _audioMode;
	}

	public void setAudioMode(AudioMode audioMode) {
		if (_audioMode != null) {
			throw new IllegalStateException("audioMode has already been set.");
		}
		_audioMode = audioMode;
	}

	public int getAudioAnimationRate() {
		if (_audioAnimationRate == 0) {
			throw new IllegalStateException("audioAnimationRate is not set.");
		}
		return _audioAnimationRate;
	}

	public void setAudioAnimationRate(int rate) {
		if (_audioAnimationRate != 0) {
			throw new IllegalStateException("audioAnimationRate has already been set.");
		}
		if (rate < 1) {
			throw new IllegalArgumentException();
		}
		_audioAnimationRate = rate;
	}

	public int getAudioFrameCount() {
		return _audioFrameCount;
	}

	public void setAudioFrameCount(int count) {
		_audioFrameCount = count;
	}

}
