/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.mmd2javie;

import java.util.Formatter;

import org.mozilla.javascript.Scriptable;

import ch.kuramo.javie.api.ColorMode;
import ch.kuramo.javie.api.Size2i;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.api.Vec3d;
import ch.kuramo.javie.app.InjectorHolder;
import ch.kuramo.javie.core.AnimatableDouble;
import ch.kuramo.javie.core.AnimatableVec3d;
import ch.kuramo.javie.core.CameraLayer;
import ch.kuramo.javie.core.CompositionItem;
import ch.kuramo.javie.core.FrameDuration;
import ch.kuramo.javie.core.Layer;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.NullLayer;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.TransformableLayer;
import ch.kuramo.javie.core.services.ProjectElementFactory;

import com.google.inject.Inject;

public class AEScriptObjects {

	public static AEScriptApp createAppObject(Project project) {
		return InjectorHolder.getInjector().getInstance(AEScriptObjects.class)
				.new AEScriptApp(project);
	}


	private final ProjectElementFactory elementFactory;

	@Inject
	AEScriptObjects(ProjectElementFactory elementFactory) {
		this.elementFactory = elementFactory;
	}


	public class AEScriptApp {

		public final AEScriptProject project;

		private AEScriptApp(Project project) {
			this.project = new AEScriptProject(project);
		}
	}


	public class AEScriptProject {

		public final AEScriptItems items;

		private AEScriptProject(Project project) {
			items = new AEScriptItems(project);
		}
	}


	public class AEScriptItems {

		private final Project project;

		private AEScriptItems(Project project) {
			this.project = project;
		}

		public AEScriptComp addComp(
				String name, int width, int height,
				double aspectRatio, double duration, double fps) {

			Time frameDuration;
			String fpsStr = new Formatter().format("%.3f", fps).toString();
			if (fpsStr.equals("23.976")) frameDuration = FrameDuration.FPS_23_976;
			else if (fpsStr.equals("24.000")) frameDuration = FrameDuration.FPS_24;
			else if (fpsStr.equals("25.000")) frameDuration = FrameDuration.FPS_25;
			else if (fpsStr.equals("29.970")) frameDuration = FrameDuration.FPS_29_97;
			else if (fpsStr.equals("30.000")) frameDuration = FrameDuration.FPS_30;
			else if (fpsStr.equals("50.000")) frameDuration = FrameDuration.FPS_50;
			else if (fpsStr.equals("59.940")) frameDuration = FrameDuration.FPS_59_94;
			else if (fpsStr.equals("60.000")) frameDuration = FrameDuration.FPS_60;
			else frameDuration = new Time((int) (10000000/fps), 10000000);

			Time dur = new Time((int)(duration*frameDuration.timeScale), frameDuration.timeScale);

			LayerComposition layerComp = elementFactory.newLayerComposition(
					ColorMode.RGBA8, new Size2i(width, height), frameDuration, dur);

			CompositionItem item = elementFactory.newCompositionItem(layerComp);
			item.setName(name);

			project.getCompositions().add(layerComp);
			project.getItems().add(item);

			return new AEScriptComp(layerComp);
		}
	}


	public class AEScriptComp {

		public final AEScriptLayers layers;

		private AEScriptComp(LayerComposition layerComp) {
			layers = new AEScriptLayers(layerComp);
		}
	}


	public class AEScriptLayers {

		private final LayerComposition layerComp;

		private AEScriptLayers(LayerComposition layerComp) {
			this.layerComp = layerComp;
		}

		private void initLayerTimes(Layer layer) {
			Time time0 = Time.fromFrameNumber(0, layerComp.getFrameDuration());
			layer.setStartTime(time0);
			layer.setInPoint(time0);
			layer.setOutPoint(layerComp.getDuration());
		}

		public AEScriptCameraLayer addCamera(String name, double[] centerPoint) {
			CameraLayer cameraLayer = elementFactory.newCameraLayer();

			double w = layerComp.getSize().width;
			//double h = layerComp.getSize().height;
			double fovx = 39.6;
			double zoom = w / (2 * Math.tan(Math.toRadians(fovx/2)));
			cameraLayer.getZoom().reset(zoom);
			cameraLayer.getNear().reset(zoom / 2);
			cameraLayer.getFar().reset(zoom * 10);
			//cameraLayer.getPosition().reset(new Vec3d(w/2, h/2, -zoom));
			//cameraLayer.getPointOfInterest().reset(new Vec3d(w/2, h/2));

			cameraLayer.setName(name);
			cameraLayer.getPointOfInterest().clearKeyframes(new Vec3d(centerPoint[0], centerPoint[1], 0));
			initLayerTimes(cameraLayer);

			layerComp.getLayers().add(cameraLayer);

			return new AEScriptCameraLayer(cameraLayer, layerComp);
		}

		public AEScriptNullLayer addNull() {
			NullLayer nullLayer = elementFactory.newNullLayer();
			initLayerTimes(nullLayer);
			layerComp.getLayers().add(nullLayer);
			return new AEScriptNullLayer(nullLayer, layerComp);
		}
	}


	public abstract class AEScriptLayer<T extends Layer> {

		protected final T layer;

		protected final LayerComposition layerComp;

		protected AEScriptLayer(T layer, LayerComposition layerComp) {
			this.layer = layer;
			this.layerComp = layerComp;
		}

		public void setParent(AEScriptLayer<?> parent) {
			layerComp.setParentLayer(layer, parent.layer);
		}

		public void setName(String name) {
			layer.setName(name);
		}

		public Object property(String propertyName) {
			return Scriptable.NOT_FOUND;
		}
	}


	public class AEScriptCameraLayer extends AEScriptLayer<CameraLayer> {

		public final AEScriptAnimatableVec3d anchorPoint;

		public final AEScriptAnimatableVec3d position;

		public final AEScriptAnimatableDouble zoom;

		public final AEScriptAnimatableDouble focusDistance;

		public final AEScriptAnimatableDouble aperture;

		private AEScriptCameraLayer(CameraLayer cameraLayer, LayerComposition layerComp) {
			super(cameraLayer, layerComp);

			// MMD2AEから書き出した.jsxファイルには、カメラレイヤーにanchorPointプロパティを設定している箇所があるが、
			// AEのカメラレイヤーにアンカーポイントは無いはず。Javieにももちろん無い。なので、ここではダミーの値を使用する。
			anchorPoint = new AEScriptAnimatableVec3d(new AnimatableVec3d(Vec3d.ZERO), layerComp);

			position = new AEScriptAnimatableVec3d(cameraLayer.getPosition(), layerComp);
			zoom = new AEScriptAnimatableDouble(cameraLayer.getZoom(), layerComp);

			// Javieにはフォーカス距離と絞りの設定は無いのでダミーの値を使用する。
			focusDistance = new AEScriptAnimatableDouble(new AnimatableDouble(0d), layerComp);
			aperture = new AEScriptAnimatableDouble(new AnimatableDouble(0d), layerComp);
		}

		public Object property(String propertyName) {
			if ("zoom".equals(propertyName)) {
				return zoom;
			} else if ("focusDistance".equals(propertyName)) {
				return focusDistance;
			} else if ("aperture".equals(propertyName)) {
				return aperture;
			} else {
				return super.property(propertyName);
			}
		}
	}


	public abstract class AEScriptTransformableLayer<T extends TransformableLayer> extends AEScriptLayer<T> {

		public final AEScriptAnimatableVec3d anchorPoint;

		public final AEScriptAnimatableVec3d position;

		public final AEScriptAnimatableDouble xRotation;

		public final AEScriptAnimatableDouble yRotation;

		public final AEScriptAnimatableDouble zRotation;

		private AEScriptTransformableLayer(T transformableLayer, LayerComposition layerComp) {
			super(transformableLayer, layerComp);

			anchorPoint = new AEScriptAnimatableVec3d(transformableLayer.getAnchorPoint(), layerComp);
			position = new AEScriptAnimatableVec3d(transformableLayer.getPosition(), layerComp);
			xRotation = new AEScriptAnimatableDouble(transformableLayer.getRotationX(), layerComp);
			yRotation = new AEScriptAnimatableDouble(transformableLayer.getRotationY(), layerComp);
			zRotation = new AEScriptAnimatableDouble(transformableLayer.getRotationZ(), layerComp);
		}

		public void setThreeDLayer(boolean threeD) {
			layer.setThreeD(threeD);
		}
	}


	public class AEScriptNullLayer extends AEScriptTransformableLayer<NullLayer> {
		
		private AEScriptNullLayer(NullLayer nullLayer, LayerComposition layerComp) {
			super(nullLayer, layerComp);
		}
	}


	public class AEScriptAnimatableDouble {

		private final AnimatableDouble adouble;

		private final LayerComposition layerComp;

		private AEScriptAnimatableDouble(AnimatableDouble adouble, LayerComposition layerComp) {
			this.adouble = adouble;
			this.layerComp = layerComp;
		}

		public void setValue(double value) {
			if (adouble.hasKeyframe()) {
				throw new IllegalStateException("has Keyframes");
			}
			adouble.clearKeyframes(value);
		}

		public void setValueAtTime(double time, double value) {
			int timeScale = layerComp.getFrameDuration().timeScale;
			long timeValue = Math.round(time * timeScale);
			adouble.putKeyframe(new Time(timeValue, timeScale), value, adouble.getDefaultInterpolation());
		}
	}


	public class AEScriptAnimatableVec3d {

		private final AnimatableVec3d avec3d;

		private final LayerComposition layerComp;

		private AEScriptAnimatableVec3d(AnimatableVec3d avec3d, LayerComposition layerComp) {
			this.avec3d = avec3d;
			this.layerComp = layerComp;
		}

		public void setValue(double[] value) {
			if (avec3d.hasKeyframe()) {
				throw new IllegalStateException("has Keyframes");
			}
			avec3d.clearKeyframes(new Vec3d(value[0], value[1], value[2]));
		}

		public void setValueAtTime(double time, double[] value) {
			int timeScale = layerComp.getFrameDuration().timeScale;
			long timeValue = Math.round(time * timeScale);
			avec3d.putKeyframe(new Time(timeValue, timeScale),
					new Vec3d(value[0], value[1], value[2]), avec3d.getDefaultInterpolation());
		}

		public void setSpatialAutoBezierAtKey(int keyIndex, boolean spatialAutoBezier) {
			// ignore
		}
	}

}
