/*
 * Copyright (c) 2009-2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.project;

import java.util.Iterator;
import java.util.List;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;

import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.app.InjectorHolder;
import ch.kuramo.javie.app.PropertyUtil;
import ch.kuramo.javie.core.AnimatableValue;
import ch.kuramo.javie.core.Effect;
import ch.kuramo.javie.core.EffectableLayer;
import ch.kuramo.javie.core.JavieRuntimeException;
import ch.kuramo.javie.core.Keyframe;
import ch.kuramo.javie.core.Layer;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.ProjectDecodeException;
import ch.kuramo.javie.core.PropertyDescriptor;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.services.ProjectDecoder;
import ch.kuramo.javie.core.services.ProjectEncoder;

import com.google.inject.Inject;

public class ShiftLayerTimesOperation extends ProjectOperation {

	private static class Entry {
		final String layerId;
		final Time[] oldTimes;
		final List<String> oldAvalues = Util.newList();

		public Entry(String layerId, Time[] oldTimes) {
			super();
			this.layerId = layerId;
			this.oldTimes = oldTimes;
		}
	}


	private final String _compId;

	private final List<Entry> _entries = Util.newList();

	private volatile Time _deltaTime;

	@Inject
	private ProjectEncoder _encoder;

	@Inject
	private ProjectDecoder _decoder;


	public ShiftLayerTimesOperation(
			ProjectManager projectManager, LayerComposition comp, List<Layer> layers, Time deltaTime, String relation) {

		super(projectManager, "レイヤー時間の変更", relation);
		InjectorHolder.getInjector().injectMembers(this);

		projectManager.checkComposition(comp);
		_compId = comp.getId();
		_deltaTime = deltaTime;

		for (Layer layer : layers) {
			if (projectManager.checkLayer(layer) != comp) {
				throw new IllegalArgumentException();
			}

			Entry e = new Entry(layer.getId(),
					new Time[] { layer.getStartTime(), layer.getInPoint(), layer.getOutPoint() });

			for (String property : PropertyUtil.getLayerPropertyNames(layer)) {
				Object o = PropertyUtil.getProperty(layer, property);
				if (o instanceof AnimatableValue<?>) {
					e.oldAvalues.add(_encoder.encodeElement(o));
				}
			}

			if (layer instanceof EffectableLayer) {
				for (Effect effect : ((EffectableLayer) layer).getEffects()) {
					for (PropertyDescriptor pd : effect.getEffectDescriptor().getPropertyDescriptors()) {
						Object o = pd.get(effect);
						if (o instanceof AnimatableValue<?>) {
							e.oldAvalues.add(_encoder.encodeElement(o));
						}
					}
				}
			}

			_entries.add(e);
		}
	}

	private <V> void shiftKeyframes(AnimatableValue<V> avalue) {
		List<Keyframe<V>> list = Util.newList();
		for (Keyframe<V> kf : avalue.getKeyframeMap().values()) {
			list.add(new Keyframe<V>(_deltaTime.add(kf.time), kf.value, kf.interpolation));
		}

		avalue.clearKeyframes(avalue.getStaticValue());
		for (Keyframe<V> kf : list) {
			avalue.putKeyframe(kf);
		}
	}

	// TODO ThreeDSwitchOperation 等にほぼ同じメソッドがあるので整理する。
	private <A extends AnimatableValue<V>, V> void restore(String data, A dst) throws ExecutionException {
		try {
			@SuppressWarnings("unchecked")
			A src = _decoder.decodeElement(data, (Class<A>) dst.getClass());

			dst.reset(src.getStaticValue());
			dst.setExpression(src.getExpression());

			for (Keyframe<V> kf : src.getKeyframeMap().values()) {
				dst.putKeyframe(kf);
			}

		} catch (ProjectDecodeException e) {
			throw new ExecutionException("error decoding AnimatableValue data", e);
		}
	}

	@Override
	protected IStatus executeOrRedo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		LayerComposition comp = project.getComposition(_compId);
		List<Layer> layers = Util.newList();

		for (Entry e : _entries) {
			Layer layer = comp.getLayer(e.layerId);
			layer.setStartTime(_deltaTime.add(e.oldTimes[0]));
			layer.setInPoint(_deltaTime.add(e.oldTimes[1]));
			layer.setOutPoint(_deltaTime.add(e.oldTimes[2]));

			Iterator<String> it = e.oldAvalues.iterator();

			for (String property : PropertyUtil.getLayerPropertyNames(layer)) {
				Object o = PropertyUtil.getProperty(layer, property);
				if (o instanceof AnimatableValue<?>) {
					AnimatableValue<?> avalue = (AnimatableValue<?>) o;
					restore(it.next(), avalue);
					shiftKeyframes(avalue);
				}
			}

			if (layer instanceof EffectableLayer) {
				for (Effect effect : ((EffectableLayer) layer).getEffects()) {
					for (PropertyDescriptor pd : effect.getEffectDescriptor().getPropertyDescriptors()) {
						Object o = pd.get(effect);
						if (o instanceof AnimatableValue<?>) {
							AnimatableValue<?> avalue = (AnimatableValue<?>) o;
							restore(it.next(), avalue);
							shiftKeyframes(avalue);
						}
					}
				}
			}

			layers.add(layer);
		}

		if (pm != null) {
			pm.fireLayerTimesChange(comp, layers);
		}

		return Status.OK_STATUS;
	}

	@Override
	protected IStatus undo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		LayerComposition comp = project.getComposition(_compId);
		List<Layer> layers = Util.newList();

		for (Entry e : _entries) {
			Layer layer = comp.getLayer(e.layerId);
			layer.setStartTime(e.oldTimes[0]);
			layer.setInPoint(e.oldTimes[1]);
			layer.setOutPoint(e.oldTimes[2]);

			Iterator<String> it = e.oldAvalues.iterator();

			for (String property : PropertyUtil.getLayerPropertyNames(layer)) {
				Object o = PropertyUtil.getProperty(layer, property);
				if (o instanceof AnimatableValue<?>) {
					restore(it.next(), (AnimatableValue<?>) o);
				}
			}

			if (layer instanceof EffectableLayer) {
				for (Effect effect : ((EffectableLayer) layer).getEffects()) {
					for (PropertyDescriptor pd : effect.getEffectDescriptor().getPropertyDescriptors()) {
						Object o = pd.get(effect);
						if (o instanceof AnimatableValue<?>) {
							restore(it.next(), (AnimatableValue<?>) o);
						}
					}
				}
			}

			layers.add(layer);
		}

		if (pm != null) {
			pm.fireLayerTimesChange(comp, layers);
		}

		return Status.OK_STATUS;
	}

	@Override
	protected boolean merge(ProjectOperation operation, IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) {

		if (pm == null) {
			try {
				executeOrRedo(monitor, info, project, null);
			} catch (ExecutionException e) {
				throw new JavieRuntimeException(e);
			}
			return true;
		}

		if (!(operation instanceof ShiftLayerTimesOperation)) {
			return false;
		}

		ShiftLayerTimesOperation newOp = (ShiftLayerTimesOperation) operation;
		if (!newOp._compId.equals(_compId) || newOp._entries.size() != _entries.size()) {
			return false;
		}

		for (int i = 0, n = _entries.size(); i < n; ++i) {
			if (!newOp._entries.get(i).layerId.equals(_entries.get(i).layerId)) {
				return false;
			}
		}

		_deltaTime = newOp._deltaTime;

		try {
			executeOrRedo(monitor, info, project, pm);
		} catch (ExecutionException e) {
			throw new JavieRuntimeException(e);
		}

		_noEffect = (_deltaTime.timeValue == 0);

		return true;
	}

}
