/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.keying;

import java.nio.FloatBuffer;
import java.util.HashSet;
import java.util.Set;

import javax.media.opengl.GLUniformData;

import ch.kuramo.javie.api.Color;
import ch.kuramo.javie.api.IAnimatableBoolean;
import ch.kuramo.javie.api.IAnimatableColor;
import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IAnimatableEnum;
import ch.kuramo.javie.api.IAnimatableVec2d;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Vec2d;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.annotations.Effect.Categories;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.api.services.IVideoRenderSupport;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.HSLKey", category=Categories.KEYING)
public class HSLKey {

	@Property("1,1,1")
	private IAnimatableColor keyColor;

	@Property(value="10", min="0", max="360")
	private IAnimatableVec2d similarityHue;

	@Property(value="10", min="0", max="200")
	private IAnimatableVec2d similaritySat;

	@Property(value="10", min="0", max="100")
	private IAnimatableVec2d similarityLuma;

	@Property(value="10", min="0", max="360")
	private IAnimatableVec2d blendHue;

	@Property(value="10", min="0", max="200")
	private IAnimatableVec2d blendSat;

	@Property(value="10", min="0", max="100")
	private IAnimatableVec2d blendLuma;

	@Property(value="1", min="0", max="3")
	private IAnimatableVec2d correctionHue;

	@Property(value="1", min="0", max="2")
	private IAnimatableDouble correctionSat;

	@Property(value="0", min="0", max="1")
	private IAnimatableDouble correctionAlpha;

	@Property
	private IAnimatableEnum<Smoothing> smoothing;

	@Property
	private IAnimatableBoolean maskOnly;


	private final IVideoEffectContext context;

	private final IVideoRenderSupport support;

	private final IShaderProgram keyingProgram;

	private final IShaderProgram maskOnlyProgram;

	private final IShaderProgram smoothingLowProgram;

	private final IShaderProgram smoothingHighProgram;

	@Inject
	public HSLKey(IVideoEffectContext context, IVideoRenderSupport support, IShaderRegistry shaders) {
		this.context = context;
		this.support = support;

		keyingProgram = shaders.getProgram(KeyingShaders.class, "HSL_KEY");
		maskOnlyProgram = shaders.getProgram(KeyingShaders.class, "HSL_KEY_MASK_ONLY");
		smoothingLowProgram = shaders.getProgram(KeyingShaders.class, "SMOOTHING_LOW");
		smoothingHighProgram = shaders.getProgram(KeyingShaders.class, "SMOOTHING_HIGH");
	}

	public IVideoBuffer doVideoEffect() {
		IVideoBuffer input = context.doPreviousEffect();

		VideoBounds bounds = input.getBounds();
		if (bounds.isEmpty()) {
			return input;
		}

		Smoothing smoothing = context.value(this.smoothing);
		boolean maskOnly = context.value(this.maskOnly);

		IShaderProgram program1 = maskOnly ? maskOnlyProgram : keyingProgram;
		IShaderProgram program2 = (smoothing == Smoothing.LOW) ? smoothingLowProgram
								: (smoothing == Smoothing.HIGH) ? smoothingHighProgram : null;

		IVideoBuffer buffer;
		try {
			buffer = support.useShaderProgram(program1, prepareUniforms(), null, input);
		} finally {
			input.dispose();
		}

		if (program2 == null) {
			return buffer;
		}

		try {
			Set<GLUniformData> uniforms = new HashSet<GLUniformData>();
			uniforms.add(new GLUniformData("texture", 0));
			return support.useShaderProgram(program2, uniforms, null, buffer);
		} finally {
			buffer.dispose();
		}
	}

	private Set<GLUniformData> prepareUniforms() {
		Set<GLUniformData> uniforms = new HashSet<GLUniformData>();
		uniforms.add(new GLUniformData("texture", 0));

		Color keyColor = context.value(this.keyColor).clamp();
		Vec2d similarityHue = context.value(this.similarityHue);
		Vec2d similaritySat = context.value(this.similaritySat);
		Vec2d similarityLuma = context.value(this.similarityLuma);
		Vec2d blendHue = context.value(this.blendHue);
		Vec2d blendSat = context.value(this.blendSat);
		Vec2d blendLuma = context.value(this.blendLuma);
		Vec2d correctionHue = context.value(this.correctionHue);
		double correctionSat = context.value(this.correctionSat);
		double correctionAlpha = context.value(this.correctionAlpha);

		double[] keyHSL = new double[3];
		rgb2hsl(keyColor, keyHSL);

		double hueMin = keyHSL[0] - similarityHue.x/360;
		double hueMax = keyHSL[0] + similarityHue.y/360;
		double satMin = keyHSL[1] - similaritySat.x/100;
		double satMax = keyHSL[1] + similaritySat.y/100;
		double lumaMin = keyHSL[2] - similarityLuma.x/100;
		double lumaMax = keyHSL[2] + similarityLuma.y/100;

		double blendHueMin = Math.max(blendHue.x/360, 1e-10f);
		double blendHueMax = Math.max(blendHue.y/360, 1e-10f);
		double blendSatMin = Math.max(blendSat.x/100, 1e-10f);
		double blendSatMax = Math.max(blendSat.y/100, 1e-10f);
		double blendLumaMin = Math.max(blendLuma.x/100, 1e-10f);
		double blendLumaMax = Math.max(blendLuma.y/100, 1e-10f);

		uniforms.add(new GLUniformData("similarityMin", 3, toFloatBuffer(hueMin, satMin, lumaMin)));
		uniforms.add(new GLUniformData("similarityMax", 3, toFloatBuffer(hueMax, satMax, lumaMax)));
		uniforms.add(new GLUniformData("blendMin", 3, toFloatBuffer(blendHueMin, blendSatMin, blendLumaMin)));
		uniforms.add(new GLUniformData("blendMax", 3, toFloatBuffer(blendHueMax, blendSatMax, blendLumaMax)));

		double[] bleachHSL = { keyHSL[0], 1.0, 0.5 };
		double[] bleachRGB = new double[3];
		hsl2rgb(bleachHSL, bleachRGB);

		uniforms.add(new GLUniformData("bleachHSL", 3, toFloatBuffer(bleachHSL[0], bleachHSL[1], bleachHSL[2])));
		uniforms.add(new GLUniformData("bleachRGB", 3, toFloatBuffer(bleachRGB[0], bleachRGB[1], bleachRGB[2])));

		uniforms.add(new GLUniformData("correctionHue", 2, toFloatBuffer(correctionHue.x, correctionHue.y)));
		uniforms.add(new GLUniformData("correctionSat", (float)(2-correctionSat)));
		uniforms.add(new GLUniformData("correctionAlpha", (float)correctionAlpha));

		return uniforms;
	}

	private FloatBuffer toFloatBuffer(double... d) {
		float[] f = new float[d.length];
		for (int i = 0; i < d.length; ++i) {
			f[i] = (float)d[i];
		}
		return FloatBuffer.wrap(f);
	}

	// TODO 以下、GradientBaseに同じものがある。

	private void rgb2hsl(Color rgb, double[] hsl) {
		double min = Math.min(Math.min(rgb.r, rgb.g), rgb.b);
		double max = Math.max(Math.max(rgb.r, rgb.g), rgb.b);
		double dmax = max - min;

		double luma = (max + min)*0.5;
		double hue, sat;

		if (dmax == 0) {
			hue = sat = 0;
		} else {
			sat = (luma < 0.5) ? dmax/(max+min) : dmax/(2-max-min);

			double dr = ((max-rgb.r)/6 + dmax/2)/dmax;
			double dg = ((max-rgb.g)/6 + dmax/2)/dmax;
			double db = ((max-rgb.b)/6 + dmax/2)/dmax;

			hue = (rgb.r == max) ? db-dg
				: (rgb.g == max) ? 1./3 + dr-db
				:				   2./3 + dg-dr;

			if (hue < 0) hue += 1;
			else if (hue > 1) hue -= 1;
		}

		hsl[0] = hue;
		hsl[1] = sat;
		hsl[2] = luma;
	}

	private void hsl2rgb(double[] hsl, double[] rgb) {
		double hue = hsl[0];
		double sat = hsl[1];
		double luma = hsl[2];

		if (hue > 1) hue -= 1;

		if (sat == 0) {
			rgb[0] = rgb[1] = rgb[2] = luma;
		} else {
			double t2 = (luma < 0.5) ? luma*(1+sat) : luma+sat-luma*sat;
			double t1 = luma*2 - t2;

			rgb[0] = hue2rgb(t1, t2, hue+1./3);
			rgb[1] = hue2rgb(t1, t2, hue);
			rgb[2] = hue2rgb(t1, t2, hue-1./3);
		}
	}

	private double hue2rgb(double t1, double t2, double hue) {
		if (hue < 0) hue += 1;
		else if (hue > 1) hue -= 1;

		return (hue*6 < 1) ? t1 + (t2-t1)*6*hue
			 : (hue*2 < 1) ? t2
			 : (hue*3 < 2) ? t1 + (t2-t1)*(2./3-hue)*6
			 :				 t1;
	}

}
