/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.keying;

import java.nio.FloatBuffer;
import java.util.HashSet;
import java.util.Set;

import javax.media.opengl.GLUniformData;

import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IAnimatableEnum;
import ch.kuramo.javie.api.IAnimatableVec2d;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Resolution;
import ch.kuramo.javie.api.Vec2d;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.annotations.ShaderSource;
import ch.kuramo.javie.api.annotations.Effect.Categories;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.api.services.IVideoRenderSupport;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.ColorChart", category=Categories.KEYING)
public class ColorChart {

	public enum ChartType { HSL, YUV }


	@Property("HSL")
	private IAnimatableEnum<ChartType> chartType;

	@Property(value="0.5", min="0", max="1.0")
	private IAnimatableDouble luma;

	@Property
	private IAnimatableVec2d location;

	@Property(value="100", min="0")
	private IAnimatableDouble size;


	private final IVideoEffectContext context;

	private final IVideoRenderSupport support;

	private final IShaderProgram yuvProgram;

	private final IShaderProgram hslProgram;

	@Inject
	public ColorChart(IVideoEffectContext context, IVideoRenderSupport support, IShaderRegistry shaders) {
		this.context = context;
		this.support = support;
		yuvProgram = shaders.getProgram(ColorChart.class, "YUV_CHART");
		hslProgram = shaders.getProgram(ColorChart.class, "HSL_CHART");
	}

	public IVideoBuffer doVideoEffect() {
		IVideoBuffer input = context.doPreviousEffect();

		VideoBounds bounds = input.getBounds();
		if (bounds.isEmpty()) {
			return input;
		}

		Resolution resolution = context.getVideoResolution();
		Vec2d location = resolution.scale(context.value(this.location));
		double size = resolution.scale(context.value(this.size));

		if (size == 0 || location.x >= bounds.x+bounds.width || location.x+size <= bounds.x
					  || location.y >= bounds.y+bounds.height || location.y+size <= bounds.y) {
			return input;
		}

		ChartType chartType = context.value(this.chartType);
		double luma = context.value(this.luma);

		try {
			switch (chartType) {
				case YUV:
					drawYUVChart(input, location, size, luma);
					break;
				case HSL:
					drawHSLChart(input, location, size, luma);
					break;
			}

			IVideoBuffer result = input;
			input = null;
			return result;

		} finally {
			if (input != null) input.dispose();
		}
	}

	private void drawYUVChart(IVideoBuffer buffer, Vec2d location, double size, double luma) {
		final VideoBounds bounds = buffer.getBounds();
		final double left = location.x;
		final double top = location.y;
		final double right = left + size;
		final double bottom = top + size;

		Set<GLUniformData> uniforms = new HashSet<GLUniformData>();
		uniforms.add(new GLUniformData("location", 2, FloatBuffer.wrap(new float[] { (float)left, (float)top })));
		uniforms.add(new GLUniformData("size", (float)size));
		uniforms.add(new GLUniformData("luma", (float)luma));

		Runnable operation = new Runnable() {
			public void run() {
				support.ortho2D(bounds);
				support.quad2D(left, top, right, bottom);
			}
		};

		support.useShaderProgram(yuvProgram, uniforms, operation, 0, buffer);
	}

	@ShaderSource
	public static final String[] YUV_CHART = {
		"uniform vec2 location;",
		"uniform float size;",
		"uniform float luma;",
		"",
		"",	// Vの項の符号が通常とは逆なのは、Y座標の向きが逆なため。
		"const vec3 rVec = vec3(1.0,  0.0    , -1.13983);",
		"const vec3 gVec = vec3(1.0, -0.39465,  0.58060);",
		"const vec3 bVec = vec3(1.0,  2.03211,  0.0);",
		"",
		"const vec3 vec3_000 = vec3(0.0);",
		"const vec3 vec3_111 = vec3(1.0);",
		"",
		"void main(void)",
		"{",
		"	vec3 yuv = vec3(luma, (gl_FragCoord.xy-location)/size-0.5);",
		"	vec3 rgb = vec3(dot(yuv, rVec), dot(yuv, gVec), dot(yuv, bVec));",
		"",
		"	if (all(greaterThanEqual(rgb, vec3_000)) && all(lessThanEqual(rgb, vec3_111))) {",
		"		gl_FragColor = vec4(rgb, 1.0);",
		"	} else {",
		"		gl_FragColor = vec4(0.5, 0.5, 0.5, 1.0);",
		"	}",
		"}",
	};

	private void drawHSLChart(IVideoBuffer buffer, Vec2d location, double size, double luma) {
		final VideoBounds bounds = buffer.getBounds();
		final double left = location.x;
		final double top = location.y;
		final double right = left + size;
		final double bottom = top + size;
		double radius = size*0.5;

		Set<GLUniformData> uniforms = new HashSet<GLUniformData>();
		uniforms.add(new GLUniformData("center", 2, FloatBuffer.wrap(new float[] { (float)(left+radius), (float)(top+radius) })));
		uniforms.add(new GLUniformData("radius", (float)radius));
		uniforms.add(new GLUniformData("luma", (float)luma));

		Runnable operation = new Runnable() {
			public void run() {
				support.ortho2D(bounds);
				support.quad2D(left, top, right, bottom);
			}
		};

		support.useShaderProgram(hslProgram, uniforms, operation, 0, buffer);
	}

	// TODO FractalNoiseにあるhsl2rgbをどこか適切な場所に移動する。
	@ShaderSource(attach="ch.kuramo.javie.effects.noiseGrain.FractalNoise.hsl2rgb")
	public static final String[] HSL_CHART = {
		"uniform vec2 center;",
		"uniform float radius;",
		"uniform float luma;",
		"",
		"const float PI2 = 2.0*3.14159265358979323846264;",
		"vec3 hsl2rgb(vec3 hsl);",
		"",
		"void main(void)",
		"{",
		"	vec2 xy = gl_FragCoord.xy-center;",
		"	float h = -0.25-atan(xy.y, xy.x)/PI2;",
		"	float s = distance(gl_FragCoord.xy, center)/radius;",
		"	if (s <= 1.0) {",
		"		vec3 hsl = vec3(h-floor(h), s, luma);",
		"		vec3 rgb = hsl2rgb(hsl);",
		"		gl_FragColor = vec4(rgb, 1.0);",
		"	} else {",
		"		gl_FragColor = vec4(0.5, 0.5, 0.5, 1.0);",
		"	}",
		"}",
	};

}
