/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.generate;

import java.nio.FloatBuffer;
import java.util.HashSet;
import java.util.Set;

import javax.media.opengl.GL2;
import javax.media.opengl.GLUniformData;
import javax.media.opengl.glu.GLU;
import javax.media.opengl.glu.GLUquadric;

import ch.kuramo.javie.api.BlendMode;
import ch.kuramo.javie.api.Color;
import ch.kuramo.javie.api.IAnimatableBoolean;
import ch.kuramo.javie.api.IAnimatableColor;
import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IAnimatableEnum;
import ch.kuramo.javie.api.IAnimatableVec2d;
import ch.kuramo.javie.api.IArray;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Resolution;
import ch.kuramo.javie.api.Vec2d;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.annotations.ShaderSource;
import ch.kuramo.javie.api.annotations.Effect.Categories;
import ch.kuramo.javie.api.services.IArrayPools;
import ch.kuramo.javie.api.services.IBlendSupport;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.api.services.IVideoRenderSupport;
import ch.kuramo.javie.effects.Texture1D;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.Circle", category=Categories.GENERATE)
public class Circle {

	public enum Edge {
		NONE,
		EDGE_RADIUS,
		THICKNESS,
		P_THICKNESS,				// PはProportionalのP
		P_THICKNESS_P_FEATHER
	}


	@Property
	private IAnimatableVec2d location;

	@Property(value="75", min="0")
	private IAnimatableDouble radius;

	@Property("0.5")
	private IAnimatableVec2d anchor;

	@Property
	private IAnimatableEnum<Edge> edge;

	@Property(value="10", min="0")
	private IAnimatableDouble edgeRadius;

	@Property(value="10", min="0", max="4000")
	private IAnimatableDouble thickness;

	@Property(min="0", max="4000")
	private IAnimatableDouble outerFeather;

	@Property(min="0", max="4000")
	private IAnimatableDouble innerFeather;

	@Property
	private IAnimatableBoolean invert;

	@Property("1,1,1")
	private IAnimatableColor color;

	@Property(value="100", min="0", max="100")
	private IAnimatableDouble opacity;

	@Property
	private IAnimatableEnum<BlendMode> blendMode;


	private final IVideoEffectContext context;

	private final IVideoRenderSupport support;

	private final IBlendSupport blendSupport;

	private final IArrayPools arrayPools;

	private final IShaderProgram circleSamplerProgram;

	private final IShaderProgram invertSamplerProgram;

	@Inject
	public Circle(
			IVideoEffectContext context, IVideoRenderSupport support, IBlendSupport blendSupport,
			IArrayPools arrayPools, IShaderRegistry shaders) {

		this.context = context;
		this.support = support;
		this.blendSupport = blendSupport;
		this.arrayPools = arrayPools;
		circleSamplerProgram = shaders.getProgram(Circle.class, "CIRCLE_SAMPLER");
		invertSamplerProgram = shaders.getProgram(Circle.class, "INVERT_SAMPLER");
	}

	private class CircleParams {
		private final Vec2d center;
		private final boolean invert;
		private final Color color;
		private final double opacity;

		private final double innerRadius;
		private final double outerRadius;
		private final double innerMiddle;
		private final double outerMiddle;

		private final int[] texture;
		private final int[] texSize = new int[2];

		private CircleParams() {
			Resolution resolution = context.getVideoResolution();

			Vec2d location = context.value(Circle.this.location);
			double radius = context.value(Circle.this.radius);
			Vec2d anchor = context.value(Circle.this.anchor);

			center = resolution.scale(new Vec2d(location.x-radius*(anchor.x-0.5)*2, location.y-radius*(anchor.y-0.5)*2));
			radius = resolution.scale(radius);

			Edge edge = context.value(Circle.this.edge);
			invert = context.value(Circle.this.invert);
			opacity = context.value(Circle.this.opacity) / 100;

			Color color = context.value(Circle.this.color).clamp();
			double a = color.a;
			this.color = new Color(color.r*a, color.g*a, color.b*a, a);

			double edgeRadius = 0;
			double thickness = 0;
			double innerFeather = 0;
			double outerFeather = 0;

			switch (edge) {
				case NONE:
					outerFeather = resolution.scale(context.value(Circle.this.outerFeather));
					break;

				case EDGE_RADIUS:
					edgeRadius = resolution.scale(context.value(Circle.this.edgeRadius));
					if (edgeRadius > radius) {
						double tmp = edgeRadius;
						edgeRadius = radius;
						radius = tmp;
					}
					innerFeather = resolution.scale(context.value(Circle.this.innerFeather));
					outerFeather = resolution.scale(context.value(Circle.this.outerFeather));
					break;

				case THICKNESS:
					thickness = resolution.scale(context.value(Circle.this.thickness));
					innerFeather = resolution.scale(context.value(Circle.this.innerFeather));
					outerFeather = resolution.scale(context.value(Circle.this.outerFeather));
					break;

				case P_THICKNESS:
					// radiusがすでにスケール済みなので、thicknessはスケールしない。
					thickness = context.value(Circle.this.thickness) * radius / 100;
					innerFeather = resolution.scale(context.value(Circle.this.innerFeather));
					outerFeather = resolution.scale(context.value(Circle.this.outerFeather));
					break;

				case P_THICKNESS_P_FEATHER:
					// radiusがすでにスケール済みなのでスケールしない。
					thickness = context.value(Circle.this.thickness) * radius / 100;
					innerFeather = context.value(Circle.this.innerFeather) * radius / 100;
					outerFeather = context.value(Circle.this.outerFeather) * radius / 100;
					break;
			}

			switch (edge) {
				case NONE:
					innerRadius = 0;
					outerRadius = radius + outerFeather;
					innerMiddle = 0;
					outerMiddle = radius;
					break;

				case EDGE_RADIUS:
					innerRadius = Math.max(0, edgeRadius - innerFeather);
					outerRadius = radius + outerFeather;
					innerMiddle = edgeRadius;
					outerMiddle = radius;
					break;

				default:
					innerRadius = Math.max(0, radius - thickness*0.5 - innerFeather);
					outerRadius = radius + thickness*0.5 + outerFeather;
					innerMiddle = radius - thickness*0.5;
					outerMiddle = radius + thickness*0.5;
					break;
			}

			texture = createCircleTextures(innerMiddle, innerFeather, outerFeather, texSize);
		}
	}

	public IVideoBuffer doVideoEffect() {
		IVideoBuffer original;
		final VideoBounds bounds;

		BlendMode blendMode = context.value(this.blendMode);

		if (blendMode == BlendMode.NONE) {
			bounds = context.getPreviousBounds();
			original = context.createVideoBuffer(bounds);
		} else {
			original = context.doPreviousEffect();
			bounds = original.getBounds();
		}

		if (bounds.isEmpty()) {
			return original;
		}


		CircleParams params = null;
		IVideoBuffer circle = null;
		try {
			final CircleParams p = params = new CircleParams();

			circle = context.createVideoBuffer(bounds);

			if (p.invert) {
				circle.clear(p.color);
			} else {
				circle.clear();
			}


			IShaderProgram program = p.invert ? invertSamplerProgram : circleSamplerProgram;

			Set<GLUniformData> uniforms = new HashSet<GLUniformData>();
			uniforms.add(new GLUniformData("texInner", 0));
			uniforms.add(new GLUniformData("texOuter", 1));
			uniforms.add(new GLUniformData("texSize", 2, FloatBuffer.wrap(new float[] { p.texSize[0], p.texSize[1] })));
			uniforms.add(new GLUniformData("texOffset", 2, FloatBuffer.wrap(new float[] {
					(float)(p.texSize[0]*0.5-p.innerMiddle), (float)(p.texSize[1]*0.5-p.outerMiddle) })));
			uniforms.add(new GLUniformData("center", 2, FloatBuffer.wrap(new float[] {
					(float)(p.center.x-bounds.x), (float)(p.center.y-bounds.y) })));
			uniforms.add(new GLUniformData("color", 4, FloatBuffer.wrap(new float[] {
					(float)p.color.r, (float)p.color.g, (float)p.color.b, (float)p.color.a })));

			Runnable operation = new Runnable() {
				public void run() {
					GL2 gl = context.getGL().getGL2();
					GLU glu = context.getGLU();

					support.ortho2D(bounds);
					gl.glMatrixMode(GL2.GL_MODELVIEW);
					gl.glTranslatef((float)p.center.x, (float)p.center.y, 0);

					gl.glActiveTexture(GL2.GL_TEXTURE0);
					gl.glBindTexture(GL2.GL_TEXTURE_1D, p.texture[0]);
					gl.glActiveTexture(GL2.GL_TEXTURE1);
					gl.glBindTexture(GL2.GL_TEXTURE_1D, p.texture[1]);

					GLUquadric quadric = glu.gluNewQuadric(/* TODO これのオーバーロードされたメソッドの引数 useGLSL って何だろう？ */);
					try {
						glu.gluQuadricDrawStyle(quadric, GLU.GLU_FILL);
						glu.gluQuadricNormals(quadric, GLU.GLU_NONE);

						// TODO slicesはこれで妥当か？ ちょっと多すぎるかも。
						int slices = (int)Math.ceil(2*Math.PI*p.outerRadius);
						glu.gluDisk(quadric, Math.max(0, p.innerRadius-2), p.outerRadius+2, slices, 1);
					} finally {
						glu.gluDeleteQuadric(quadric);
					}
				}
			};

			support.useShaderProgram(program, uniforms, operation, GL2.GL_TEXTURE_BIT, circle);


			if (blendMode == BlendMode.NONE && p.opacity == 1) {
				IVideoBuffer result = circle;
				circle = null;
				return result;

			} else {
				return blendSupport.blend(circle, original, null, blendMode, p.opacity, context);
			}

		} finally {
			if (original != null) original.dispose();
			if (circle != null) circle.dispose();
			if (params != null) deleteTextures(params.texture);
		}
	}

	private int[] createCircleTextures(
			double innerMiddle, double innerFeather, double outerFeather, int[] texSize) {

		int tex0 = 0, tex1 = 0;
		IArray<float[]> data0 = null, data1 = null;
		try {
			if (innerMiddle <= 0 && innerFeather == 0) {
				tex0 = Texture1D.fromArray(new float[] { 1, 1 }, context);
				texSize[0] = 2;
			} else {
				data0 = createCircleData(true, innerFeather);
				tex0 = Texture1D.fromArray(data0, context);
				texSize[0] = data0.getLength();
			}

			data1 = createCircleData(false, outerFeather);
			tex1 = Texture1D.fromArray(data1, context);
			texSize[1] = data1.getLength();

			int[] result = new int[] { tex0, tex1 };
			tex0 = tex1 = 0;
			return result;

		} finally {
			if (data0 != null) data0.release();
			if (data1 != null) data1.release();
			if (tex0 != 0) deleteTextures(tex0);
			if (tex1 != 0) deleteTextures(tex1);
		}
	}

	private IArray<float[]> createCircleData(boolean inner, double feather) {
		int halfLen = (int)Math.ceil(feather) + 1;

		IArray<float[]> data = arrayPools.getFloatArray(halfLen*2);
		float[] array = data.getArray();
		int arrayLen = data.getLength();

		double lowerOutside = inner ? 0 : 1;
		double upperOutside = inner ? 1 : 0;
		double offset = halfLen + (inner ? -feather : feather);

		for (int i = 0; i < arrayLen; ++i) {
			double r = i + 0.5;
			double d = (r <= halfLen-feather) ? lowerOutside
					 : (r >= halfLen+feather) ? upperOutside
					 : 0.5*(1-Math.cos(2*Math.PI*(r-offset)/(4*feather)));

			array[i] = (float) d;
		}

		return data;
	}

	private void deleteTextures(int... texture) {
		GL2 gl = context.getGL().getGL2();
		gl.glDeleteTextures(texture.length, texture, 0);
	}

	private static final String[] createSamplerProgram(boolean invert) {
		return new String[] {
				"uniform sampler1D texInner;",
				"uniform sampler1D texOuter;",
				"uniform vec2 texSize;",
				"uniform vec2 texOffset;",
				"uniform vec2 center;",
				"uniform vec4 color;",		// ATIでgl_Colorを使うには頂点シェーダで明示的にglFrontColorを設定する必要がある。
				"",							// ここではそれをせず、uniform変数で色を渡している。
				"void main(void)",
				"{",
				"	float d = distance(center, gl_FragCoord.st);",
				"	vec2 texCoord = (vec2(d, d) + texOffset) / texSize;",
				"",
				"	float inner = texture1D(texInner, texCoord.x).a;",
				"	float outer = texture1D(texOuter, texCoord.y).a;",
				"",
	   invert ? "	gl_FragColor = color * (1.0-inner*outer);"
			  : "	gl_FragColor = color * (inner*outer);",
				"}"
		};
	}

	@ShaderSource
	public static final String[] CIRCLE_SAMPLER = createSamplerProgram(false);

	@ShaderSource
	public static final String[] INVERT_SAMPLER = createSamplerProgram(true);

}
